#pragma once

#include "cache.h"
#include "data.h"

#include <travel/hotels/lib/cpp/grpc/grpc_async_client.h>
#include <travel/hotels/lib/cpp/mon/counter_hypercube.h>
#include <travel/hotels/lib/cpp/logging/common_logger.h>
#include <travel/hotels/pricechecker/proto/config.pb.h>
#include <travel/hotels/pricechecker/proto/reqans_logrecord.pb.h>
#include <travel/hotels/proto/offercache_grpc/offercache_service.grpc.pb.h>

#include <util/generic/hash_set.h>

namespace NTravel {
namespace NPriceChecker {

class TJobProcessor {
public:
    using TOnJobCompletion = std::function<void(void)>;

    using TLogRecord = NTravelProto::NPriceChecker::TReqAnsLogRecord;

    static const TString OfferCacheClientId;

    TJobProcessor(const NTravelProto::NPriceChecker::TConfig::TJobProcessor& config, const TCache& cache,
                  TCommonLogger<TLogRecord>& reqAnsLogger);
    ~TJobProcessor() = default;

    void RegisterCounters(NMonitor::TCounterSource& source);

    void Start();
    void Stop();

    void InitializeCountersForJobType(const TString& typeName);

    void AddInternalJob(const TOfferWithRequestRef& offerWithRequest, const TString& typeName, TInstant receiptTime);
    void AddExternalJob(const TOfferId& offerId, TMaybe<TInstant> cacheTimestamp,
                        const TOnJobCompletion& onCompletionCallback, TInstant receiptTime);

    void MaybeProcessResponse(const TSearcherResponse& searcherResponse);
private:
    struct TGenericCounters : public NMonitor::TCounterSource {
        NMonitor::TDerivCounter NAncientCacheMisses;
        NMonitor::TDerivCounter NOldCacheMisses;
        NMonitor::TDerivCounter NFreshCacheMisses;

        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    struct TSpecificCounters : public NMonitor::TCounterSource
                             , public TThrRefBase
    {
        NMonitor::TCounter NRunningChecks;

        NMonitor::TDerivCounter NCompletedChecks;
        NMonitor::TDerivCounter NExpiredChecks;
        NMonitor::TDerivCounter NBusResponseErrors;
        NMonitor::TDerivCounter NRpcErrors;

        NMonitor::TDerivCounter NSentRpcRequests;

        NMonitor::TDerivCounter NPriceNotFound;
        NMonitor::TDerivCounter NPriceIsSame;
        NMonitor::TDerivCounter NPriceIsSmaller;
        NMonitor::TDerivCounter NPriceIsBigger;
        NMonitor::TDerivCounter NPriceIsApproxSame;
        NMonitor::TDerivCounter NPriceIsApproxSmaller;
        NMonitor::TDerivCounter NPriceIsApproxBigger;

        NMonitor::TDerivHistogramCounter NPricePercentSmaller;
        NMonitor::TDerivHistogramCounter NPricePercentBigger;

        TSpecificCounters();
        void QueryCounters(NMonitor::TCounterTable* ct) const override;
    };

    using TSpecificCountersRef = NMonitor::TCounterHypercube<TSpecificCounters>::TCountersRef;

    struct TJob : public TThrRefBase {
        const TOfferId OfferId;
        const TString TypeName;
        const TInstant ReceiptTime;
        const TString LogPrefix;

        TInstant StartTime;
        TOfferWithRequestRef OfferWithRequest;
        TSpecificCountersRef Counters;
        TOnJobCompletion OnCompletionCallback;

        TJob(const TOfferId& offerId, const TString& typeName, TInstant receiptTime);
        TLogRecord GenerateLogRecord(TLogRecord::TCheckResult checkResult,
                                     const TString& errorMessage = TString());
    };

    using TJobRef = TIntrusivePtr<TJob>;

    using TOfferCacheGrpcClient = NGrpc::TAsyncClient<NTravelProto::NOfferCacheGrpc::OfferCacheServiceV1>;

    const TDuration GenerationSwapPeriod_;
    const TDuration AncientCacheMissLag_;
    const TDuration OldCacheMissLag_;
    const TDuration FreshCacheMissRetryDelay_;
    const size_t FreshCacheMissRetryCount_;
    const NTravelProto::ERequestClass InternalJobRequestClass_;
    const NTravelProto::ERequestClass ExternalJobRequestClass_;
    const TCache& Cache_;
    TCommonLogger<TLogRecord>& ReqAnsLogger_;
    TOfferCacheGrpcClient OfferCacheClient_;
    THashMap<int, TString> OperatorIdToName_;

    TGenericCounters GenericCounters_;
    NMonitor::TCounterHypercube<TSpecificCounters> SpecificCounters_;

    TMutex Lock_;
    THashMap<TRequestId, TJobRef> CheckingRequestIdToJobFirstGen_;
    THashMap<TRequestId, TJobRef> CheckingRequestIdToJobSecondGen_;

    TString ConvertOperatorIdToName(int operatorId);

    void SwapGenerations();

    void PrepareExternalJob(TJobRef job, TDuration requestLag, size_t attempt);
    void StartJob(const TJobRef& job, NTravelProto::ERequestClass requestClass);
    void FinishJob(const TJobRef& job, const TLogRecord& logRecord);

    void CheckDifference(const TJobRef& job, const TVector<TOfferWithRequestRef>& offersWithRequests);
};

} // namespace NPriceChecker
} // namespace NTravel
