#include "service.h"

#include <travel/hotels/lib/cpp/scheduler/scheduler.h>
#include <travel/hotels/proto2/hotels.pb.h>
#include <travel/hotels/proto2/bus_messages.pb.h>

#include <util/digest/multi.h>
#include <util/generic/maybe.h>

namespace NTravel {
namespace NPriceChecker {

void TService::TCounters::QueryCounters(NMonitor::TCounterTable* ct) const {
    ct->insert(MAKE_COUNTER_PAIR(IsWaiting));
    ct->insert(MAKE_COUNTER_PAIR(IsWorking));
    ct->insert(MAKE_COUNTER_PAIR(IsReady));

    ct->insert(MAKE_COUNTER_PAIR(NLatePriceCheckRequests));
}

TService::TService(const NTravelProto::NPriceChecker::TConfig& config)
    : Config_(config)
    , RequestMaxLateness_(TDuration::Seconds(config.GetOther().GetRequestMaxLatenessSec()))
    , DistributedLock_(config.GetDistributedLock())
    , YtConfigOperators_("YtConfigOperators", config.GetYtConfigOperators())
    // PriceChecker кэширует только офферы, а не запросы, поэтому ему можно пользоваться оптимизацией с ExpireTimestamp
    , OfferBus_(config.GetOfferBus(), "OfferBus")
    , PriceCheckReqBus_(config.GetPriceCheckReqBus(), "PriceCheckReqBus", TDuration::Seconds(config.GetOther().GetPriceCheckReqBusMaxAgeSec()))
    , ReqAnsLogger_("ReqAnsLogger", config.GetReqAnsLogger())
    , Cache_(config.GetCache())
    , JobProcessor_(config.GetJobProcessor(), Cache_, ReqAnsLogger_)
    , JobGenerator_(config.GetJobGenerator(), JobProcessor_)
    , RestartDetector_(config.GetOther().GetRestartDetectorStateFile())
    , OfferTracker_(*this, "OfferTracker", config.GetJobProcessor().GetOfferCache(), config.GetTrackingStrategy(), config.GetStateBusReader(), config.GetStateBusWriter(), config.GetOfferTracker())
    , TrackingResultsConsumer_(config.GetTrackingResultsLogger())
    , PriceFilterChecker_(
        config.GetJobProcessor().GetOfferCache(),
        config.GetPriceFilterTable(),
        config.GetPermalinkToClusterMapper(),
        config.GetPermalinkToOriginalIdsMapper(),
        config.GetYtConfigPartners(),
        config.GetOther().GetMaxPriceFilterCheckerRps()) {
    CountersPage_.AddToHttpService(MonWebService_);
    CountersPage_.RegisterSource(&Counters_, "Server");
    Http_.RegisterCounters(CountersPage_);
    OfferBus_.RegisterCounters(CountersPage_);
    PriceCheckReqBus_.RegisterCounters(CountersPage_);
    Cache_.RegisterCounters(CountersPage_);
    JobProcessor_.RegisterCounters(CountersPage_);
    JobGenerator_.RegisterCounters(CountersPage_);
    RestartDetector_.RegisterCounters(CountersPage_);
    OfferTracker_.RegisterCounters(CountersPage_);
    TrackingResultsConsumer_.RegisterCounters(CountersPage_, "TrackingResults");
    YtConfigOperators_.RegisterCounters(CountersPage_);
    PriceFilterChecker_.RegisterCounters(CountersPage_, "PriceFilterChecker");
    ReqAnsLogger_.RegisterCounters(CountersPage_);
    TScheduler::Instance().RegisterCounters(CountersPage_);

    Http_.AddHandler("/reopen-logs", NHttp::Local(), this, &TService::OnReopenLogs);
    Http_.AddHandler("/setlog",        NHttp::Local(),    this, &TService::OnSetLogLevel);

    auto enablePcV2 = Config_.GetOther().GetEnablePriceCheckerV2();
    auto enablePriceFilterChecker = Config_.GetOther().GetEnablePriceFilterChecker();

    DistributedLock_.SetStartHandler([this, enablePcV2, enablePriceFilterChecker]() {
        if (enablePcV2) {
            YtConfigOperators_.Start();
            YtConfigOperators_.SetOnUpdateHandler([this](bool first) {
                if (first) {
                    OfferTracker_.Start();
                }
            });
            OfferTracker_.SetInitializationFinishHandler([this]() {
                OfferBus_.Start();
            });
        } else {
            OfferBus_.Start(); // PriceCheckReqBus is started after OfferBus is ready
        }
        if (enablePriceFilterChecker) {
            PriceFilterChecker_.Start();
        }
        JobProcessor_.Start();
    });
    DistributedLock_.SetStopHandler([this]() {
        Counters_.IsWaiting = 0;
        Counters_.IsWorking = 0;
        Counters_.IsReady = 0;
        TScheduler::Instance().Stop();
        JobProcessor_.Stop();
        PriceFilterChecker_.Stop();
        PriceCheckReqBus_.Stop();
        OfferTracker_.Stop();
        OfferBus_.Stop();
    });
    OfferBus_.SetReadinessNotifier([this]() {
        INFO_LOG << "OfferBus is ready, starting PriceCheckReqBus" << Endl;
        PriceCheckReqBus_.Start();
        Counters_.IsWaiting = 0;
        Counters_.IsWorking = 1;
        Counters_.IsReady = 1;
    });

    if (enablePcV2) {
        OfferTracker_.Subscribe([this](const TOfferTrackingState& trackingState) {
            TrackingResultsConsumer_.HandleTrackingResult(trackingState);
        });
    }

    OfferBus_.Ignore(ru::yandex::travel::hotels::TPingMessage());
    OfferBus_.Ignore(NTravelProto::TTravellineCacheEvent());
    OfferBus_.Subscribe(ru::yandex::travel::hotels::TSearcherMessage(), [this, enablePcV2, enablePriceFilterChecker](const TYtQueueMessage& busMessage) {
        ru::yandex::travel::hotels::TSearcherMessage message;
        if (!message.ParseFromString(busMessage.Bytes)) {
            throw yexception() << "Failed to parse TSearcherMessage record";
        }
        auto firstResult = ProcessSearcherMessage(busMessage, message);
        auto secondResult = enablePcV2 ? OfferTracker_.ProcessSearcherMessage(busMessage, message) : false;
        auto thirdResult = enablePriceFilterChecker ? PriceFilterChecker_.ProcessSearcherMessage(busMessage, message) : false;
        return firstResult || secondResult || thirdResult;
    });

    PriceCheckReqBus_.Subscribe(ru::yandex::travel::hotels::TPriceCheckReq(), this, &TService::ProcessPriceCheckRequest);

    for (const auto& samplingRule : JobGenerator_.GetSamplingRules()) {
        JobProcessor_.InitializeCountersForJobType(samplingRule.Name);
    }
}

void TService::Run() {
    TScheduler::Instance().Start();
    TrackingResultsConsumer_.Start();
    ReqAnsLogger_.Start();
    MonWebService_.Start(Config_.GetOther().GetMonitoringPort());
    Http_.Start(Config_.GetHttp());
    Counters_.IsWaiting = 1;

    DistributedLock_.Run();

    RestartDetector_.ReportShutdown();
    Http_.Stop();
    MonWebService_.Stop();
    ReqAnsLogger_.Stop();
    TrackingResultsConsumer_.Stop();
    YtConfigOperators_.Stop();
}

void TService::Stop() {
    DistributedLock_.Stop();
}

void TService::OnReopenLogs(const NHttp::TRequest&, const NHttp::TOnResponse& responseCb) {
    INFO_LOG << "Reopening ReqAns log" << Endl;
    ReqAnsLogger_.Reopen();
    INFO_LOG << "Reopening TrackingResults log" << Endl;
    TrackingResultsConsumer_.ReopenLog();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    if (auto v = httpReq.Query().Get("level")) {
        auto p = FromString<ELogPriority>(v);
        DoInitGlobalLog(CreateLogBackend(Config_.GetOther().GetMainLogFile(), p, true));
        responseCb(NHttp::TResponse::CreateText("Log level was changed to " + ToString(v) + "\n"));
    } else {
        responseCb(THttpResponse(HTTP_BAD_REQUEST));
    }
}

bool TService::ProcessSearcherMessage(const TYtQueueMessage& busMessage, const ru::yandex::travel::hotels::TSearcherMessage& parsedMessage) {
    auto searcherResponse = ParseSearcherResponse(parsedMessage.GetRequest(), parsedMessage.GetResponse());

    Cache_.AddOffers(busMessage.ExpireTimestamp, searcherResponse);
    JobProcessor_.MaybeProcessResponse(searcherResponse);
    if (searcherResponse.RequestOfferCacheClientId != TJobProcessor::OfferCacheClientId &&
        searcherResponse.RequestOfferCacheClientId != TOfferTracker::PriceCheckerOfferCacheClientId) {
        JobGenerator_.MaybeGenerateJob(busMessage.Timestamp, searcherResponse.OffersWithRequests);
    }
    return true;
}

TSearcherResponse TService::ParseSearcherResponse(const NTravelProto::TSearchOffersReq& req,
                                                  const NTravelProto::TSearchOffersRsp& rsp)
{
    TSearcherResponse parsedResponse;
    parsedResponse.RequestId = req.GetId();
    parsedResponse.RequestOfferCacheClientId = req.GetAttribution().GetOfferCacheClientId();

    if (rsp.HasPlaceholder()) {
        parsedResponse.Type = ESearcherResponseType::PLACEHOLDER;
        return parsedResponse;
    }
    if (rsp.HasError()) {
        parsedResponse.Type = ESearcherResponseType::ERROR;
        parsedResponse.Error = rsp.GetError().GetMessage();
        return parsedResponse;
    }

    parsedResponse.Type = ESearcherResponseType::OFFER;

    auto cutRequest = MakeAtomicShared<NTravelProto::TSearchOffersReq>();
    *cutRequest->MutableHotelId() = req.GetHotelId();
    cutRequest->SetCheckInDate(req.GetCheckInDate());
    cutRequest->SetCheckOutDate(req.GetCheckOutDate());
    cutRequest->SetOccupancy(req.GetOccupancy());
    cutRequest->SetCurrency(req.GetCurrency());
    cutRequest->SetPermalink(req.GetPermalink());

    for (const auto& offer : rsp.GetOffers().GetOffer()) {
        auto offerWithRequest = MakeIntrusive<TOfferWithRequest>();
        offerWithRequest->RequestPb = cutRequest;
        offerWithRequest->OperatorId = offer.GetOperatorId();
        if (!GetUuid(offer.GetId(), offerWithRequest->OfferId)) {
            ythrow yexception() << "Can't parse offer id as uuid: " << offer.GetId();
        }
        offerWithRequest->OfferHash = MultiHash(offer.GetOperatorId(),
                                                offer.GetPrice().GetCurrency(),
                                                offer.GetCapacity(),
                                                offer.GetDisplayedTitle().value(),
                                                offer.GetPansion(),
                                                offer.HasFreeCancellation(),
                                                offer.GetFreeCancellation().value());
        offerWithRequest->OfferPrice = offer.GetPrice().GetAmount();

        parsedResponse.OffersWithRequests.push_back(offerWithRequest);
    }

    return parsedResponse;
}

bool TService::ProcessPriceCheckRequest(const TYtQueueMessage& busMessage) {
    ru::yandex::travel::hotels::TPriceCheckReq message;
    if (!message.ParseFromString(busMessage.Bytes)) {
        throw yexception() << "Failed to parse TPriceCheckReq record";
    }

    if (Now() - busMessage.Timestamp > RequestMaxLateness_) {
        Counters_.NLatePriceCheckRequests.Inc();
        WARNING_LOG << "Got a late price check request, timestamp = " << busMessage.Timestamp <<
                       ", offer id = " << message.GetOfferId() << Endl;
        return false;
    }

    TMaybe<TInstant> cacheTimestamp;
    if (message.HasCacheTimestamp()) {
        cacheTimestamp = TInstant::Seconds(message.GetCacheTimestamp().seconds());
    }
    TString messageId = busMessage.MessageId;
    auto onCompletionCallback = [this, messageId]() {
        PriceCheckReqBus_.MarkRecordAsCompleted(messageId);
    };

    TOfferId offerId;
    if (!GetUuid(message.GetOfferId(), offerId)) {
        ythrow yexception() << "Can't parse offer id as uuid: " << message.GetOfferId();
    }
    JobProcessor_.AddExternalJob(offerId, cacheTimestamp, onCompletionCallback, busMessage.Timestamp);
    return true;
}

EPartnerId TService::GetPartnerIdByOperatorId(EOperatorId opId) const {
    auto op = YtConfigOperators_.GetById(opId);
    if (op) {
        return op->GetPartnerId();
    }
    return NTravelProto::PI_UNUSED;
}

} // namespace NPriceChecker
} // namespace NTravel
