package ru.yandex.travel.hotels.promogranter.repositories

import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.DependsOn
import org.springframework.stereotype.Component
import ru.yandex.travel.hotels.promogranter.PromoGranterProperties
import ru.yandex.travel.hotels.promogranter.db.tables.AdditionalBindings
import ru.yandex.travel.hotels.promogranter.db.tables.AdditionalBindings.ADDITIONAL_BINDINGS
import ru.yandex.travel.hotels.promogranter.db.tables.records.AdditionalBindingsRecord
import java.time.LocalDateTime
import java.util.*


@Component
@DependsOn("flyway", "flywayInitializer")
class AdditionalBindingsRepository(
    private val dsl: DSLContext,
    private val promoGranterProperties: PromoGranterProperties,
) {
    private val log = LoggerFactory.getLogger(this.javaClass)

    fun getBindingForOrder(topupCategory: String, travelOrderId: String): AdditionalBindingsRecord? {
        return dsl.fetchOne(ADDITIONAL_BINDINGS, ADDITIONAL_BINDINGS.TOPUP_CATEGORY.eq(topupCategory)
            .and(ADDITIONAL_BINDINGS.TRAVEL_ORDER_ID.eq(travelOrderId)))
    }

    fun getAll(): List<AdditionalBindingsRecord> {
        return dsl.select()
            .from(ADDITIONAL_BINDINGS)
            .fetch()
            .map { x -> x as AdditionalBindingsRecord }
    }

    fun createAdditionalBindingIfNotExist(
        category: String, travelOrderId: String, userIp: String, passportId: String
    ): Boolean {
        var created = false

        dsl.transactionSerializable(promoGranterProperties) { configuration ->
            val ctx = DSL.using(configuration)

            if (ctx.fetchExists(ADDITIONAL_BINDINGS, ADDITIONAL_BINDINGS.TOPUP_CATEGORY.eq(category).and(ADDITIONAL_BINDINGS.TRAVEL_ORDER_ID.eq(travelOrderId)))) {
                return@transactionSerializable
            }

            val record = ctx.newRecord(ADDITIONAL_BINDINGS)
            record.bindingId = UUID.randomUUID()
            record.topupCategory = category
            record.travelOrderId = travelOrderId
            record.createdAt = LocalDateTime.now()
            record.updatedAt = LocalDateTime.now()
            record.userIp = userIp
            record.passportId = passportId

            log.debug("Creating additional binding\n$record")

            record.store()

            created = true
        }

        return created
    }

}
