package ru.yandex.travel.hotels.promogranter.repositories

import com.google.common.base.Preconditions
import org.jooq.*
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.context.annotation.DependsOn
import org.springframework.stereotype.Component
import ru.yandex.travel.hotels.promogranter.PromoGranterProperties
import ru.yandex.travel.hotels.promogranter.db.tables.PlusTopups.PLUS_TOPUPS
import ru.yandex.travel.hotels.promogranter.db.tables.records.PlusTopupsRecord
import ru.yandex.travel.hotels.promogranter.proto.ETopupStatus
import java.math.BigDecimal
import java.time.LocalDateTime
import java.util.*


@Component
@DependsOn("flyway", "flywayInitializer")
class PlusTopupsRepository(
    private val dsl: DSLContext,
    private val promoGranterProperties: PromoGranterProperties,
) {
    private val log = LoggerFactory.getLogger(this.javaClass)

    fun getAll(): List<PlusTopupsRecord> {
        return dsl.select()
            .from(PLUS_TOPUPS)
            .fetch()
            .map { x -> x as PlusTopupsRecord }
    }

    fun getNonScheduled(): List<PlusTopupsRecord> {
        return dsl.select()
            .from(PLUS_TOPUPS)
            .where(PLUS_TOPUPS.SCHEDULED.isFalse)
            .fetch()
            .map { x -> x as PlusTopupsRecord }
    }

    fun getScheduledNonFinished(): List<PlusTopupsRecord> {
        return dsl.select()
            .from(PLUS_TOPUPS)
            .where(PLUS_TOPUPS.SCHEDULED.isTrue.and(PLUS_TOPUPS.FINISHED.isFalse))
            .fetch()
            .map { x -> x as PlusTopupsRecord }
    }

    fun tryGetByTravelOrderId(category: String, travelOrderId: String): PlusTopupsRecord? {
        val res = dsl.select()
            .from(PLUS_TOPUPS)
            .where(PLUS_TOPUPS.TOPUP_CATEGORY.eq(category).and(PLUS_TOPUPS.TRAVEL_ORDER_ID.eq(travelOrderId)))
            .fetch()
            .map { x -> x as PlusTopupsRecord }
        Preconditions.checkState(res.size <= 1, "Found several orders with category=$category and travelOrderId=$travelOrderId")
        return if (res.isEmpty()) null else res.get(0)
    }

    fun markScheduled(topupId: UUID) {
        val topup = dsl.fetchOne(PLUS_TOPUPS, PLUS_TOPUPS.PLUS_TOPUP_ID.eq(topupId))
        topup.scheduled = true
        topup.store()
    }

    fun updateStatus(topupId: UUID, topupStatus: ETopupStatus, purchaseToken: String?, finishedAt: LocalDateTime?) {
        val topup = dsl.fetchOne(PLUS_TOPUPS, PLUS_TOPUPS.PLUS_TOPUP_ID.eq(topupId))
        topup.status = topupStatus.number
        if (topupStatus == ETopupStatus.TS_SUCCEED) {
            topup.finished = true
            topup.finishedAt = finishedAt
            topup.trustPurchaseToken = purchaseToken
        }
        topup.store()
    }

    fun createTopupIfNotExist(
        category: String, travelOrderId: String, topupAmount: BigDecimal,
        orderAmount: BigDecimal, commissionAmount: BigDecimal, vatCommissionAmount: BigDecimal,
        userIp: String, passportId: String
    ): Boolean {
        var created = false

        dsl.transactionSerializable(promoGranterProperties) { configuration ->
            val ctx = DSL.using(configuration)

            if (ctx.fetchExists(PLUS_TOPUPS, PLUS_TOPUPS.TOPUP_CATEGORY.eq(category).and(PLUS_TOPUPS.TRAVEL_ORDER_ID.eq(travelOrderId)))) {
                return@transactionSerializable
            }
            val record = ctx.newRecord(PLUS_TOPUPS)
            record.plusTopupId = UUID.randomUUID()
            record.topupCategory = category
            record.travelOrderId = travelOrderId
            record.createdAt = LocalDateTime.now()
            record.updatedAt = LocalDateTime.now()
            record.topupAmount = topupAmount
            record.orderAmount = orderAmount
            record.commissionAmount = commissionAmount
            record.vatCommissionAmount = vatCommissionAmount
            record.userIp = userIp
            record.passportId = passportId
            record.scheduled = false
            record.finished = false

            log.debug("Creating topup\n$record")

            record.store()

            created = true
        }

        return created
    }
}
