package ru.yandex.travel.hotels.promogranter.services.orders_client;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.retry.RetryRateLimiter;
import ru.yandex.travel.commons.retry.RetryStrategy;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusReq;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusRsp;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupReq;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupRsp;
import ru.yandex.travel.orders.yandex_plus.YandexPlusServiceGrpc;

import static ru.yandex.travel.commons.concurrent.FutureUtils.buildCompletableFuture;

@Slf4j
@Component
public class OrchestratorClient {
    private final HAGrpcChannelFactory haGrpcChannelFactory;
    private final RetryRateLimiter rateLimiter;
    private final Retry retryHelper;

    public OrchestratorClient(@Qualifier("OrchestratorGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory,
                             Retry retryHelper) {
        this.haGrpcChannelFactory = haGrpcChannelFactory;
        this.rateLimiter = new RetryRateLimiter(0.3);
        this.retryHelper = retryHelper;
    }

    public CompletableFuture<TSchedulePlusTopupRsp> schedulePlusTopup(String logId, TSchedulePlusTopupReq schedulePlusTopupReq) {
        return retryHelper.withRetry("OrchestratorClient::schedulePlusTopup/" + logId,
                req -> buildCompletableFuture(getYandexPlusFutureStub().schedulePlusTopup(req)),
                schedulePlusTopupReq,
                getRetryStrategy(),
                rateLimiter);
    }

    public CompletableFuture<TGetPlusTopupStatusRsp> getPlusTopupStatus(String logId, TGetPlusTopupStatusReq getPlusTopupStatusReq) {
        return retryHelper.withRetry("OrchestratorClient::getPlusTopupStatus/" + logId,
                req -> buildCompletableFuture(getYandexPlusFutureStub().getPlusTopupStatus(req)),
                getPlusTopupStatusReq,
                getRetryStrategy(),
                rateLimiter);
    }

    private <T> RetryStrategy<T> getRetryStrategy() {
        // TODO: use proper retry strategy
        return new RetryStrategy<>() {
            @Override
            public boolean shouldRetryOnException(Exception ex) {
                return true;
            }

            @Override
            public void validateResult(T result) {

            }

            @Override
            public Duration getWaitDuration(int iteration, Exception ex, T result) {
                return Duration.ofSeconds(10);
            }

            @Override
            public int getNumRetries() {
                return 5;
            }
        };
    }

    private YandexPlusServiceGrpc.YandexPlusServiceFutureStub getYandexPlusFutureStub() {
        return YandexPlusServiceGrpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
    }
}
