package ru.yandex.travel.hotels.promogranter.services.plus_topup

import com.google.common.base.Preconditions
import io.grpc.Status
import io.micrometer.core.instrument.Metrics
import io.micrometer.core.instrument.Tags
import org.javamoney.moneta.Money
import org.slf4j.LoggerFactory
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Component
import ru.yandex.misc.thread.factory.ThreadNameThreadFactory
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.commons.proto.ECurrency
import ru.yandex.travel.commons.proto.EErrorCode
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.cpa.tours.EEnvironment
import ru.yandex.travel.hotels.promogranter.db.tables.records.AdditionalBindingsRecord
import ru.yandex.travel.hotels.promogranter.db.tables.records.PlusTopupsRecord
import ru.yandex.travel.hotels.promogranter.proto.ETopupEligibilityStatus
import ru.yandex.travel.hotels.promogranter.proto.ETopupStatus
import ru.yandex.travel.hotels.promogranter.proto.TToursCpaRecord
import ru.yandex.travel.hotels.promogranter.proto.TToursTopupInfo
import ru.yandex.travel.hotels.promogranter.repositories.AdditionalBindingsRepository
import ru.yandex.travel.hotels.promogranter.repositories.PlusTopupsRepository
import ru.yandex.travel.hotels.promogranter.services.orders_client.OrchestratorClient
import ru.yandex.travel.hotels.promogranter.services.remote_locks.RemoteLockService
import ru.yandex.travel.hotels.promogranter.services.tours_cpa.ToursCpaService
import ru.yandex.travel.hotels.promogranter.services.yt_publish.YtDataPublisher
import ru.yandex.travel.hotels.proto.promogranter_service.TCreateAdditionalBindingRsp
import ru.yandex.travel.orders.yandex_plus.EPaymentProfile
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusReq
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusRsp
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupReq
import java.math.BigDecimal
import java.math.RoundingMode
import java.time.Instant
import java.time.LocalDate
import java.time.ZoneId
import java.util.*
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.TimeUnit
import java.util.stream.Collectors
import javax.annotation.PostConstruct
import javax.annotation.PreDestroy

@Component
@EnableConfigurationProperties(PlusTopupInitializerServiceProperties::class)
class PlusTopupInitializerService(private val config: PlusTopupInitializerServiceProperties,
                                  private val toursCpaService: ToursCpaService,
                                  private val plusTopupsRepository: PlusTopupsRepository,
                                  private val orchestratorClient: OrchestratorClient,
                                  private val additionalBindingsRepository: AdditionalBindingsRepository,
                                  private val remoteLockService: RemoteLockService,
                                  private val ytDataPublisher: YtDataPublisher) {
    private val executor: ScheduledExecutorService = Executors.newSingleThreadScheduledExecutor(ThreadNameThreadFactory("topup-initializer-thread"))
    private val log = LoggerFactory.getLogger(this.javaClass)

    @PostConstruct
    fun init() {
        if (config.enabled) {
            log.info("Starting plus topup initializer")
            executor.scheduleWithFixedDelay({ initializeNewTopups() }, 0, config.period.seconds, TimeUnit.SECONDS)
            executor.scheduleWithFixedDelay({ scheduleNewTopups() }, 0, config.period.seconds, TimeUnit.SECONDS)
            executor.scheduleWithFixedDelay({ syncTopupStatuses() }, 0, config.period.seconds, TimeUnit.SECONDS)
            executor.scheduleWithFixedDelay({ publishToursToYt() }, 0, config.toursTopupInfoYtExportPeriod.seconds, TimeUnit.SECONDS)
        }
    }

    private fun initializeNewTopups() {
        if (!config.initializationOfTopupsEnabled) {
            return
        }
        if (!toursCpaService.isReady) {
            log.info("Skipping initialization of topups because toursCpaService is not ready")
            return
        }
        runJob(RemoteLockService.LockId.PLUS_TOPUP_INITIALIZATION, "initialization-of-topups") {
            val allToursInCpa = toursCpaService.allTours
            val orderIdsWithTopups = plusTopupsRepository.getAll().stream()
                .map { obj: PlusTopupsRecord -> obj.travelOrderId }
                .collect(Collectors.toUnmodifiableSet())
            for (tour in allToursInCpa) {
                if (orderIdsWithTopups.contains(tour.travelOrderId)) {
                    continue
                }

                Preconditions.checkArgument(tour.currencyCode == "RUB", "Found tour with non-rub currency")

                val additionalBindingsRecord = additionalBindingsRepository.getBindingForOrder(config.topupCategory, tour.travelOrderId)
                val status = getTopupEligibilityStatus(tour, additionalBindingsRecord)
                if (status != ETopupEligibilityStatus.TES_ELIGIBLE) {
                    continue
                }

                val userIp = if (additionalBindingsRecord != null) additionalBindingsRecord.userIp else tour.labelUserIp
                val passportId = if (additionalBindingsRecord != null) additionalBindingsRecord.passportId else tour.labelPassportUid

                val plusMultiplier = BigDecimal.valueOf(config.toursPlusPercent).divide(BigDecimal.valueOf(100))
                val vatMultiplier = BigDecimal.valueOf(config.toursCommissionVatPercent).divide(BigDecimal.valueOf(100))

                val priceBase = tour.orderAmount + tour.fuelCharge
                val plusAmount = minOf(BigDecimal.valueOf(priceBase).multiply(plusMultiplier),
                    BigDecimal.valueOf(config.toursPlusMaxAmount)).setScale(0, RoundingMode.HALF_UP)

                plusTopupsRepository.createTopupIfNotExist(config.topupCategory,
                    tour.travelOrderId,
                    plusAmount,
                    BigDecimal.valueOf(tour.orderAmount).setScale(2, RoundingMode.HALF_UP),
                    BigDecimal.valueOf(tour.profitAmount).setScale(2, RoundingMode.HALF_UP),
                    BigDecimal.valueOf(tour.profitAmount).multiply(vatMultiplier).setScale(2, RoundingMode.HALF_UP),
                    userIp,
                    passportId)
            }
        }
    }

    private fun scheduleNewTopups() {
        if (!config.schedulingOfTopupsEnabled) {
            return
        }
        runJob(RemoteLockService.LockId.PLUS_TOPUP_SCHEDULING, "scheduling-of-topups") {
            val allUnprocessedTopups = plusTopupsRepository.getNonScheduled()
            for (topup in allUnprocessedTopups) {
                log.info("Scheduling plus topup ${topup.plusTopupId} (order id ${topup.travelOrderId})")
                val plusTopupReq = TSchedulePlusTopupReq.newBuilder()
                    .setOperationId(topup.plusTopupId.toString())
                    .setPoints(topup.topupAmount.intValueExact())
                    .setCurrency(ECurrency.C_RUB) // we store only rub topups
                    .setPassportId(topup.passportId)
                    .setExternalOrderId(topup.travelOrderId)
                    .setTotalAmountForPayload(ProtoUtils.toTPrice(Money.of(topup.orderAmount, "RUB")))
                    .setCommissionAmountForPayload(ProtoUtils.toTPrice(Money.of(topup.commissionAmount, "RUB")))
                    .setVatCommissionAmountForPayload(ProtoUtils.toTPrice(Money.of(topup.vatCommissionAmount, "RUB")))
                    .setPaymentProfile(EPaymentProfile.PP_HOTEL)
                    .setUserIp(topup.userIp)
                    .setSkipFinancialEvents(true)
                    .build()
                orchestratorClient.schedulePlusTopup(topup.plusTopupId.toString(), plusTopupReq)
                    .handle { rsp, error ->
                        val alreadyExistsError = Optional.ofNullable(error)
                            .map { Status.trailersFromThrowable(it) }
                            .map { it?.get(ServerUtils.METADATA_ERROR_KEY) }
                            .map { it!!.code }
                            .filter { it === EErrorCode.EC_ALREADY_EXISTS }
                            .isPresent

                        if (error != null) {
                            if (!alreadyExistsError) {
                                throw RuntimeException("Failed to schedule topup ${topup.plusTopupId}", error)
                            }
                        } else {
                            if (!rsp.scheduled) {
                                throw RuntimeException("Failed to schedule topup ${topup.plusTopupId} (rsp.scheduled = false)")
                            }
                        }
                        plusTopupsRepository.markScheduled(topup.plusTopupId)
                    }
                    .join()
            }
        }
    }

    private fun syncTopupStatuses() {
        if (!config.syncTopupStatusesEnabled) {
            return
        }
        runJob(RemoteLockService.LockId.PLUS_TOPUP_STATUSES_SYNC, "topup-statuses-sync") {
            val allUnprocessedTopups = plusTopupsRepository.getScheduledNonFinished()
            for (topup in allUnprocessedTopups) {
                log.info("Checking plus topup status ${topup.plusTopupId} (order id ${topup.travelOrderId})")

                val plusTopupStatusReq = TGetPlusTopupStatusReq.newBuilder()
                    .setOperationId(topup.plusTopupId.toString())
                    .build()
                orchestratorClient.getPlusTopupStatus(topup.plusTopupId.toString(), plusTopupStatusReq)
                    .handle { rsp, error ->
                        if (error != null) {
                            throw RuntimeException("Failed to check topup status ${topup.plusTopupId}", error)
                        }

                        val topupStatus = getTopupStatus(rsp)
                        if (rsp.hasTopupInfo() && rsp.topupInfo.amount != topup.topupAmount.intValueExact()) {
                            throw RuntimeException("Unexpected amount in topup ${topup.plusTopupId} (${rsp.topupInfo.amount} != ${topup.topupAmount.intValueExact()})")
                        }
                        if (topupStatus == ETopupStatus.TS_SUCCEED) {
                            if (rsp.topupInfo.purchaseToken.isNullOrEmpty()) {
                                throw RuntimeException("Empty purchaseToken in topup ${topup.plusTopupId}")
                            }
                            plusTopupsRepository.updateStatus(topup.plusTopupId, topupStatus, rsp.topupInfo.purchaseToken, ProtoUtils.toLocalDateTime(rsp.topupInfo.finishedAt))
                        } else {
                            plusTopupsRepository.updateStatus(topup.plusTopupId, topupStatus, null, null)
                        }
                    }
                    .join()
            }
        }
    }

    private fun publishToursToYt() {
        if (!config.exportToYtEnabled) {
            return
        }
        if (!toursCpaService.isReady) {
            log.info("Skipping export to yt because toursCpaService is not ready")
            return
        }
        runJob(RemoteLockService.LockId.EXPORT_TOURS_TO_YT, "export-tours-to-yt") {
            val allToursInCpa = toursCpaService.getAllTours()
            val plusTopupRecords = plusTopupsRepository.getAll()
            val additionalBindings = additionalBindingsRepository.getAll()

            val topupRecordsMap = plusTopupRecords.stream().collect(Collectors.toUnmodifiableMap({ x -> Pair(x.topupCategory, x.travelOrderId) }, { x -> x }))
            val additionalBindingsMap = additionalBindings.stream().collect(Collectors.toUnmodifiableMap({ x -> Pair(x.topupCategory, x.travelOrderId) }, { x -> x }))
            val toursTopupInfoRecords = allToursInCpa.stream()
                .map { cpaRecord ->
                    val key = Pair(config.topupCategory, cpaRecord.travelOrderId)
                    val plusTopupRecord = topupRecordsMap.get(key)
                    val additionalBindingsRecord = additionalBindingsMap.get(key)

                    val builder = TToursTopupInfo.newBuilder()

                    builder.setTravelOrderId(cpaRecord.travelOrderId)
                    builder.setPartnerOrderId(cpaRecord.partnerOrderId)
                    builder.setPartnerName(cpaRecord.partnerName)
                    val eligibilityStatus = getTopupEligibilityStatus(cpaRecord, additionalBindingsRecord)
                    builder.setPlusEligibilityStatus(eligibilityStatus)
                    if (eligibilityStatus == ETopupEligibilityStatus.TES_ELIGIBLE ||
                        eligibilityStatus == ETopupEligibilityStatus.TES_ORDER_NOT_CONFIRMED ||
                        eligibilityStatus == ETopupEligibilityStatus.TES_TOO_EARLY) {
                        builder.setPlusExpectedTopupDate(getExpectedTopupDate(cpaRecord, additionalBindingsRecord).toString())
                    }
                    if (plusTopupRecord != null) {
                        builder.setPlusAmount(plusTopupRecord.topupAmount.longValueExact())
                        if (plusTopupRecord.status != null) {
                            builder.setPlusTopupStatus(ETopupStatus.forNumber(plusTopupRecord.status))
                        }
                        if (plusTopupRecord.finishedAt != null) {
                            builder.setPlusActualTopupDate(plusTopupRecord.finishedAt.toLocalDate().toString())
                        }
                    }

                    builder.build()
                }
                .collect(Collectors.toUnmodifiableList())

            ytDataPublisher.publishToYt(toursTopupInfoRecords, config.toursTopupInfoYtPath)
        }
    }

    private fun runJob(lockId: RemoteLockService.LockId, jobId: String, action: () -> Unit) {
        try {
            remoteLockService.executeWithLock(lockId) {
                log.info("Running $jobId")
                action()
            }
            Metrics.globalRegistry.counter("regular-job-runner.success", Tags.of("job_id", jobId)).increment()
        } catch (e: Exception) {
            Metrics.globalRegistry.counter("regular-job-runner.errors", Tags.of("job_id", jobId)).increment()
            log.error("Failed to execute '$jobId'", e)
        }
    }

    fun tryCreateAdditionalBinding(tourId: String, checkInDate: String, checkOutDate: String, userIp: String, passportId: String): Pair<TCreateAdditionalBindingRsp.EBindingCreationStatus, LocalDate?> {
        val optionalCpaTour = toursCpaService.tryGetByPartnerOrderId(tourId)
        if (optionalCpaTour.isEmpty || optionalCpaTour.get().checkIn != checkInDate || optionalCpaTour.get().checkOut != checkOutDate) {
            return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ORDER_NOT_FOUND, null)
        }
        val cpaTour = optionalCpaTour.get()
        val travelOrderId = cpaTour.travelOrderId

        val existingTopup = plusTopupsRepository.tryGetByTravelOrderId(config.topupCategory, travelOrderId)
        if (existingTopup != null) {
            if (existingTopup.passportId == passportId) {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_CURRENT_USER, existingTopup.createdAt.toLocalDate())
            } else {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_OTHER_USER, null)
            }
        }

        val existingBinding = additionalBindingsRepository.getBindingForOrder(config.topupCategory, travelOrderId)

        val expectedTopupDate = getExpectedTopupDate(cpaTour, null);

        if (existingBinding != null) {
            if (existingBinding.passportId == passportId) {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_CURRENT_USER, expectedTopupDate)
            } else {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_OTHER_USER, null)
            }
        }

        val eligibilityStatus = getTopupEligibilityStatus(cpaTour, null)

        if (eligibilityStatus == ETopupEligibilityStatus.TES_NO_LABEL || eligibilityStatus == ETopupEligibilityStatus.TES_LABEL_IS_TOO_OLD) {
            return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_NO_LABEL, null)
        }

        if (eligibilityStatus == ETopupEligibilityStatus.TES_ORDER_NOT_CONFIRMED ||
            eligibilityStatus == ETopupEligibilityStatus.TES_TOO_EARLY ||
            eligibilityStatus == ETopupEligibilityStatus.TES_ELIGIBLE) {
            if (cpaTour.labelPassportUid == passportId) {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_CURRENT_USER, expectedTopupDate)
            } else {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_OTHER_USER, null)
            }
        }

        if (LocalDate.parse(cpaTour.checkOut).plusDays(config.additionalBindingTtlDays) < LocalDate.now()) {
            return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ORDER_IS_TOO_OLD, null)
        }

        if (eligibilityStatus != ETopupEligibilityStatus.TES_NO_PASSPORT_ID && eligibilityStatus != ETopupEligibilityStatus.TES_NOT_PLUS_USER) {
            log.error("Unexpectd status $eligibilityStatus for travelOrderId=$travelOrderId")
            return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ERROR, null)
        }

        val created = additionalBindingsRepository.createAdditionalBindingIfNotExist(config.topupCategory, travelOrderId, userIp, passportId)
        val newBinding = additionalBindingsRepository.getBindingForOrder(config.topupCategory, travelOrderId)
        Preconditions.checkState(newBinding != null, if (created) "Binding created but not found then" else "Binding not created but not found then")
        if (!created) {
            if (newBinding!!.passportId == passportId) {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_CURRENT_USER, getExpectedTopupDate(cpaTour, newBinding))
            } else {
                return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_ALREADY_BOUND_TO_OTHER_USER, null)
            }
        } else {
            return Pair(TCreateAdditionalBindingRsp.EBindingCreationStatus.BCS_CREATED, getExpectedTopupDate(cpaTour, newBinding))
        }
    }

    private fun getTopupStatus(rsp: TGetPlusTopupStatusRsp): ETopupStatus {
        if (rsp.schedulingStatus == TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_UNKNOWN) {
            return ETopupStatus.TS_WAITING_SCHEDULING
        }
        if (rsp.schedulingStatus == TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_FAILED) {
            return ETopupStatus.TS_SCHEDULING_FAILED
        }
        if (rsp.schedulingStatus == TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_SUCCEED) {
            if (!rsp.hasTopupInfo() || !rsp.topupInfo.succeed) {
                return ETopupStatus.TS_WAITING_FOR_TOPUP
            }
            return ETopupStatus.TS_SUCCEED
        }
        return ETopupStatus.TS_UNKNOWN
    }

    private fun getTopupEligibilityStatus(tour: TToursCpaRecord, additionalBindingsRecord: AdditionalBindingsRecord?): ETopupEligibilityStatus {
        if (!tour.hasLabel) {
            return ETopupEligibilityStatus.TES_NO_LABEL
        }
        if (config.expectedLabelEnv != tour.labelEnvironment) {
            return ETopupEligibilityStatus.TES_NO_LABEL
        }
        if (LocalDate.parse(tour.labelLabelGenerationDate).plusDays(config.toursLabelTtlDays) < getTourCreationDate(tour)) {
            return ETopupEligibilityStatus.TES_LABEL_IS_TOO_OLD
        }
        if (additionalBindingsRecord == null) {
            if (tour.labelPassportUid.isNullOrEmpty()) {
                return ETopupEligibilityStatus.TES_NO_PASSPORT_ID
            }
            if (!tour.labelIsPlusUser) {
                return ETopupEligibilityStatus.TES_NOT_PLUS_USER
            }
        }
        if ((tour.status != "confirmed" && !isEarlyTopup(tour)) || tour.status == "cancelled") {
            return ETopupEligibilityStatus.TES_ORDER_NOT_CONFIRMED
        }
        if (getExpectedTopupDate(tour, additionalBindingsRecord) > LocalDate.now()) {
            return ETopupEligibilityStatus.TES_TOO_EARLY
        }
        return ETopupEligibilityStatus.TES_ELIGIBLE
    }

    private fun getExpectedTopupDate(tour: TToursCpaRecord, additionalBindingsRecord: AdditionalBindingsRecord?): LocalDate {
        if (isEarlyTopup(tour)) {
            return getTourCreationDate(tour)
        }
        val minDate = LocalDate.parse(tour.checkOut)
        if (additionalBindingsRecord == null) {
            return minDate
        }
        return maxOf(additionalBindingsRecord.createdAt.toLocalDate(), minDate)
    }

    private fun isEarlyTopup(tour: TToursCpaRecord): Boolean {
        return config.dangerouslyTopupImmediatelyUseOnlyInTesting && tour.labelEnvironment == EEnvironment.E_Testing
    }

    private fun getTourCreationDate(tour: TToursCpaRecord): LocalDate {
        return LocalDate.ofInstant(Instant.ofEpochSecond(tour.createdAt), ZoneId.of("UTC"))
    }

    @PreDestroy
    fun destroy() {
        executor.shutdownNow()
    }
}
