package ru.yandex.travel.hotels.promogranter.services.remote_locks

import org.jooq.DSLContext
import org.jooq.impl.DSL
import org.slf4j.LoggerFactory
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Component
import ru.yandex.travel.hotels.promogranter.PromoGranterProperties
import ru.yandex.travel.hotels.promogranter.services.plus_topup.PlusTopupInitializerServiceProperties


@Component
@EnableConfigurationProperties(PromoGranterProperties::class)
class RemoteLockService(
    private val dsl: DSLContext,
    private val promoGranterProperties: PromoGranterProperties
) {
    enum class LockId(val id: Int) {
        PLUS_TOPUP_INITIALIZATION(1),
        PLUS_TOPUP_SCHEDULING(2),
        PLUS_TOPUP_STATUSES_SYNC(3),
        EXPORT_TOURS_TO_YT(4),
    }

    private val log = LoggerFactory.getLogger(this.javaClass)

    fun executeWithLock(lockId: LockId, action: () -> Unit): Boolean {
        if (promoGranterProperties.usePostgresSpecificCommands) {
            var result = false
            dsl.connection { connection ->
                val ctx = DSL.using(connection)
                log.debug("trying to get lock $lockId")
                if (!lock(ctx, lockId)) {
                    result = false
                    return@connection
                }
                try {
                    action()
                } finally {
                    unlock(ctx, lockId)
                }
                result = true
                return@connection
            }
            return result
        } else {
            action()
            return true
        }
    }

    private fun lock(ctx: DSLContext, lockId: LockId): Boolean {
        val res = ctx.fetchOne("SELECT pg_try_advisory_lock(?)", lockId.id)?.get(0) as Boolean
        if (res) {
            log.debug("Successfully acquired lock $lockId")
        } else {
            log.debug("Failed to acquire lock $lockId")
        }
        return res
    }

    private fun unlock(ctx: DSLContext, lockId: LockId): Boolean {
        val res = ctx.fetchOne("SELECT pg_advisory_unlock(?)", lockId.id)?.get(0) as Boolean
        if (res) {
            log.debug("Successfully released lock $lockId")
        } else {
            log.error("Failed to release lock $lockId")
        }
        return res
    }
}
