package ru.yandex.travel.hotels.promogranter.services.tours_cpa;

import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.promogranter.proto.TToursCpaRecord;
import ru.yandex.travel.yt_lucene_index.MultiMapPersistentConfig;

@Component
@EnableConfigurationProperties(ToursCpaServiceProperties.class)
@ConditionalOnProperty("tours-cpa.enabled")
@Slf4j
public class ToursCpaServiceImpl implements ToursCpaService {

    private final MultiMapPersistentConfig<String, TToursCpaRecord, TToursCpaRecord> persistentConfig;

    public ToursCpaServiceImpl(ToursCpaServiceProperties params) {
        if (params.isEnabled()) {
            persistentConfig = new MultiMapPersistentConfig<>(params, "ToursCpa",
                    TToursCpaRecord::newBuilder, TToursCpaRecord::getPartnerOrderId, Function.identity(),
                    Function.identity());
        } else {
            persistentConfig = null;
        }
    }

    public List<TToursCpaRecord> getAllTours() {
        ensureEnabled();
        return persistentConfig.getAll().values().stream().flatMap(Collection::stream).collect(Collectors.toUnmodifiableList());
    }

    public Optional<TToursCpaRecord> tryGetByPartnerOrderId(String partnerOrderId) {
        ensureEnabled();
        var orders = persistentConfig.getByKey(partnerOrderId);
        if (orders == null) {
            return Optional.empty();
        }
        Preconditions.checkState(orders.size() >= 1, "Found several orders with partnerOrderId=%s", partnerOrderId);
        return orders.stream().findFirst();
    }

    private void ensureEnabled() {
        if (persistentConfig == null) {
            throw new IllegalStateException("ToursCpa service disabled");
        }
    }

    @PostConstruct
    public void init() {
        if (persistentConfig != null) {
            persistentConfig.start();
        }
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        if (persistentConfig != null) {
            persistentConfig.stop();
        }
    }

    public boolean isReady() {
        return persistentConfig == null || persistentConfig.isReady();
    }
}
