package ru.yandex.travel.hotels.promogranter.services.yt_publish

import io.micrometer.core.instrument.Counter
import io.micrometer.core.instrument.Metrics
import io.micrometer.core.instrument.Timer
import org.slf4j.LoggerFactory
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.stereotype.Component
import ru.yandex.inside.yt.kosher.Yt
import ru.yandex.inside.yt.kosher.impl.YtUtils
import ru.yandex.inside.yt.kosher.tables.types.NativeProtobufEntryType
import ru.yandex.travel.commons.metrics.MetricsUtils
import ru.yandex.travel.hotels.promogranter.proto.TToursTopupInfo
import ru.yandex.travel.yt.util.YtHelper
import ru.yandex.travel.yt.util.YtProtoUtils
import java.util.*
import java.util.function.Function
import java.util.stream.Collectors


@Component
@EnableConfigurationProperties(YtPublisherProperties::class)
@ConditionalOnProperty("yt-publisher.enabled")
class YtDataPublisherImpl(private val ytPublisherProperties: YtPublisherProperties) : YtDataPublisher {
    private val log = LoggerFactory.getLogger(this.javaClass)
    private val ytClusters: Map<String, ClusterInfo> = ytPublisherProperties.clusters.stream()
        .collect(Collectors.toMap(Function.identity(), { clusterName -> ClusterInfo(clusterName, ytPublisherProperties.token) }))

    class ClusterInfo(
        clusterName: String,
        token: String,
    ) {
        val yt: Yt = YtUtils.http(clusterName, token)
        val errorCounter: Counter = Counter.builder("yt-publisher.errors")
            .tag("clusterName", clusterName)
            .register(Metrics.globalRegistry)
        val timer: Timer = Timer.builder("yt-publisher.publish-time")
            .tag("clusterName", clusterName)
            .serviceLevelObjectives(*MetricsUtils.mediumDurationSla())
            .publishPercentiles(*MetricsUtils.higherPercentiles())
            .publishPercentileHistogram()
            .register(Metrics.globalRegistry)
    }

    override fun publishToYt(records: List<TToursTopupInfo>, path: String) {
        val occurredExceptions: MutableList<Exception> = ArrayList()
        for ((cluster, clusterInfo) in ytClusters) {
            clusterInfo.timer.record {
                try {
                    YtHelper.doInTx(clusterInfo.yt, ytPublisherProperties.transactionDuration) { txId ->
                        log.info("Publishing tours topup infos in yt transaction [$txId] to $cluster")

                        val toursTopupInfoTable = YtHelper.createTable(clusterInfo.yt, txId, true,
                                path, YtProtoUtils.getTableSchemaForMessage(TToursTopupInfo.getDefaultInstance()))
                        val toursTopupInfoEntryType = NativeProtobufEntryType<TToursTopupInfo>(TToursTopupInfo.newBuilder())

                        clusterInfo.yt.tables().write(Optional.of(txId), true, toursTopupInfoTable, toursTopupInfoEntryType, records.iterator())
                    }
                } catch (e: Exception) {
                    clusterInfo.errorCounter.increment()
                    occurredExceptions.add(e)
                }
            }
        }
        if (!occurredExceptions.isEmpty()) {
            val resultingException = RuntimeException("${occurredExceptions.size} exceptions occurred while publishing tours topup infos to yt")
            for (exception in occurredExceptions) {
                resultingException.addSuppressed(exception)
            }
            throw resultingException
        }
    }
}
