#include <travel/hotels/redir/service/service.h>
#include <travel/hotels/redir/proto/config.pb.h>
#include <travel/hotels/lib/cpp/protobuf/config.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>
#include <travel/hotels/lib/cpp/util/signals.h>

#include <library/cpp/getopt/opt.h>
#include <library/cpp/logger/global/global.h>
#include <library/cpp/sighandler/async_signals_handler.h>


int main(int argc, const char* argv[]) {
    SetupSignalHandlers();
    InitGlobalLog2Console(TLOG_INFO);
    TString configDir, env, labelKeyPath, travelTokenKeyPath, ytTokenPath, addInfoKeyPath;
    bool dbg;
    {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption('c', "config-dir", "Path to config dir")
            .Required()
            .StoreResult(&configDir);
        opts.AddLongOption('e', "env", "Environment")
            .StoreResult(&env)
            .DefaultValue("dev");
        opts.AddLongOption('l', "labelkey-path", "Label Key Path")
            .StoreResult(&labelKeyPath);
        opts.AddLongOption(  0, "travel-token-path", "Travel Token Key Path")
            .StoreResult(&travelTokenKeyPath);
        opts.AddLongOption(  0, "add-info-key-path", "AddInfo Key Path")
            .StoreResult(&addInfoKeyPath);
        opts.AddLongOption('t', "yt-token-path", "Yt TokenPath")
            .StoreResult(&ytTokenPath);
        opts.AddLongOption('d', "debug", "Enable debug log")
            .SetFlag(&dbg).NoArgument();
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
    }

    //-------- Read service config
    auto pbCfg = NTravel::ReadAppConfigWithSubstitutions<NTravelProto::NRedir::TConfig>("redir", configDir, env, ytTokenPath, "");

    if (labelKeyPath) {
        INFO_LOG << "Label Key Path override to " << labelKeyPath << Endl;
        pbCfg.MutableOther()->SetLabelKeyPath(labelKeyPath);
    }

    if (travelTokenKeyPath) {
        INFO_LOG << "TravelToken  Key Path override to " << travelTokenKeyPath << Endl;
        pbCfg.MutableOther()->SetTravelTokenKeyPath(travelTokenKeyPath);
    }

    if (addInfoKeyPath) {
        INFO_LOG << "AddInfo  Key Path override to " << addInfoKeyPath << Endl;
        pbCfg.MutableOther()->SetAddInfoKeyPath(addInfoKeyPath);
    }

    DoInitGlobalLog(pbCfg.GetOther().GetMainLogFile(), dbg ? TLOG_DEBUG : TLOG_INFO, false, false);

    {
        NTravel::NRedir::TService service(pbCfg);
        for (int signal: {SIGINT, SIGTERM}) {
            SetAsyncSignalFunction(signal, [&service](int) {
                service.Stop();
            });
        }
        service.Run();
    }
    INFO_LOG << "Shutting down YT RPC..." << Endl;
    NTravel::NYtRpc::Shutdown();
    INFO_LOG << "Shutdown finished" << Endl;
    return 0;
}
