#include "hotels_job.h"

#include "service.h"
#include "tools.h"

#include <travel/hotels/redir/proto/reqans_hotels.pb.h>
#include <travel/hotels/proto/search_flow_offer_data/offer_data.pb.h>

#include <travel/hotels/lib/cpp/ages/ages.h>
#include <travel/hotels/lib/cpp/label/label.h>
#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h>
#include <travel/hotels/lib/cpp/protobuf/tools.h>
#include <travel/hotels/lib/cpp/util/base64.h>
#include <travel/hotels/lib/cpp/util/host_filter.h>

#include <util/string/vector.h>

#include <library/cpp/digest/crc32c/crc32c.h>

#define JOB_LOG LogPrefix

namespace NTravel {
namespace NRedir {
namespace NHotels {

TJob::TJob(TService& svc, TYtKeyValueStorage& searchFlowOfferDataStorage, NHttp::TOnResponse onResp, ui64 id)
    : Service(svc)
    , ResponseCb(onResp)
    , LogPrefix("Hotels_Id_" + ToString(id) + ": ")
    , Started(Now())
    , SearchFlowOfferDataStorage(searchFlowOfferDataStorage)
    , OpCounters(svc.HotelsPerRequestCounters(EOperatorId::OI_UNUSED))
    , LabelHashOverriden(false)
{
}

TJob::~TJob() {
}

void TJob::Run(const NHttp::TRequest& httpReq) {
    INFO_LOG << JOB_LOG << "Start job, query: " << httpReq.Cgi() << Endl;
    try {
        NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &Req);
        Parse();
    } catch (...) {
        ERROR_LOG << JOB_LOG << "Bad request: " << CurrentExceptionMessage() << Endl;
        OpCounters->NRequestErrors.Inc();
        NHttp::TResponse resp = NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST);
        resp.LogPrefix = LogPrefix;
        ResponseCb(resp);
        return;
    }
    switch (Req.GetMode()) {
        case NTravelProto::NRedir::NApi::THotels_RedirReq::Normal:
            ProcessModeNormal();
            break;
        case NTravelProto::NRedir::NApi::THotels_RedirReq::MMAds:
            ProcessModeMMAds();
            break;
    }
}

void TJob::ProcessModeNormal() {
    OpCounters->NRequests.Inc();
    auto cacheAgeSec = Started.Seconds() - Label.GetCacheTimestamp();
    OpCounters->CacheAgeHours.Update(cacheAgeSec / 3600);

    NTravelProto::ESurface surface = Label.GetSurface();
    if (surface > NTravelProto::ESurface_MAX) {
        surface = NTravelProto::S_UNKNOWN;
    }
    OpCounters->NSurface[surface].Inc();

    if (Service.Config().GetOther().HasMaxCacheAgeSec() &&
        cacheAgeSec > Service.Config().GetOther().GetMaxCacheAgeSec() &&
        Req.GetRedir()) {
        OpCounters->NRequestsByOldOffer.Inc();
        ProcessOldRequest();
    } else if (Req.HasOfferId() && Service.Config().GetOther().GetAllowOfferId()) {
        OpCounters->NRequestsByOfferId.Inc();
        ProcessByOfferId();
    } else if (Req.HasPUrl()) {
        OpCounters->NRequestsByUrl.Inc();
        ProcessByPUrl();
    } else if (Req.HasToken()) {
        OpCounters->NRequestsByToken.Inc();
        ProcessByToken();
    } else {
        OpCounters->NRequestsUnknown.Inc();
        throw yexception() << "No OfferId or Url/Token in request";
    }
}

void TJob::ProcessModeMMAds() {
    OpCounters->NRequestsMMAds.Inc();

    Url.Parse(Service.Config().GetOther().GetMMAdsUrl());

    Label.SetRedirDestination(NTravelProto::RD_Partner);
    OpCounters->NDestination[NTravelProto::RD_Partner].Inc();
    CalcLabelHash();
    PutLabelToUrl(true);
    Reply();

    WriteReqAns();
}

void TJob::Parse() {
    if (Req.HasProtoLabel()) {
        try {
            Service.LabelCodec().DecodeToMessage(Req.GetProtoLabel(), &Label, NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false));
        } catch (...) {
            OpCounters->NLabelDecodeErrors.Inc();
            throw yexception() << "Failed to decode proto label '" << Req.GetProtoLabel() << "', cause " << CurrentExceptionMessage();
        }
        OpCounters = Service.HotelsPerRequestCounters(Label.GetOperatorId());
    } else {
        if (Req.GetMode() != NTravelProto::NRedir::NApi::THotels_RedirReq::MMAds) {
            OpCounters->NRequestsNoLabel.Inc();
            throw yexception() << "No label in request";
        }
    }
#define OVERRIDE_UTM_IN_LABEL(_X_)                    \
    if (Req.has_utm_##_X_()) {                        \
        Label.set_##_X_(Req.utm_##_X_());             \
    }
    OVERRIDE_UTM_IN_LABEL(source);
    OVERRIDE_UTM_IN_LABEL(medium);
    OVERRIDE_UTM_IN_LABEL(campaign);
    OVERRIDE_UTM_IN_LABEL(content);
    OVERRIDE_UTM_IN_LABEL(term);

#undef OVERRIDE_UTM_IN_LABEL
    if (Req.HasToken()) {
        try {
            TString tokenBytes = Service.TokenCodec().Decode(Req.GetToken());
            if (!Token.ParseFromString(tokenBytes)) {
                throw yexception() << "Failed to parse protobuf in token";
            }
        } catch (...) {
            OpCounters->NTokenDecodeErrors.Inc();
            throw yexception() << "Failed to parse token: " << CurrentExceptionMessage();
        }
    }

    if (Req.HasPUrl()) {
        try {
            TString urlString = Service.UrlCodec().Decode(Req.GetPUrl());
            Url.Parse(urlString);
        } catch (...) {
            OpCounters->NUrlDecodeErrors.Inc();
            throw yexception() << "Failed to parse url: " << CurrentExceptionMessage();
        }
    }

    if (Req.GetLabelHash()) {
        LabelHash = Req.GetLabelHash();
        // TODO - logging is wrong when LabelHash is set; improve logging!
        LabelHashOverriden = true;
    }
    if (Req.GetAddInfo()) {
        try {
            TString addInfoBytes = Service.AddInfoCodec().Decode(Req.GetAddInfo());
            if (!AddInfo.ParseFromString(addInfoBytes)) {
                throw yexception() << "Failed to parse protobuf in AddInfo";
            }
        } catch (...) {
            OpCounters->NAddInfoDecodeErrors.Inc();
            throw yexception() << "Failed to parse AddInfo: " << CurrentExceptionMessage();
        }
    }
    Exp.Init(AddInfo.GetExp(), AddInfo.GetStrExp());
}

void TJob::ProcessOldRequest() {
    SetDefaultPortalSchemeAndHost();
    Url.SetPath("/hotels");
    OpCounters->NDestination[NTravelProto::RD_PortalFallback].Inc();
    Reply();
}

void TJob::ProcessByPUrl() {
    DoRedirByPUrl();
}

void TJob::DoRedirByPUrl() {
    Label.SetRedirDestination(NTravelProto::RD_Partner);
    OpCounters->NDestination[NTravelProto::RD_Partner].Inc();
    TryParsePreviousLabelFromLabelHash();
    CalcLabelHash();
    PutLabelToUrl(true);
    Reply();

    WritePriceCheckerRequest();
    WriteReqAns();
}

void TJob::TryParsePreviousLabelFromLabelHash() {
    // Если нам пришёл LabelHash, содержащий ProtoLabel, то надо извлечь его, и использовать в качестве
    // Label-а, с небольшой заменой полей
    if (!Service.LabelCodec().HasPrefix(LabelHash)) {
        return;
    }
    NTravelProto::TLabel prevLabel;
    LabelHashOverriden = false;
    try {
        Service.LabelCodec().DecodeToMessage(LabelHash, &prevLabel, NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(true));
    } catch (...) {
        ERROR_LOG << JOB_LOG << "Failed to decode LabelHash as ProtoLabel: '" << LabelHash << "', cause: " << CurrentExceptionMessage() << Endl;
        LabelHash.clear();
        return;
    }
    // Из новой аттрибуции берем портальные TestId/Buckets, см. TRAVELBACK-1402
    prevLabel.MutableIntPortalTestIds()->CopyFrom(Label.GetIntPortalTestIds());
    prevLabel.MutableIntPortalTestBuckets()->CopyFrom(Label.GetIntPortalTestBuckets());
    // Флаги стаффовости и плюсовости пользователя тоже берём из новой (портальной) аттрибуции, потому что СЕРП про это нам не сообщает
    prevLabel.SetIsStaffUser(Label.GetIsStaffUser());
    prevLabel.SetIsPlusUser(Label.GetIsPlusUser());
    // Метричные поля тоже берём новые, иначе не склеятся брони в метрике
    prevLabel.SetMetrikaClientId(Label.GetMetrikaClientId());
    prevLabel.SetMetrikaUserId(Label.GetMetrikaUserId());
    Label.Swap(&prevLabel);
    LabelHash.clear();
}

void TJob::ProcessByToken() {
    DoRedirByToken(Token, Req.GetToken());
}

void TJob::DoRedirByToken(const NTravelProto::TTravelToken& token, const TString& rawToken) {
    SetDefaultPortalSchemeAndHost();

    const auto& rule = Service.GetBoYRule(Label.GetSurface());

    NTravelProto::ERedirDestination destination;
    if (Req.HasForceDest()) {
        destination = Req.GetForceDest();
    } else {
        destination = rule.GetDestination();
    }
    if (Service.Config().GetOther().GetEnableDebug() && token.GetPermalink() < 1000) {
        destination = NTravelProto::RD_PortalBookPage;
    }
    if (destination <= NTravelProto::ERedirDestination_MIN || destination > NTravelProto::ERedirDestination_MAX) {
        destination = NTravelProto::RD_PortalBookPage;
    }

    OpCounters->NDestination[destination].Inc();
    Label.SetRedirDestination(destination);

#define PASS_UTM(_X_)                                 \
    if (Req.has_utm_##_X_()) {                        \
        Url.SetCgiParam("utm_"#_X_, Req.utm_##_X_()); \
    }
    PASS_UTM(source);
    PASS_UTM(medium);
    PASS_UTM(campaign);
    PASS_UTM(content);
    PASS_UTM(term);

#undef OVERRIDE_UTM

    switch (destination) {
        case NTravelProto::RD_PortalHotelPage:
        case NTravelProto::RD_PortalHotelPageRooms: {
            Url.SetPath("/hotels/hotel/");
            Url.SetCgiParam("hotelPermalink", ToString(token.GetPermalink()));

            NOrdinalDate::TOrdinalDate epoch = NOrdinalDate::FromString("2019-01-01");
            Url.SetCgiParam("checkinDate", NOrdinalDate::ToString(epoch + token.GetCheckInDateDaysSinceEpoch()));
            Url.SetCgiParam("checkoutDate", NOrdinalDate::ToString(epoch + token.GetCheckOutDateDaysSinceEpoch()));

            TAges ages = TAges::FromOccupancyString(token.GetOccupancy());
            Url.SetCgiParam("adults", ToString(ages.GetAdultCount()));
            Url.SetCgiParam("childrenAges", JoinVectorIntoString(ages.GetChildrenAges(), TStringBuf(",")));

            if (destination == NTravelProto::RD_PortalHotelPageRooms) {
                Url.SetCgiParam("activeTabId", "offers");
            }
            break;
        }
        case NTravelProto::RD_Partner:
        case NTravelProto::RD_PortalBookPage:
            Url.SetPath("/hotels/book/");
            break;
        default: // I Hate proto3, it makes me cry
            Url.SetPath("/hotels/book/");
            break;

    }
    TryParsePreviousLabelFromLabelHash();
    CalcLabelHash();
    PutLabelToUrl(rule.GetLabelAsHash() || LabelHashOverriden);
    Url.SetCgiParam("token", rawToken);
    Reply();

    WritePriceCheckerRequest();
    if (rule.GetLog()) {
        WriteReqAns();
    }
}

void TJob::SetDefaultPortalSchemeAndHost() {
    Url.SetScheme(NUri::TScheme::SchemeHTTPS);
    Url.SetHost(GetAllowedHostOrDefault(Req.GetDebugPortalHost(), Service.Config().GetOther().GetAllowedPortalHost(), Service.Config().GetOther().GetDefaultPortalHost()));
}

void TJob::DoFallbackRedirByLabel() {
    SetDefaultPortalSchemeAndHost();
    Url.SetPath("/hotels/hotel/");
    Url.SetCgiParam("hotelPermalink", ToString(Label.GetPermalink()));

    Url.SetCgiParam("checkinDate", Label.GetCheckInDate());
    Url.SetCgiParam("checkoutDate", NOrdinalDate::PlusDays(Label.GetCheckInDate(), Label.GetNights()));

    TAges ages = TAges::FromOccupancyString(Label.GetOccupancy());
    Url.SetCgiParam("adults", ToString(ages.GetAdultCount()));
    Url.SetCgiParam("childrenAges", JoinVectorIntoString(ages.GetChildrenAges(), TStringBuf(",")));

    OpCounters->NDestination[NTravelProto::RD_PortalFallback].Inc();
    Reply();
}

void TJob::ProcessByOfferId() {
    auto cacheAgeSec = Started.Seconds() - Label.GetCacheTimestamp();
    if (cacheAgeSec > Service.Config().GetOther().GetMaxCacheAgeSecForRequestByOfferId()) {
        OpCounters->NReqByOfferIdTooOld.Inc();
        INFO_LOG << JOB_LOG << "Too old req by offer id (" << Req.GetOfferId() << "), age is " << cacheAgeSec << " sec" << Endl;
        DoFallbackRedirByLabel();
        return;
    }

    bool validOfferIdHash = false;
    if (!Req.GetOfferIdHash().empty()) {
        TString OfferIdWithSalt = Service.Config().GetOther().GetOfferIdHashSalt() + Req.GetOfferId();
        auto currentOfferIdHash = ToString(Crc32c(OfferIdWithSalt.data(), OfferIdWithSalt.length())); // It's not for security, so it's ok to use crc32
        validOfferIdHash = (currentOfferIdHash == Req.GetOfferIdHash());
    }

    if (!validOfferIdHash) {
        OpCounters->NReqByOfferIdWithWrongHash.Inc();
    }

    TMaybe<NTravelProto::NSearchFlowOfferData::TOfferDataMessage> offerData;
    try {
        offerData = SearchFlowOfferDataStorage.Read<NTravelProto::NSearchFlowOfferData::TOfferDataMessage>(Req.GetOfferId());
    } catch (...) {
        if (validOfferIdHash) {
            OpCounters->NFailedToReadOfferData.Inc();
            ERROR_LOG << JOB_LOG << "Failed to get offer data by id (" << Req.GetOfferId() << ") " << CurrentExceptionMessage() << Endl;
        }
        DoFallbackRedirByLabel();
        return;
    }

    if (!offerData.Defined()) {
        if (validOfferIdHash) {
            OpCounters->NOfferDataNotFound.Inc();
            ERROR_LOG << JOB_LOG << "Offer not found by id " << Req.GetOfferId() << Endl;
        }
        DoFallbackRedirByLabel();
        return;
    }

    if (!validOfferIdHash) {
        WARNING_LOG << JOB_LOG << "Successful req by OfferId with wrong hash. OfferId: " << Req.GetOfferId() << " OfferIdHash: " << Req.GetOfferIdHash() << Endl;
        OpCounters->NSuccessfulReqByOfferIdWithWrongHash.Inc();
    }

    TString aid = Exp.StrExpVal("BookingAid");

    if (Label.GetPartnerId() == NTravelProto::PI_BOOKING && aid) {
        try {
            if (aid == "2192269") {
                Url.Parse(offerData->GetLandingInfo().GetBookingSearchPageLandingUrl());
            } else if (aid == "2192270") {
                Url.Parse(offerData->GetLandingInfo().GetBookingHotelPageLandingUrl());
            }
        } catch (...) {
            OpCounters->NOfferDataUrlDecodeErrors.Inc();
            throw yexception() << "Failed to parse url: " << CurrentExceptionMessage();
        }
        DoRedirByPUrl();
    } else if (offerData->GetLandingInfo().HasLandingPageUrl()) {
        try {
            Url.Parse(offerData->GetLandingInfo().GetLandingPageUrl());
        } catch (...) {
            OpCounters->NOfferDataUrlDecodeErrors.Inc();
            throw yexception() << "Failed to parse url: " << CurrentExceptionMessage();
        }
        DoRedirByPUrl();
    } else if (offerData->GetLandingInfo().HasLandingTravelToken()) {
        NTravelProto::TTravelToken token;
        try {
            if (!token.ParseFromString(Service.TokenCodec().Decode(offerData->GetLandingInfo().GetLandingTravelToken()))) {
                throw yexception() << "Failed to parse protobuf in token";
            }
        } catch (...) {
            OpCounters->NOfferDataTokenDecodeErrors.Inc();
            throw yexception() << "Failed to parse token: " << CurrentExceptionMessage();
        }
        DoRedirByToken(token, offerData->GetLandingInfo().GetLandingTravelToken());
    } else {
        OpCounters->NOfferDataWithoutRedirInfo.Inc();
        ERROR_LOG << JOB_LOG << "Offer with id " << Req.GetOfferId() << " has not redir info" << Endl;
        DoFallbackRedirByLabel();
    }
}

void TJob::CalcLabelHash() {
    if (!LabelHashOverriden) {
        LabelHash = NTravel::NLabel::CalcLabelHash(Label);
    }
}

void TJob::PutLabelToUrl(bool asHash) {
    TString labelParamName;
    auto partner = Service.GetPartnerConfig(Label.GetPartnerId());
    if (!partner) {
        ERROR_LOG << JOB_LOG << "Unknown partner id " << Label.GetPartnerId() << " in label" << Endl;
        labelParamName = "label";
    } else {
        labelParamName = partner->GetLabelParameter();
    }

    UrlWithoutLabel = Url.ToString();
    if (asHash) {
        Url.SetCgiParam(labelParamName, LabelHash);
    } else {
        Url.SetCgiParam(labelParamName, Service.LabelCodec().Encode(Label, NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(true)));
    }
}

void TJob::Reply() const {
    NHttp::TResponse resp;
    resp.LogPrefix = LogPrefix;
    if (Req.GetRedir()) {
        resp.SetHttpCode(HTTP_FOUND);// 302
        resp.AddHeader("Location", Url.ToString());
        OpCounters->NResponsesOK.Inc();
    } else {
        resp.SetHttpCode(HTTP_OK);
        resp.SetContentType("text/plain");
        resp.SetContent(Url.ToString());
        OpCounters->NResponsesNoRedir.Inc();
    }
    ResponseCb(resp);
    INFO_LOG << JOB_LOG << "Redirect to " << Url.ToString() << " (redir=" << Req.GetRedir() << ")" << Endl;
}

void TJob::WriteReqAns() const {
    if (LabelHashOverriden) {
        // TODO - logging is wrong when LabelHash is set; improve logging!
        return;
    }
    if (!Req.GetRedir()) {
        // No redir -> No logging!
        return;
    }
    NTravelProto::NRedir::NHotels::TReqAnsLogRecord logRecord;
    logRecord.set_unixtime(Started.Seconds());
    logRecord.SetTargetUrl(UrlWithoutLabel);
    logRecord.SetLabel(LabelHash);
    logRecord.SetProto(Base64EncodeUrlShort(Label.SerializeAsString()));
    PutFieldsMapToLog(Label, logRecord.MutableFieldsMap());
    logRecord.MutableAddInfo()->CopyFrom(AddInfo);
    Service.WriteReqAnsHotels(logRecord);
}

void TJob::WritePriceCheckerRequest() const {
    if (!Req.GetRedir() || !Label.GetOfferId()) {
        return;
    }
    ru::yandex::travel::hotels::TPriceCheckReq priceCheckReq;
    priceCheckReq.SetOfferId(Label.GetOfferId());
    google::protobuf::Timestamp cacheTimestamp;
    cacheTimestamp.set_seconds(Label.GetCacheTimestamp());
    *priceCheckReq.MutableCacheTimestamp() = cacheTimestamp;
    Service.WritePriceCheckerRequest(priceCheckReq);
}


} // namespace NHotels
} // namespace NRedir
} // namespace NTravel
