#include "service.h"

#include "hotels_job.h"
#include "trains_job.h"
#include "buses_job.h"
#include "suburban_job.h"
#include "tours_job.h"
#include "tools.h"

#include <travel/hotels/redir/proto/api.pb.h>

#include <travel/hotels/lib/cpp/util/base64.h>
#include <travel/hotels/lib/cpp/scheduler/scheduler.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/http/client/client.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/protobuf/json/proto2json.h>

#include <util/datetime/base.h>
#include <util/generic/hash.h>
#include <util/generic/serialized_enum.h>
#include <util/system/hostname.h>

#include <openssl/sha.h>

namespace NTravel {
namespace NRedir {


TService::TService(const NTravelProto::NRedir::TConfig& pbCfg)
    : Config_(pbCfg)
    , YtConfigPartners_("YtConfigPartners", pbCfg.GetYtConfigPartners())
    , ServiceCounters_(*this)
    , HotelsPerRequestCounters_({"operator"})
    , TrainsPerRequestCounters_({})
    , BusesPerRequestCounters_({})
    , SuburbanPerRequestCounters_({})
    , ToursPerRequestCounters_({})
    , UrlCodec_(NEncryption::TUrlCodec(Config_.GetOther().GetLabelKeyPath()))
    , TokenCodec_(NEncryption::TTokenCodec(Config_.GetOther().GetTravelTokenKeyPath()))
    , AddInfoCodec_(NEncryption::TTokenCodec(Config_.GetOther().GetAddInfoKeyPath()))
    , ReqAnsHotelsLogger_("ReqAnsHotelsLogger", Config_.GetReqAnsHotelsLogger())
    , ReqAnsTrainsLogger_("ReqAnsTrainsLogger", Config_.GetReqAnsTrainsLogger())
    , ReqAnsBusesLogger_("ReqAnsBusesLogger", Config_.GetReqAnsBusesLogger())
    , ReqAnsSuburbanLogger_("ReqAnsSuburbanLogger", Config_.GetReqAnsSuburbanLogger())
    , ReqAnsToursLogger_("ReqAnsToursLogger", Config_.GetReqAnsToursLogger())
    , PriceCheckReqBus_(Config_.GetPriceCheckReqBus(), "PriceCheckReqBus")
    , SearchFlowOfferDataStorage_(Config_.GetSearchFlowOfferDataStorage(), "SearchFlowOfferDataStorage")
    , PingEnabled_(true)
    , LastPing_(Now())
    , RestartDetector_(Config_.GetOther().GetRestartDetectorStateFile())
    , LastJobId_(0)
    , ActiveJobCount_(0)
{
    Http_.AddHandler("/redir",           NHttp::ExternalWithoutTvm(), this, &TService::OnHotelsRedir);
    Http_.AddHandler("/trains/label_to_hash", NHttp::ExternalWithoutTvm(), this, &TService::OnTrainsLabelToHash);
    Http_.AddHandler("/buses/label_to_hash", NHttp::ExternalWithoutTvm(), this, &TService::OnBusesLabelToHash);
    Http_.AddHandler("/suburban/label_to_hash", NHttp::ExternalWithoutTvm(), this, &TService::OnSuburbanLabelToHash);
    Http_.AddHandler("/tours/label_to_hash", NHttp::ExternalWithoutTvm(), this, &TService::OnToursLabelToHash);
    Http_.AddHandler("/fail",            NHttp::ExternalWithoutTvm(), this, &TService::OnFail);
    Http_.AddHandler("/ping",            NHttp::ExternalWithoutTvm(), this, &TService::OnPing);
    Http_.AddHandler("/ping_nanny",      NHttp::ExternalWithoutTvm(), this, &TService::OnPingNanny);
    Http_.AddHandler("/ping/disable",    NHttp::Local(),    this, &TService::OnPingDisable);
    Http_.AddHandler("/setlog",          NHttp::Local(),    this, &TService::OnSetLogLevel);
    Http_.AddHandler("/reopen-ra-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsHotelsLog);// TODO remove me
    Http_.AddHandler("/reopen-ra-hotels-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsHotelsLog);
    Http_.AddHandler("/reopen-ra-trains-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsTrainsLog);
    Http_.AddHandler("/reopen-ra-suburban-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsSuburbanLog);
    Http_.AddHandler("/reopen-ra-tours-log",   NHttp::Local(),    this, &TService::OnReopenReqAnsToursLog);
    Http_.AddHandler("/reopen-main-log", NHttp::Local(),    this, &TService::OnReopenMainLog);
    Http_.AddHandler("/shutdown",        NHttp::Local(),    this, &TService::OnShutdown);
    // For autotests only
    Http_.AddHandler("/wait-flush",      NHttp::Local(),    this, &TService::OnWaitFlush);

    CountersPage_.RegisterSource(&ServiceCounters_, "Server");
    CountersPage_.RegisterSource(&HotelsPerRequestCounters_, "PerReq");
    CountersPage_.RegisterSource(&TrainsPerRequestCounters_, "TrainsPerReq");
    CountersPage_.RegisterSource(&SuburbanPerRequestCounters_, "SuburbanPerReq");
    CountersPage_.RegisterSource(&ToursPerRequestCounters_, "ToursPerReq");
    Http_.RegisterCounters(CountersPage_);
    PriceCheckReqBus_.RegisterCounters(CountersPage_);
    RestartDetector_.RegisterCounters(CountersPage_);
    TScheduler::Instance().RegisterCounters(CountersPage_);
    YtConfigPartners_.RegisterCounters(CountersPage_);
    ReqAnsHotelsLogger_.RegisterCounters(CountersPage_);
    ReqAnsTrainsLogger_.RegisterCounters(CountersPage_);
    ReqAnsBusesLogger_.RegisterCounters(CountersPage_);
    ReqAnsSuburbanLogger_.RegisterCounters(CountersPage_);
    ReqAnsToursLogger_.RegisterCounters(CountersPage_);
    SearchFlowOfferDataStorage_.RegisterCounters(CountersPage_);

    CountersPage_.AddToHttpService(MonWebService_);
    for (ui32 opId = NTravelProto::EOperatorId_MIN; opId < NTravelProto::EOperatorId_MAX; ++opId) {
        if (opId != NTravelProto::OI_UNUSED) {
            HotelsPerRequestCounters((EOperatorId) opId);
        }
    }
    for (const auto& rule: pbCfg.GetBoYRules().GetRule()) {
        BoYRules_[rule.GetSurface()] = rule;
    }
}

void TService::Run() {
    TScheduler::Instance().Start();
    YtConfigPartners_.Start();
    ReqAnsHotelsLogger_.Start();
    ReqAnsTrainsLogger_.Start();
    ReqAnsBusesLogger_.Start();
    ReqAnsSuburbanLogger_.Start();
    ReqAnsToursLogger_.Start();
    PriceCheckReqBus_.Start();
    SearchFlowOfferDataStorage_.Start();
    Http_.Start(Config_.GetHttp());
    MonWebService_.Start(Config_.GetOther().GetMonitoringPort());

    INFO_LOG << "Listening on " << Config_.GetHttp().GetPort() << Endl;
    StopEvent_.WaitI();
    INFO_LOG << "Got stop signal" << Endl;
}

void TService::Stop() {
    RestartDetector_.ReportShutdown();
    TScheduler::Instance().Stop();
    MonWebService_.Stop();
    Http_.Stop();
    SearchFlowOfferDataStorage_.Stop();
    PriceCheckReqBus_.Stop();
    ReqAnsToursLogger_.Stop();
    ReqAnsSuburbanLogger_.Stop();
    ReqAnsTrainsLogger_.Stop();
    ReqAnsBusesLogger_.Stop();
    ReqAnsHotelsLogger_.Stop();
    YtConfigPartners_.Stop();
    StopEvent_.Signal();
}

TService::~TService() {
}

bool TService::IsReady() const {
    return PingEnabled_ && YtConfigPartners_.IsReady() && SearchFlowOfferDataStorage_.IsReady();
}

bool TService::IsPinged() const {
    with_lock (LastPingLock_) {
        return (Now() - LastPing_) < TDuration::Seconds(30);
    }
}

bool TService::IsPingEnabled() const {
    return PingEnabled_;
}

void TService::OnPing(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    with_lock (LastPingLock_) {
        LastPing_ = Now();
    }
    SeenBalancerPing_.Set();
    responseCb(THttpResponse(IsReady() ? HTTP_OK : HTTP_BAD_GATEWAY));
}

void TService::OnPingNanny(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    // Если нам не приходил пинг от балансера, не отвечаем няне, чтобы она дождалась, пока балансер узнает про нас TRAVELBACK-132
    bool isReady = IsReady() && SeenBalancerPing_;
    responseCb(THttpResponse(isReady ? HTTP_OK : HTTP_BAD_GATEWAY));
}

void TService::OnPingDisable(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    PingEnabled_.Clear();
    INFO_LOG << "Ping disabled" << Endl;
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnSetLogLevel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    if (auto v = httpReq.Query().Get("level")) {
        auto p = FromString<ELogPriority>(v);
        DoInitGlobalLog(Config_.GetOther().GetMainLogFile(), p, false, false);
        responseCb(THttpResponse(HTTP_OK));
    } else {
        responseCb(THttpResponse(HTTP_BAD_REQUEST));
    }
}

void TService::OnReopenReqAnsHotelsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns-hotels log" << Endl;
    ReqAnsHotelsLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenReqAnsTrainsLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns-trains log" << Endl;
    ReqAnsTrainsLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenReqAnsBusesLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns-buses log" << Endl;
    ReqAnsBusesLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenReqAnsSuburbanLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns-suburban log" << Endl;
    ReqAnsSuburbanLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenReqAnsToursLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening ReqAns-tours log" << Endl;
    ReqAnsToursLogger_.Reopen();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnReopenMainLog(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    INFO_LOG << "Reopening main log" << Endl;
    TLoggerOperator<TGlobalLog>::Log().ReopenLog();
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnWaitFlush(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    while (true) {
        INFO_LOG << "Waiting for data to be flushed" << Endl;
        if (AtomicGet(ActiveJobCount_) == 0 && ReqAnsHotelsLogger_.IsFlushed() && ReqAnsTrainsLogger_.IsFlushed() && PriceCheckReqBus_.IsFlushed()
            && ReqAnsToursLogger_.IsFlushed()) {
            break;
        }
        Sleep(TDuration::MilliSeconds(10));
    }
    responseCb(THttpResponse(HTTP_OK));
}

void TService::OnShutdown(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    Y_UNUSED(httpReq);
    if (IsShuttingDown_) {
        responseCb(NHttp::TResponse::CreateText("Already shutting down\n"));
        return;
    }
    IsShuttingDown_.Set();
    INFO_LOG << "Shutting down!" << Endl;
    SystemThreadFactory()->Run([this](){
        TDuration dur = TDuration::Seconds(12);
        PingEnabled_.Clear();
        INFO_LOG << "Ping disabled, wait " << dur << Endl;
        StopEvent_.WaitT(dur);// Даём время, чтобы балансер увидел, что сюда ходить не надо

        Http_.Shutdown();
        dur = TDuration::Seconds(7);
        INFO_LOG << "Http Shutdown called,  wait " << dur << Endl;
        StopEvent_.WaitT(dur);// Даём время завершить текущие соединения

        Stop(); // Завершаемся, закрывая все соединения
    });
    responseCb(NHttp::TResponse::CreateText("Shutdown procedure started\n"));
}

const NTravelProto::NRedir::TConfig& TService::Config() const {
    return Config_;
}

TServiceCounters& TService::ServiceCounters() {
    return ServiceCounters_;
}

NHotels::TPerRequestCountersRef  TService::HotelsPerRequestCounters(EOperatorId opId) {
    return HotelsPerRequestCounters_.GetOrCreate({NTravelProto::EOperatorId_Name(opId)});
}

NTrains::TPerRequestCountersRef  TService::TrainsPerRequestCounters() {
    return TrainsPerRequestCounters_.GetOrCreate({});
}

NBuses::TPerRequestCountersRef TService::BusesPerRequestCounters() {
    return BusesPerRequestCounters_.GetOrCreate({});
}

NSuburban::TPerRequestCountersRef  TService::SuburbanPerRequestCounters() {
    return SuburbanPerRequestCounters_.GetOrCreate({});
}

NTours::TPerRequestCountersRef  TService::ToursPerRequestCounters() {
    return ToursPerRequestCounters_.GetOrCreate({});
}

NEncryption::TUrlCodec& TService::UrlCodec() {
    return UrlCodec_;
}

NEncryption::TTokenCodec& TService::TokenCodec() {
    return TokenCodec_;
}

NEncryption::TTokenCodec& TService::AddInfoCodec() {
    return AddInfoCodec_;
}

NLabel::TLabelCodec& TService::LabelCodec() {
    return LabelCodec_;
}

std::shared_ptr<const NTravelProto::NConfig::TPartner> TService::GetPartnerConfig(EPartnerId pId) {
    return YtConfigPartners_.GetById(pId);
}

const NTravelProto::NRedir::TConfig::TBoYRules::TRule& TService::GetBoYRule(NTravelProto::ESurface surface) const {
    auto it = BoYRules_.find(surface);
    if (it != BoYRules_.end()) {
        return it->second;
    }
    return Config_.GetBoYRules().GetDefaultRule();
}

void TService::WritePriceCheckerRequest(const ru::yandex::travel::hotels::TPriceCheckReq& req) {
    PriceCheckReqBus_.Write(req);
}

void TService::WriteReqAnsHotels(const NTravelProto::NRedir::NHotels::TReqAnsLogRecord& record) {
    ReqAnsHotelsLogger_.AddRecord(record);
}

void TService::WriteReqAnsTrains(const NTravelProto::NRedir::NTrains::TReqAnsLogRecord& record) {
    ReqAnsTrainsLogger_.AddRecord(record);
}

void TService::WriteReqAnsBuses(const NTravelProto::NRedir::NBuses::TReqAnsLogRecord& record) {
    ReqAnsBusesLogger_.AddRecord(record);
}

void TService::WriteReqAnsSuburban(const NTravelProto::NRedir::NSuburban::TReqAnsLogRecord& record) {
    ReqAnsSuburbanLogger_.AddRecord(record);
}

void TService::WriteReqAnsTours(const NTravelProto::NRedir::NTours::TReqAnsLogRecord& record) {
    ReqAnsToursLogger_.AddRecord(record);
}

void TService::OnHotelsRedir(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    AtomicIncrement(ActiveJobCount_);
    {
        NHotels::TJob job(*this, SearchFlowOfferDataStorage_, responseCb, AtomicAdd(LastJobId_, 1));
        job.Run(httpReq);
        // Job is fully synchronious now
    }
    AtomicDecrement(ActiveJobCount_);
}

void TService::OnTrainsLabelToHash(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    AtomicIncrement(ActiveJobCount_);
    {
        NTrains::TJob job(*this, responseCb, AtomicAdd(LastJobId_, 1));
        job.Run(httpReq);
        // Job is fully synchronious now
    }
    AtomicDecrement(ActiveJobCount_);
}

void TService::OnBusesLabelToHash(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    AtomicIncrement(ActiveJobCount_);
    {
        NBuses::TJob job(*this, responseCb, AtomicAdd(LastJobId_, 1));
        job.Run(httpReq);
        // Job is fully synchronious now
    }
    AtomicDecrement(ActiveJobCount_);
}

void TService::OnSuburbanLabelToHash(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    AtomicIncrement(ActiveJobCount_);
    {
        NSuburban::TJob job(*this, responseCb, AtomicAdd(LastJobId_, 1));
        job.Run(httpReq);
        // Job is fully synchronious now
    }
    AtomicDecrement(ActiveJobCount_);
}

void TService::OnToursLabelToHash(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    AtomicIncrement(ActiveJobCount_);
    {
        NTours::TJob job(*this, responseCb, AtomicAdd(LastJobId_, 1));
        job.Run(httpReq);
        // Job is fully synchronious now
    }
    AtomicDecrement(ActiveJobCount_);
}

void TService::OnFail(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
    TString text = httpReq.Query().Get("Text");
    int code = HTTP_INTERNAL_SERVER_ERROR;
    auto it = httpReq.Query().find("Code");
    if (it != httpReq.Query().end()) {
        TryFromString(it->second, code);
    }
    responseCb(NHttp::TResponse::CreateText(text, (HttpCodes)code));
}

} // namespace NRedir
} // namespace NTravel
