#include "tours_job.h"

#include "service.h"
#include "tools.h"

#include <travel/hotels/redir/proto/reqans_tours.pb.h>

#include <travel/hotels/lib/cpp/util/base64.h>
#include <travel/hotels/lib/cpp/encryption/codec.h>
#include <travel/hotels/lib/cpp/label/label.h>

#define JOB_LOG LogPrefix

namespace NTravel::NRedir::NTours {

    TJob::TJob(TService& svc, NHttp::TOnResponse onResp, ui64 id)
        : Service(svc)
        , ResponseCb(onResp)
        , LogPrefix("Tours_Id_" + ToString(id) + ": ")
        , Started(Now())
        , Counters(svc.ToursPerRequestCounters())
    {
    }

    TJob::~TJob() {
    }

    void TJob::Run(const NHttp::TRequest& httpReq) {
        INFO_LOG << JOB_LOG << "Start job, query: " << httpReq.Cgi() << Endl;
        try {
            NTravel::NProtobuf::ParseCgiRequest(httpReq.Query(), &Req);
            Parse();
        } catch (...) {
            ERROR_LOG << JOB_LOG << "Bad request: " << CurrentExceptionMessage() << Endl;
            Counters->NRequestErrors.Inc();
            NHttp::TResponse resp = NHttp::TResponse::CreateText(CurrentExceptionMessage() + "\n", HTTP_BAD_REQUEST);
            resp.LogPrefix = LogPrefix;
            ResponseCb(resp);
            return;
        }
        Counters->NRequests.Inc();
        Process();
    }

    void TJob::Parse() {
        if (Req.HasLabelParams()) {
            try {
                TString labelOrigProtoBytes = NEncryption::TBase64Codec().Decode(Req.GetLabelParams());
                if (!LabelParams.ParseFromString(labelOrigProtoBytes)) {
                    throw yexception() << "Failed to parse protobuf in label";
                }
            } catch (...) {
                Counters->NLabelDecodeErrors.Inc();
                throw yexception() << "Failed to decode LabelParams '" << Req.GetLabelParams() << "', cause " << CurrentExceptionMessage();
            }
        } else {
            Counters->NRequestsNoLabel.Inc();
            throw yexception() << "No LabelParams in request";
        }
    }

    void TJob::Process() {
        LabelHash = NTravel::NLabel::CalcLabelHash(LabelParams);
        Reply();
        WriteReqAns();
    }

    void TJob::Reply() const {
        NHttp::TResponse resp = NHttp::TResponse::CreateText(LabelHash);
        resp.LogPrefix = LogPrefix;
        resp.EnableKeepAlive = Service.IsPingEnabled();
        ResponseCb(resp);
        INFO_LOG << JOB_LOG << "Finished, LabelHash = " << LabelHash << Endl;
    }

    void TJob::WriteReqAns() const {
        NTravelProto::NRedir::NTours::TReqAnsLogRecord logRecord;
        logRecord.set_unixtime(Started.Seconds());
        logRecord.SetLabelHash(LabelHash);
        logRecord.SetProto(Base64EncodeUrlShort(LabelParams.SerializeAsString()));
        PutFieldsMapToLog(LabelParams, logRecord.MutableFieldsMap());
        Service.WriteReqAnsTours(logRecord);
    }

}
