package ru.yandex.travel.hotels.searcher;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.travel.hotels.common.token.Occupancy;

public class Capacity {
    private boolean exactMatch;
    private int maxAdults;
    private List<Integer> maxChildrenAge;

    private Capacity(boolean exactMatch, int maxAdults, List<Integer> maxChildrenAge) {
        this.exactMatch = exactMatch;
        this.maxAdults = maxAdults;
        this.maxChildrenAge = maxChildrenAge;
    }

    @Override
    public String toString() {
        StringBuilder bldr = new StringBuilder();
        if (exactMatch) {
            bldr.append("==");
        } else {
            bldr.append("<=");
        }
        bldr.append(maxAdults);
        if (maxChildrenAge.size() > 0) {
            bldr.append('-');
            bldr.append(maxChildrenAge.stream().map(Object::toString).collect(Collectors.joining(",")));
        }
        return bldr.toString();
    }

    public static Capacity fromRule(int maxAdults, int childrenCount, int maxChildAgeInYears) {
        return new Capacity(false, maxAdults, Collections.nCopies(childrenCount, maxChildAgeInYears));
    }

    public static Capacity fromOccupancy(Occupancy occupancy) {
        return new Capacity(true, occupancy.getAdults(), occupancy.getChildren());
    }

    public static Capacity fromOccupancyAndAdults(Occupancy occupancy, int maxAdults) {
        return new Capacity(false, maxAdults, occupancy.getChildren());
    }

    public Capacity multiply(int times) {
        List<Integer> newMaxChildrenAge = new ArrayList<>();
        for (int i = 0; i < times; ++i) {
            newMaxChildrenAge.addAll(maxChildrenAge);
        }
        return new Capacity(exactMatch, maxAdults * times, newMaxChildrenAge);
    }

    public boolean matches(Occupancy occupancy) {
        if (exactMatch) {
            if (this.maxAdults != occupancy.getAdults()) {
                return false;
            }
            if (this.maxChildrenAge.size() != occupancy.getChildren().size()) {
                return false;
            }
            return this.maxChildrenAge.equals(occupancy.getChildren());
        } else {
            if (this.maxAdults < occupancy.getAdults()) {
                return false;
            }
            int maxChildren = maxChildrenAge.size();
            int maxAge = maxChildren > 0 ? maxChildrenAge.get(0) : 0;
            int numExtraChildren = occupancy.getChildren().size() - (int) occupancy.getChildren().stream().filter(age -> age <= maxAge).limit(maxChildren).count();
            return numExtraChildren + occupancy.getAdults() <= this.maxAdults;
        }
    }

    public int calculateRoomCount(Occupancy occupancy) {
        if (this.matches(occupancy)) {
            return 1;
        }
        if (maxAdults == 0 || occupancy.getAdults() == 0) {
            return 0;
        }
        // Current calculator only takes adults into account
        int roomCount = (occupancy.getAdults() + maxAdults - 1) / maxAdults;
        Capacity capacityOfMultipleRooms = multiply(roomCount);
        if (capacityOfMultipleRooms.matches(occupancy)) {
            return roomCount;
        }
        return 0;
    }
}


