package ru.yandex.travel.hotels.searcher;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;

import com.google.common.collect.ImmutableList;
import io.grpc.stub.StreamObserver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.actuate.health.HealthEndpoint;
import org.springframework.boot.actuate.health.Status;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
import ru.yandex.travel.hotels.proto.OfferSearchServiceV1Grpc;
import ru.yandex.travel.hotels.proto.TPingRpcRsp;
import ru.yandex.travel.hotels.proto.TPlaceholder;
import ru.yandex.travel.hotels.proto.TSearchOffersReq;
import ru.yandex.travel.hotels.proto.TSearchOffersRpcReq;
import ru.yandex.travel.hotels.proto.TSearchOffersRpcRsp;
import ru.yandex.travel.hotels.proto.TSearchOffersRsp;

@GrpcService
public class DefaultOfferSearchService extends OfferSearchServiceV1Grpc.OfferSearchServiceV1ImplBase {
    private static final Logger logger = LoggerFactory.getLogger(DefaultOfferSearchService.class);

    private static final Pattern DATE_PATTERN = Pattern.compile("\\d\\d\\d\\d-\\d\\d-\\d\\d");
    private static final Pattern OCCUPANCY_PATTERN = Pattern.compile("\\d{1,2}(?:-\\d{1,2}(?:,\\d{1,2})*)?");

    private final PartnerDispatcher partnerDispatcher;
    private final HealthEndpoint healthEndpoint;

    public DefaultOfferSearchService(PartnerDispatcher partnerDispatcher, HealthEndpoint healthEndpoint) {
        this.partnerDispatcher = partnerDispatcher;
        this.healthEndpoint = healthEndpoint;
    }

    @Override
    public void ping(ru.yandex.travel.hotels.proto.TPingRpcReq request,
                     io.grpc.stub.StreamObserver<ru.yandex.travel.hotels.proto.TPingRpcRsp> responseObserver) {
        TPingRpcRsp.Builder builder = TPingRpcRsp.newBuilder();
        builder.setIsReady(healthEndpoint.health().getStatus() == Status.UP);
        responseObserver.onNext(builder.build());
        responseObserver.onCompleted();
    }

    @Override
    public void searchOffers(TSearchOffersRpcReq request, StreamObserver<TSearchOffersRpcRsp> responseObserver) {
        boolean includeDebug = request.getIncludeDebug();
        if (!request.getSync() && request.hasTestContext()) {
            responseObserver.onError(
                    new IllegalArgumentException("Test Context may only be passed with Sync requests"));
            responseObserver.onCompleted();
            return;
        }
        TSearchOffersRpcRsp.Builder builder = TSearchOffersRpcRsp.newBuilder();
        List<CompletableFuture<Void>> futures = new ArrayList<>(request.getSubrequestCount());
        Map<EPartnerId, List<Task>> tasksByPartner = new HashMap<>();
        long createdAtNanos = System.nanoTime();
        for (int index_ = 0; index_ < request.getSubrequestCount(); ++index_) {
            int index = index_;
            TSearchOffersReq subrequest = request.getSubrequest(index);

            // Validate subrequest.
            ImmutableList<TError> errors = validateRequest(subrequest);
            if (!errors.isEmpty()) {
                TError.Builder errorBuilder = TError.newBuilder()
                        .setCode(EErrorCode.EC_INVALID_ARGUMENT)
                        .setMessage("Invalid request");
                for (TError error : errors) {
                    errorBuilder.addNestedError(error);
                }
                builder.addSubresponse(index, TSearchOffersRsp.newBuilder().setError(errorBuilder));
                continue;
            }

            // Create task and subscribe to its completion.
            Task task = new Task(subrequest, includeDebug, createdAtNanos, CallContext.forSearcher(request, subrequest));
            tasksByPartner.computeIfAbsent(subrequest.getHotelId().getPartnerId(), k -> new ArrayList<>()).add(task);

            CompletableFuture<TSearchOffersRsp> responseFuture;
            if (request.getSync()) {
                responseFuture = task.getCompletionFuture().handle((ignored, throwable) ->
                        {
                            if (throwable != null) {
                                return TSearchOffersRsp.newBuilder()
                                        .setError(ProtoUtils.errorFromThrowable(throwable, includeDebug))
                                        .build();
                            } else {
                                return task.dumpResult();
                            }
                        }
                );
            } else {
                responseFuture = CompletableFuture.completedFuture(TSearchOffersRsp.newBuilder()
                        .setPlaceholder(TPlaceholder.newBuilder())
                        .build());
            }
            CompletableFuture<Void> taskReadyFuture = responseFuture.thenAccept(rsp -> {
                synchronized (builder) {
                    builder.addSubresponse(index, rsp);
                }
            });
            futures.add(taskReadyFuture);
        }

        // Now submit tasks for handling by partner handlers.
        for (Map.Entry<EPartnerId, List<Task>> entry : tasksByPartner.entrySet()) {
            Objects.requireNonNull(partnerDispatcher.get(entry.getKey()))
                    .startHandle(entry.getValue());
        }
        // Wait for all tasks to complete.
        CompletableFuture[] cfs = futures.toArray(new CompletableFuture[0]);
        CompletableFuture.allOf(cfs).
                whenComplete((aVoid, throwable) -> {
                    synchronized (builder) {
                        responseObserver.onNext(builder.build());
                    }
                    responseObserver.onCompleted();
                });
    }

    private ImmutableList<TError> validateRequest(TSearchOffersReq subrequest) {
        ImmutableList.Builder<TError> errors = ImmutableList.builder();
        if (StringUtils.isEmpty(subrequest.getId())) {
            errors.add(ProtoUtils.validationError("Empty request id"));
        }

        EPartnerId partnerId = subrequest.getHotelId().getPartnerId();
        if (!partnerDispatcher.has(partnerId)) {
            errors.add(ProtoUtils.validationError("Unknown partner id",
                    "partner_id", partnerId.toString()));
        }

        String originalId = subrequest.getHotelId().getOriginalId();
        if (originalId.isEmpty()) {
            errors.add(ProtoUtils.validationError("Empty original id"));
        }

        String checkInDate = subrequest.getCheckInDate();
        if (checkInDate.isEmpty()) {
            errors.add(ProtoUtils.validationError("Empty check-in date"));
        } else if (!DATE_PATTERN.matcher(checkInDate).matches()) {
            errors.add(ProtoUtils.validationError("Check-in date must be in form YYYY-MM-DD",
                    "checkin_date", checkInDate));
        }

        String checkOutDate = subrequest.getCheckOutDate();
        if (checkOutDate.isEmpty()) {
            errors.add(ProtoUtils.validationError("Empty check-out date"));
        } else if (!DATE_PATTERN.matcher(checkOutDate).matches()) {
            errors.add(ProtoUtils.validationError("Check-out date must be in form YYYY-MM-DD",
                    "checkout_date", checkOutDate));
        }

        String occupancy = subrequest.getOccupancy();
        if (occupancy.isEmpty()) {
            errors.add(ProtoUtils.validationError("Empty occupancy"));
        } else if (!OCCUPANCY_PATTERN.matcher(occupancy).matches()) {
            errors.add(ProtoUtils.validationError("Occupancy must be in form N[-A,B,C,...]",
                    "occupancy", occupancy));
        }

        if (subrequest.getCurrency() == ECurrency.UNRECOGNIZED || subrequest.getCurrency() == ECurrency.C_UNKNOWN) {
            errors.add(ProtoUtils.validationError("Invalid currency",
                    "currency", Integer.toString(subrequest.getCurrencyValue())));
        }

        if (subrequest.getRequestClass() == ERequestClass.UNRECOGNIZED) {
            errors.add(ProtoUtils.validationError("Unrecognized request class",
                    "request_class", Integer.toString(subrequest.getRequestClassValue())));
        }
        if (subrequest.getPermalink() == 0) {
            errors.add(ProtoUtils.validationError("Empty permalink"));
        }
        return errors.build();
    }
}
