package ru.yandex.travel.hotels.searcher;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.stream.Collectors;

import javax.annotation.PreDestroy;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.stereotype.Component;

@Component
@Slf4j
public class DestroyPrototypeBeansPostProcessor implements BeanPostProcessor, DisposableBean {

    private final Deque<BeanRecord> prototypeBeans = new LinkedList<>();
    private BeanFactory beanFactory;

    @Autowired
    public void setBeanFactory(BeanFactory beanFactory) throws BeansException {
        this.beanFactory = beanFactory;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (isPrototype(beanName)) {
            synchronized (prototypeBeans) {
                prototypeBeans.push(new BeanRecord(bean, beanName));
            }
        }
        return bean;
    }

    @Override
    public void destroy() throws Exception {
        synchronized (prototypeBeans) {
            for (BeanRecord record : prototypeBeans) {
                try {
                    for (Method preDestroyMethod : getPreDestroyMethods(record.bean)) {
                        preDestroyMethod.invoke(record.bean);
                    }
                } catch (Exception e) {
                    log.error("Error on calling one of @PreDestroy methods of bean {} named {}", record.bean.getClass().getCanonicalName(), record.name);
                }

                if (record.bean instanceof DisposableBean) {
                    log.info("Dispose {}", record.name);
                    try {
                        ((DisposableBean) record.bean).destroy();
                    } catch (Exception e) {
                        log.error("Error on destroying bean '{}' named '{}'",
                                record.bean.getClass().getCanonicalName(), record.name, e);
                    }
                }
            }
            prototypeBeans.clear();
        }
    }

    private boolean isPrototype(String name) {
        try {
            return beanFactory.isPrototype(name);
        } catch (NoSuchBeanDefinitionException e) {
            /* In case of NoSuchBeanDefinitionException make assumption that bean is not prototype */
            return false;
        }
    }

    private static List<Method> getPreDestroyMethods(Object bean) {
        return Arrays.stream(bean.getClass().getMethods())
                .filter(method -> method.isAnnotationPresent(PreDestroy.class))
                .collect(Collectors.toList());
    }

    @AllArgsConstructor
    @Data
    private static class BeanRecord {
        private final Object bean;
        private final String name;
    }
}
