package ru.yandex.travel.hotels.searcher;

import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Dsl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClientProperties;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoUidMap;
import ru.yandex.travel.hotels.common.partners.bnovo.DefaultBNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.DefaultBNovoClientWithPublicRemapping;
import ru.yandex.travel.hotels.common.partners.booking.BookingClient;
import ru.yandex.travel.hotels.common.partners.booking.BookingClientProperties;
import ru.yandex.travel.hotels.common.partners.booking.DefaultBookingClient;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClientProperties;
import ru.yandex.travel.hotels.common.partners.bronevik.DefaultBronevikClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DefaultDolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClientProperties;
import ru.yandex.travel.hotels.common.partners.expedia.DefaultExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClientProperties;
import ru.yandex.travel.hotels.common.partners.travelline.DefaultTravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClientProperties;
import ru.yandex.travel.hotels.common.partners.tvil.DefaultTvilClient;
import ru.yandex.travel.hotels.common.partners.tvil.TvilClient;
import ru.yandex.travel.hotels.common.partners.tvil.TvilClientProperties;
import ru.yandex.travel.hotels.common.partners.vipservice.DefaultVipserviceClient;
import ru.yandex.travel.hotels.common.partners.vipservice.VipserviceClient;
import ru.yandex.travel.hotels.common.partners.vipservice.VipserviceClientProperties;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
import ru.yandex.travel.hotels.searcher.partners.BNovoTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.BookingPartnerTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.BronevikTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.DolphinTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.ExpediaTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.ThrottledTravellineClient;
import ru.yandex.travel.hotels.searcher.partners.TravellineTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.TvilTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.partners.VipserviceTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.BNovoYtCacheProperties;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.YtBNovoUidMap;

@Configuration
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties({
        TravellineTaskHandlerProperties.class,
        DolphinTaskHandlerProperties.class,
        ExpediaTaskHandlerProperties.class,
        BNovoTaskHandlerProperties.class,
        TvilTaskHandlerProperties.class,
        BookingPartnerTaskHandlerProperties.class,
        BronevikTaskHandlerProperties.class,
        VipserviceTaskHandlerProperties.class
})
public class HttpClientsConfiguration {

    private static Logger DEFAULT_HTTP_LOGGER = LoggerFactory.getLogger(
            "ru.yandex.travel.hotels.searcher.PartnerHttpLogger");

    private final BookingPartnerTaskHandlerProperties bookingPartnerTaskHandlerProperties;
    private final TravellineTaskHandlerProperties travellineTaskHandlerProperties;
    private final DolphinTaskHandlerProperties dolphinTaskHandlerProperties;
    private final ExpediaTaskHandlerProperties expediaTaskHandlerProperties;
    private final BronevikTaskHandlerProperties bronevikTaskHandlerProperties;
    private final VipserviceTaskHandlerProperties vipserviceTaskHandlerProperties;

    @Bean
    public AsyncHttpClient bookingAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_BOOKING.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient travellineAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_TRAVELLINE.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient dolphinAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_DOLPHIN.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient bNovoAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_BNOVO.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient expediaAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_EXPEDIA.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient tvilAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_TVIL.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClient vipserviceAsyncClient() {
        return Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName(EPartnerId.PI_VIPSERVICE.toString() + "_POOL")
                .build());
    }

    @Bean
    public AsyncHttpClientWrapper bookingAhcWrapper(AsyncHttpClient bookingAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(bookingAsyncClient,
                DEFAULT_HTTP_LOGGER, "booking", tracer, DefaultBookingClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper travellineAhcWrapper(AsyncHttpClient travellineAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(travellineAsyncClient,
                DEFAULT_HTTP_LOGGER, "travelline", tracer, DefaultTravellineClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper dolphinAhcWrapper(AsyncHttpClient dolphinAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(dolphinAsyncClient,
                DEFAULT_HTTP_LOGGER, "dolphin", tracer, DefaultDolphinClient.getMethods().getNames());
    }

    @Bean
    AsyncHttpClientWrapper bNovoAhcWrapper(AsyncHttpClient bNovoAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(bNovoAsyncClient, DEFAULT_HTTP_LOGGER, "bnovo", tracer,
                DefaultBNovoClient.getMethods().getNames());
    }

    @Bean
    AsyncHttpClientWrapper expediaAhcWrapper(AsyncHttpClient expediaAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(expediaAsyncClient, DEFAULT_HTTP_LOGGER, "expedia", tracer,
                DefaultExpediaClient.getMethods().getNames());
    }

    @Bean
    AsyncHttpClientWrapper tvilAhcWrapper(AsyncHttpClient tvilAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(tvilAsyncClient, DEFAULT_HTTP_LOGGER, "tvil", tracer,
                DefaultTvilClient.getMethods().getNames());
    }

    @Bean
    AsyncHttpClientWrapper vipserviceAhcWrapper(AsyncHttpClient vipserviceAsyncClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(vipserviceAsyncClient, DEFAULT_HTTP_LOGGER, "vipservice", tracer,
                DefaultVipserviceClient.getMethods().getNames());
    }

    @Bean
    public BookingClient bookingClient(AsyncHttpClientWrapper bookingAhcWrapper, Retry retryHelper) {
        BookingClientProperties clientProperties = new BookingClientProperties();
        clientProperties.setBaseUrl(bookingPartnerTaskHandlerProperties.getBaseUrl());
        clientProperties.setHttpReadTimeout(bookingPartnerTaskHandlerProperties.getHttpReadTimeout());
        clientProperties.setHttpRequestTimeout(bookingPartnerTaskHandlerProperties.getHttpRequestTimeout());
        clientProperties.setEnableRetries(true);
        clientProperties.setAffiliateId(bookingPartnerTaskHandlerProperties.getAffiliateId());
        clientProperties.setUsername(bookingPartnerTaskHandlerProperties.getUsername());
        clientProperties.setPassword(bookingPartnerTaskHandlerProperties.getPassword());
        return new DefaultBookingClient(bookingAhcWrapper, clientProperties, retryHelper);
    }

    @Bean
    public DolphinClient dolphinClient(AsyncHttpClientWrapper dolphinAhcWrapper,
                                       Retry retryHelper) {
        DolphinClientProperties dolphinClientProperties = new DolphinClientProperties();
        dolphinClientProperties.setBaseUrl(dolphinTaskHandlerProperties.getBaseUrl());
        dolphinClientProperties.setHttpReadTimeout(dolphinTaskHandlerProperties.getHttpReadTimeout());
        dolphinClientProperties.setHttpRequestTimeout(dolphinTaskHandlerProperties.getHttpRequestTimeout());
        dolphinClientProperties.setPassword(dolphinTaskHandlerProperties.getPassword());
        dolphinClientProperties.setLogin(dolphinTaskHandlerProperties.getLogin());
        return new DefaultDolphinClient(dolphinAhcWrapper, dolphinClientProperties, retryHelper);
    }

    @Bean
    public TravellineClient travellineClient(AsyncHttpClientWrapper travellineAhcWrapper,
                                             Retry retryHelper) {
        //TODO extract these properties once all partners use wrappedHttpClient
        TravellineClientProperties clientProperties = new TravellineClientProperties();
        clientProperties.setBaseUrl(travellineTaskHandlerProperties.getBaseUrl());
        clientProperties.setHttpReadTimeout(travellineTaskHandlerProperties.getHttpReadTimeout());
        clientProperties.setHttpRequestTimeout(travellineTaskHandlerProperties.getHttpRequestTimeout());
        clientProperties.setEnableRetries(true);
        clientProperties.setApiKey(travellineTaskHandlerProperties.getApiKey());
        return new DefaultTravellineClient(travellineAhcWrapper, clientProperties, retryHelper);
    }

    @Bean
    public ExpediaClient expediaClient(AsyncHttpClientWrapper expediaAhcWrapper,
                                       Retry retryHelper) {
        ExpediaClientProperties clientProperties = new ExpediaClientProperties();
        clientProperties.setBaseUrl(expediaTaskHandlerProperties.getBaseUrl());
        clientProperties.setHttpReadTimeout(expediaTaskHandlerProperties.getHttpReadTimeout());
        clientProperties.setHttpRequestTimeout(expediaTaskHandlerProperties.getHttpRequestTimeout());
        clientProperties.setEnableRetries(true);
        clientProperties.setProfileType(expediaTaskHandlerProperties.getProfileType());
        clientProperties.setApiKey(expediaTaskHandlerProperties.getApiKey());
        clientProperties.setApiSecret(expediaTaskHandlerProperties.getSecret());
        clientProperties.setDefaultApiVersion(expediaTaskHandlerProperties.getDefaultApiVersion());
        return new DefaultExpediaClient(clientProperties, expediaAhcWrapper, retryHelper);
    }

    @Bean
    public BronevikClient bronevikClient(Retry retryHelper) {
        BronevikClientProperties clientProperties = bronevikTaskHandlerProperties.getClient();
        return new DefaultBronevikClient(clientProperties, DEFAULT_HTTP_LOGGER, retryHelper);
    }

    @Bean
    @ConditionalOnProperty("partners.bnovo.remap-ids")
    public BNovoUidMap bNovoUidMap(BNovoYtCacheProperties properties) {
        return new YtBNovoUidMap(properties.getUidMapParams());
    }

    @Bean
    @ConditionalOnProperty(value = "partners.bnovo.remap-ids", havingValue = "false", matchIfMissing = true)
    public BNovoClient bNovoClient(AsyncHttpClientWrapper bNovoAhcWrapper,
                                   BNovoTaskHandlerProperties bNovoTaskHandlerProperties, Retry retryHelper) {
        BNovoClientProperties clientProperties = buildBNovoClientProperties(bNovoTaskHandlerProperties);
        return new DefaultBNovoClient(bNovoAhcWrapper, clientProperties, retryHelper);
    }

    private BNovoClientProperties buildBNovoClientProperties(BNovoTaskHandlerProperties bNovoTaskHandlerProperties) {
        BNovoClientProperties clientProperties = new BNovoClientProperties();
        clientProperties.setBaseUrl(bNovoTaskHandlerProperties.getBaseUrl());
        clientProperties.setBaseUrl(bNovoTaskHandlerProperties.getBaseUrl());
        clientProperties.setHttpReadTimeout(bNovoTaskHandlerProperties.getHttpReadTimeout());
        clientProperties.setHttpRequestTimeout(bNovoTaskHandlerProperties.getHttpRequestTimeout());
        clientProperties.setUsername(bNovoTaskHandlerProperties.getUsername());
        clientProperties.setPassword(bNovoTaskHandlerProperties.getPassword());
        clientProperties.setPrivateApiBaseUrl(bNovoTaskHandlerProperties.getPrivateApiBaseUrl());
        clientProperties.setPricesLosApiBaseUrl(bNovoTaskHandlerProperties.getPricesLosApiBaseUrl());
        clientProperties.setTokenValidityDuration(bNovoTaskHandlerProperties.getTokenValidityDuration());
        clientProperties.setEnableRetries(bNovoTaskHandlerProperties.isEnableRetries());
        clientProperties.setAuthenticationMaxRetryCount(bNovoTaskHandlerProperties.getAuthenticationMaxRetryCount());
        return clientProperties;
    }

    @Bean
    @ConditionalOnProperty("partners.bnovo.remap-ids")
    public BNovoClient remappingBNovoClient(AsyncHttpClientWrapper bNovoAhcWrapper,
                                            BNovoTaskHandlerProperties bNovoTaskHandlerProperties, Retry retryHelper,
                                            BNovoUidMap uidMap) {
        BNovoClientProperties clientProperties = buildBNovoClientProperties(bNovoTaskHandlerProperties);
        return new DefaultBNovoClientWithPublicRemapping(bNovoAhcWrapper, clientProperties, uidMap, retryHelper);
    }

    @Bean
    public TvilClient tvilClient(AsyncHttpClientWrapper tvilAhcWrapper,
                                 TvilTaskHandlerProperties taskHandlerProperties, Retry retryHelper) {
        TvilClientProperties clientProperties = TvilClientProperties.builder()
                .baseUrl(taskHandlerProperties.getBaseUrl())
                .httpRequestTimeout(taskHandlerProperties.getHttpRequestTimeout())
                .httpReadTimeout(taskHandlerProperties.getHttpReadTimeout())
                .enableRetries(true)
                .build();
        return new DefaultTvilClient(clientProperties, tvilAhcWrapper, retryHelper);
    }

    @Bean
    public TravellineClient backgroundThrottledTravellineClient(TravellineClient travellineClient) {
        final TravellineTaskHandlerProperties.ThrottlerProperties props =
                travellineTaskHandlerProperties.getThrottlers().getBackground();
        var parameters = new ThrottledWrapper.ThrottlingParameters(
                props.getRateLimit(),
                props.getConcurrencyLimit(),
                props.getQueueLimit(),
                props.getBucket(),
                props.getWindow(),
                travellineTaskHandlerProperties.getConcurrencyReschedulePeriod());
        return new ThrottledTravellineClient(travellineClient, parameters, "http.client.travelline",
                "class", ERequestClass.RC_BACKGROUND.name());
    }

    @Bean
    public TravellineClient interactiveThrottledTravellineClient(TravellineClient travellineClient) {
        final TravellineTaskHandlerProperties.ThrottlerProperties props =
                travellineTaskHandlerProperties.getThrottlers().getInteractive();
        var parameters = new ThrottledWrapper.ThrottlingParameters(
                props.getRateLimit(),
                props.getConcurrencyLimit(),
                0,
                props.getBucket(),
                props.getWindow(),
                travellineTaskHandlerProperties.getConcurrencyReschedulePeriod());
        return new ThrottledTravellineClient(travellineClient, parameters, "http.client.travelline",
                "class", ERequestClass.RC_INTERACTIVE.name());
    }

    @Bean
    public TravellineClient inventoryThrottledTravellineClient(TravellineClient travellineClient) {
        final TravellineTaskHandlerProperties.ThrottlerProperties props =
                travellineTaskHandlerProperties.getThrottlers().getInventory();
        var parameters = new ThrottledWrapper.ThrottlingParameters(
                props.getRateLimit(),
                props.getConcurrencyLimit(),
                props.getQueueLimit(),
                props.getBucket(),
                props.getWindow(),
                travellineTaskHandlerProperties.getConcurrencyReschedulePeriod());
        return new ThrottledTravellineClient(travellineClient, parameters, "http.client.travelline",
                "class", "INVENTORY");
    }

    @Bean
    public VipserviceClient vipserviceClient(AsyncHttpClientWrapper vipserviceAhcWrapper, Retry retryHelper) {
        VipserviceClientProperties clientProperties = new VipserviceClientProperties();
        clientProperties.setBaseUrl(vipserviceTaskHandlerProperties.getBaseUrl());
        clientProperties.setHttpReadTimeout(vipserviceTaskHandlerProperties.getHttpReadTimeout());
        clientProperties.setHttpRequestTimeout(vipserviceTaskHandlerProperties.getHttpRequestTimeout());
        clientProperties.setApiKey(vipserviceTaskHandlerProperties.getApiKey());
        return new DefaultVipserviceClient(clientProperties, vipserviceAhcWrapper, retryHelper);
    }
}
