package ru.yandex.travel.hotels.searcher;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanInitializationException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Component;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.searcher.partners.PartnerTaskHandler;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Consumer;


@Component
public class PartnerDispatcher implements ApplicationContextAware, InitializingBean {
    private ApplicationContext context;
    private Map<EPartnerId, PartnerTaskHandler> partners = new HashMap<>();
    private Logger logger = LoggerFactory.getLogger(this.getClass());

    @Override
    public void afterPropertiesSet() throws Exception {
        context.getBeansWithAnnotation(PartnerBean.class).values().forEach(o -> {
            if (!(o instanceof PartnerTaskHandler)) {
                throw new BeanInitializationException(String.format("Bean '%1$s' is not an instance of '%2$s'",
                        o.getClass().getName(),
                        PartnerTaskHandler.class.getName()));
            }
            PartnerTaskHandler h = (PartnerTaskHandler) o;
            EPartnerId key = h.getClass().getDeclaredAnnotation(PartnerBean.class).value();
            if (partners.containsKey(key)) {
                throw new BeanInitializationException(String.format(
                        "Bean '%1$s' can't handle partner '%2$s' as it is already being handled by bean '%3$s'",
                        o.getClass().getName(), key, partners.get(key).getClass().getName()));
            }
            logger.info("Registered '{}' as a handler for partner '{}'", h.getClass().getName(), key.toString());
            partners.put(key, h);
        });
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        context = applicationContext;
    }

    public void forEach(Consumer<? super PartnerTaskHandler> action) {
        partners.values().forEach(action);
    }

    public PartnerTaskHandler get(EPartnerId partnerId) {
        return partners.get(partnerId);
    }

    boolean has(EPartnerId partnerId) {
        return partners.containsKey(partnerId);
    }
}
