package ru.yandex.travel.hotels.searcher;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.validation.constraints.NotNull;
import javax.validation.constraints.Positive;

import com.google.common.base.Preconditions;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.misc.thread.factory.ThreadNameThreadFactory;
import ru.yandex.travel.commons.rate.Throttler;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
@Slf4j
public class QueueConsumer {
    @NotNull
    @Getter
    private final ERequestClass requestClass;
    @NotNull
    private final TaskStarter taskStarter;
    @NotNull
    @Getter
    private final Throttler throttler;
    @NotNull
    @Getter
    private final BatchedTaskQueue queue;
    private final int batchSize;
    @NotNull
    private final ScheduledExecutorService queueConsumerExecutor;
    private final ExecutorService taskExecutor;
    @NotNull
    private final AtomicBoolean isRunning = new AtomicBoolean();
    @NotNull
    private final boolean dropOnLimit;
    @NotNull
    private final Consumer<? super Task> onRateLimitDrop;
    @NotNull
    private final Consumer<? super Task> onConcurrencyLimitDrop;
    @NotNull
    private final Duration rateLimiterRescheduleDelay;
    @NotNull
    private final Duration concurrencyRescheduleDelay;

    private QueueConsumer(Builder builder) {
        this.requestClass = builder.requestClass;
        this.taskStarter = builder.taskStarter;
        this.throttler = new Throttler(builder.rateLimit, builder.concurrencyLimit,
                builder.rateLimiterBucket, builder.rateLimiterWindow);
        this.queue = new BatchedTaskQueue(builder.queueSize);
        this.batchSize = builder.batchSize;
        this.queueConsumerExecutor = builder.queueConsumerExecutor.apply(new ThreadNameThreadFactory(
                String.format("%s-handler-consumer-thread-%s", builder.partnerId, builder.requestClass)
        ));
        this.taskExecutor = builder.taskExecutor.apply(new ThreadNameThreadFactory(
                String.format("%s-handler-task-thread-%s", builder.partnerId, builder.requestClass)
        ));
        this.dropOnLimit = builder.dropOnLimit;
        this.onRateLimitDrop = builder.onRateLimitDrop;
        this.onConcurrencyLimitDrop = builder.onConcurrencyLimitDrop;
        this.rateLimiterRescheduleDelay = builder.rateLimiterRescheduleDelay;
        this.concurrencyRescheduleDelay = builder.concurrencyRescheduleDelay;
    }

    public static class Builder {
        private final EPartnerId partnerId;
        private final ERequestClass requestClass;
        private TaskStarter taskStarter;
        private int rateLimit = 20;
        private int concurrencyLimit = 20;
        private Duration rateLimiterBucket = Duration.ofSeconds(1);
        private Duration rateLimiterWindow = Duration.ofSeconds(5);
        private int queueSize = Integer.MAX_VALUE;
        private int batchSize = 1;
        private Function<ThreadNameThreadFactory, ScheduledExecutorService> queueConsumerExecutor;
        private Function<ThreadNameThreadFactory, ExecutorService> taskExecutor;
        private boolean dropOnLimit = true;
        private Consumer<? super Task> onRateLimitDrop;
        private Consumer<? super Task> onConcurrencyLimitDrop;
        private Duration rateLimiterRescheduleDelay;
        private Duration concurrencyRescheduleDelay;

        public Builder(EPartnerId partnerId, ERequestClass requestClass) {
            Preconditions.checkNotNull(requestClass);
            Preconditions.checkNotNull(partnerId);
            this.requestClass = requestClass;
            this.partnerId = partnerId;
        }

        public Builder withThrottlerParams(
                @Positive int rateLimit,
                @Positive int concurrencyLimit,
                @NotNull Duration rateLimiterBucket,
                @NotNull Duration rateLimiterWindow
        ) {
            Preconditions.checkState(rateLimit >= 0);
            Preconditions.checkState(concurrencyLimit >= 0);
            Preconditions.checkNotNull(rateLimiterBucket);
            Preconditions.checkNotNull(rateLimiterWindow);
            this.rateLimit = rateLimit;
            this.concurrencyLimit = concurrencyLimit;
            this.rateLimiterBucket = rateLimiterBucket;
            this.rateLimiterWindow = rateLimiterWindow;
            return this;
        }

        /**
         * Task starter MUST release throttler lock, otherwise it would lock up all the processing
         */
        public Builder withTaskStarter(@NotNull TaskStarter taskStarter) {
            Preconditions.checkNotNull(taskStarter);
            this.taskStarter = taskStarter;
            return this;
        }

        public Builder withLimitedQueueSize(@Positive int size) {
            Preconditions.checkState(size > 0);
            this.queueSize = size;
            return this;
        }

        public Builder dontDropOnLimit(@NotNull Duration rateLimiterRescheduleDelay,
                                       @NotNull Duration concurrencyRescheduleDelay) {
            Preconditions.checkNotNull(rateLimiterRescheduleDelay);
            Preconditions.checkNotNull(concurrencyRescheduleDelay);
            this.dropOnLimit = false;
            this.rateLimiterRescheduleDelay = rateLimiterRescheduleDelay;
            this.concurrencyRescheduleDelay = concurrencyRescheduleDelay;
            return this;
        }

        /**
         * Sets executor to {@link #startSelfRestartingQueueConsumer} on.
         */
        public Builder withQueueConsumerExecutor(Function<ThreadNameThreadFactory, ScheduledExecutorService> executor) {
            this.queueConsumerExecutor = executor;
            return this;
        }

        /**
         * Sets {@link #taskStarter} executor
         *
         * @param executor an executor to run {@link #taskStarter} on.
         */
        public Builder withTaskExecutor(Function<ThreadNameThreadFactory, ExecutorService> executor) {
            this.taskExecutor = executor;
            return this;
        }

        public Builder withDropOnLimitActions(
                Consumer<? super Task> onRateLimitDrop,
                Consumer<? super Task> onConcurrencyLimitDrop
        ) {
            this.onRateLimitDrop = onRateLimitDrop;
            this.onConcurrencyLimitDrop = onConcurrencyLimitDrop;
            return this;
        }

        public Builder withBatchSize(int batchSize) {
            this.batchSize = batchSize;
            return this;
        }

        public QueueConsumer build() {
            Preconditions.checkNotNull(requestClass);
            Preconditions.checkNotNull(taskStarter, "Provide `startNow` method via withTaskStarter()");
            Preconditions.checkState(rateLimit >= 0);
            Preconditions.checkState(concurrencyLimit >= 0);
            Preconditions.checkNotNull(rateLimiterBucket);
            Preconditions.checkNotNull(rateLimiterWindow);
            Preconditions.checkState(queueSize > 0);
            Preconditions.checkState(batchSize > 0);
            Preconditions.checkNotNull(queueConsumerExecutor, "Prove executor for queue consumption via withQueueConsumerExecutor()");
            Preconditions.checkNotNull(taskExecutor, "Prove executor for workload via withTaskExecutor()");
            if (this.dropOnLimit) {
                Preconditions.checkState(this.onRateLimitDrop != null && this.onConcurrencyLimitDrop != null,
                        "Either set Builder.dontDropOnLimit() or" +
                                " provide limit actions via Builder.withDropOnLimitActions(), not both");
            } else {
                Preconditions.checkState(this.onRateLimitDrop == null && this.onConcurrencyLimitDrop == null,
                        "Either set Builder.dontDropOnLimit() or " +
                                "provide limit actions via Builder.withDropOnLimitActions(), not both"
                );
                Preconditions.checkNotNull(this.rateLimiterRescheduleDelay,
                        "Provide non-null rateLimiterRescheduleDelay");
                Preconditions.checkNotNull(this.concurrencyRescheduleDelay,
                        "Provide non-null concurrencyRescheduleDelay");
            }
            return new QueueConsumer(this);
        }

    }

    public List<Runnable> shutdownNow() {
        List<Runnable> shuttingDown = new ArrayList<>();
        shuttingDown.addAll(this.queueConsumerExecutor.shutdownNow());
        shuttingDown.addAll(this.taskExecutor.shutdownNow());
        return shuttingDown;
    }

    public void trySchedulingQueueConsumer() {
        if (isRunning.compareAndSet(false, true)) {
            startSelfRestartingQueueConsumer();
        }
    }

    private void startSelfRestartingQueueConsumer() {
        queueConsumerExecutor.submit(() -> {
            Duration resumeAfter;
            try {
                resumeAfter = consumeQueue();
            }catch(Exception e){
                log.error("Error consuming queue", e);
                resumeAfter = Duration.ofSeconds(1); // restart broken queue consumer in a second
            }
            if (resumeAfter != null) {
                queueConsumerExecutor.schedule(this::startSelfRestartingQueueConsumer,
                        resumeAfter.toMillis(), TimeUnit.MILLISECONDS);
            }
        });
    }

    private Duration consumeQueue() {
        while (true) {
            if (queue.isEmpty()) {
                // T1: offer -> schedule
                // T2: peek -> unschedule
                // Avoid bad execution like T2:peek -> T1:offer -> T1:schedule -> T2:unschedule
                isRunning.set(false);
                if (!queue.isEmpty()) {
                    trySchedulingQueueConsumer();
                }
                return null;
            }
            BatchedTaskQueue.Batch batch = queue.getBatch(batchSize);
            if (batch == null) {
                // that should not be possible without concurrent queue drainage;
                // batch may be null only if queue is empty, so lets just recheck it
                continue;
            }
            List<Task> batchTasks = batch.getTasks();
            Throttler.EDecision decision = throttler.acquire(System.currentTimeMillis());
            switch (decision) {
                case RATE_LIMIT:
                    if (dropOnLimit) {
                        batchTasks.forEach(onRateLimitDrop);
                    } else {
                        return rateLimiterRescheduleDelay;
                    }
                    break;
                case CONCURRENCY_LIMIT:
                    if (dropOnLimit) {
                        batchTasks.forEach(onConcurrencyLimitDrop);
                    } else {
                        return concurrencyRescheduleDelay;
                    }
                    break;
                case PASS:
                    try {
                        taskExecutor.execute(() -> taskStarter.startNow(batch.getKey(), batchTasks, this));
                    } catch (RejectedExecutionException e) {
                        if (dropOnLimit) {
                            batchTasks.forEach(onConcurrencyLimitDrop);
                        } else {
                            return concurrencyRescheduleDelay;
                        }
                    }
                    break;
            }
        }
    }

    @FunctionalInterface
    public interface TaskStarter {
        void startNow(Task.GroupingKey groupingKey, List<Task> batch, QueueConsumer queueExecutor);
    }

}
