package ru.yandex.travel.hotels.searcher;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import com.google.protobuf.UInt32Value;
import lombok.Getter;
import lombok.Setter;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
import ru.yandex.travel.hotels.proto.ESearchWarningCode;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferList;
import ru.yandex.travel.hotels.proto.TSearchOffersReq;
import ru.yandex.travel.hotels.proto.TSearchOffersRsp;
import ru.yandex.travel.hotels.proto.TSearchWarnings;

public class Task {
    private final TSearchOffersReq request;
    private final Map<String, Message> messageStorage;
    private final Map<ESearchWarningCode, Integer> warningsStorage;
    private CompletableFuture<Void> completionFuture;
    private TOfferList.Builder offerListBuilder = TOfferList.newBuilder();
    private TError offerError;
    @Getter
    @Setter
    private Duration cacheLifetime;
    private String id;
    @Getter
    @Setter
    private String httpRequestId;
    @Getter
    @Setter
    private long createdAtNanos;
    @Getter
    @Setter
    private long startedAtNanos = 0;
    @Getter
    @Setter
    private long completedAtNanos = 0;
    private boolean includeDebug;
    @Getter
    private final CallContext callContext;

    public Task(TSearchOffersReq request, boolean includeDebug) {
        this(request, includeDebug, System.nanoTime(), CallContext.forSearcher(null, null));
    }

    public Task(TSearchOffersReq request, boolean includeDebug, long createdAtNanos) {
        this(request, includeDebug, createdAtNanos, CallContext.forSearcher(null, null));
    }

    public Task(TSearchOffersReq request, boolean includeDebug, long createdAtNanos, CallContext callContext) {
        Preconditions.checkArgument(!StringUtils.isEmpty(request.getId()), "Request has no Id");
        this.createdAtNanos = createdAtNanos;
        this.request = request;
        this.includeDebug = includeDebug;
        this.completionFuture = new CompletableFuture<>();
        this.id = request.getId();
        this.messageStorage = new HashMap<>();
        this.warningsStorage = new HashMap<>();
        this.callContext = callContext;
    }

    public TSearchOffersReq getRequest() {
        return request;
    }

    public Occupancy getOccupancy() {
        return Occupancy.fromString(request.getOccupancy());
    }

    public TSearchOffersRsp.Builder dumpResultTo(TSearchOffersRsp.Builder builder) {
        if (offerError != null) {
            builder.setError(offerError);
        } else {
            offerListBuilder.setCacheTimeSec(UInt32Value.of((int)cacheLifetime.toSeconds()));
            if (!warningsStorage.isEmpty()) {
                var warningsBuilder = TSearchWarnings.newBuilder();
                for (var w : warningsStorage.entrySet()) {
                    warningsBuilder.addWarnings(TSearchWarnings.TSearchWarning.newBuilder()
                            .setCode(w.getKey())
                            .setCount(w.getValue())
                            .build());
                }
                offerListBuilder.setWarnings(warningsBuilder.build());
            }
            builder.setOffers(offerListBuilder);
        }
        return builder;
    }

    public int getOfferCount() {
        return offerListBuilder.getOfferCount();
    }

    public long getUnknownPansionCount() {
        return offerListBuilder.getOfferList().stream().filter(o -> o.getPansion() == EPansionType.PT_UNKNOWN).count();
    }

    public boolean isEmpty() {
        return offerListBuilder.getOfferCount() == 0;
    }

    public TSearchOffersRsp dumpResult() {
        return dumpResultTo(TSearchOffersRsp.newBuilder()).build();
    }

    public void onError(TError error) {
        offerError = error;
    }

    public void onError(TError.Builder builder) {
        onError(builder.build());
    }

    public void onWarning(ESearchWarningCode warning) {
        warningsStorage.put(warning, warningsStorage.getOrDefault(warning, 0) + 1);
    }

    public boolean hasError() {
        return offerError != null;
    }

    public String getId() {
        return id;
    }

    // This method should be called only from AbstractPartnerTaskHandler
    // Do not call it from ConcretePartnerTaskHandler
    public void onOfferImpl(TOffer.Builder builder) {
        if (StringUtils.isEmpty(builder.getId())) {
            builder.setId(ProtoUtils.randomId());
        }
        if (!builder.hasActualizationTime()) {
            builder.setActualizationTime(ProtoUtils.timestamp());
        }
        if (builder.getOperatorId() == EOperatorId.UNRECOGNIZED) {
            throw new IllegalArgumentException("Offer has no operator id");
        }
        if (!builder.hasPrice()) {
            throw new IllegalArgumentException("Offer has no price");
        }
        if (builder.getCapacity().isEmpty()) {
            throw new IllegalArgumentException("Offer has no capacity");
        }
        if (builder.getPansion() == EPansionType.UNRECOGNIZED) {
            throw new IllegalArgumentException("Offer has no pansion");
        }
        offerListBuilder.addOffer(builder);
    }

    public void storeMessage(String key, Message message) {
        messageStorage.put(key, message);
    }

    public void onComplete() {
        this.completionFuture.complete(null);
    }

    public boolean isIncludeDebug() {
        return includeDebug;
    }

    public GroupingKey getGroupingKey() {
        return new GroupingKey(request);
    }

    public CompletableFuture<Void> getCompletionFuture() {
        return completionFuture;
    }

    public TError getOfferError() {
        return offerError;
    }

    public List<TOffer> getOfferList() {
        return offerListBuilder.getOfferList();
    }

    public Map<String, Message> getMessagesToPutToStorage() {
        return Collections.unmodifiableMap(messageStorage);
    }

    public void clearOfferList() {
        offerListBuilder.clearOffer();
    }

    public static class GroupingKey {
        private final TSearchOffersReq request;

        public GroupingKey(TSearchOffersReq request) {
            this.request = request;
        }

        @Override
        public int hashCode() {
            return Objects.hashCode(request.getHotelId().getPartnerId(),
                    request.getCheckInDate(),
                    request.getCheckOutDate(),
                    request.getOccupancy(),
                    request.getCurrency(),
                    request.getRequestClass());
        }

        @Override
        public boolean equals(Object obj) {
            if (!(obj instanceof GroupingKey)) {
                return false;
            }
            TSearchOffersReq other = ((GroupingKey) obj).request;
            return request.getHotelId().getPartnerId() == other.getHotelId().getPartnerId() &&
                    request.getCheckInDate().equals(other.getCheckInDate()) &&
                    request.getCheckOutDate().equals(other.getCheckOutDate()) &&
                    request.getOccupancy().equals(other.getOccupancy()) &&
                    request.getCurrency() == other.getCurrency() &&
                    request.getRequestClass() == other.getRequestClass();
        }

        public EPartnerId getPartnerId() {
            return request.getHotelId().getPartnerId();
        }

        public String getCheckInDate() {
            return request.getCheckInDate();
        }

        public String getCheckOutDate() {
            return request.getCheckOutDate();
        }

        public Occupancy getOccupancy() {
            return Occupancy.fromString(request.getOccupancy());
        }

        public ECurrency getCurrency() {
            return request.getCurrency();
        }

        public ERequestClass getRequestClass() {
            return request.getRequestClass();
        }
    }
}
