package ru.yandex.travel.hotels.searcher;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.rate.Throttler;

@Slf4j
public class ThrottledWrapper<T> implements InitializingBean, DisposableBean {
    protected final T wrapped;

    private final Throttler throttler;
    private final LinkedBlockingQueue<QueuedRequest<?>> queue;
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final ThrottlingParameters parameters;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final Counter passedRequests;
    private final Counter throttledByRateLimit;
    private final Counter throttledByConcurrecy;
    private final Counter throttledByQueue;


    public ThrottledWrapper(T wrapped, ThrottlingParameters parameters, String name, String... tags) {
        this.parameters = parameters;
        this.wrapped = wrapped;
        throttler = new Throttler(parameters.rateLimit, parameters.concurrencyLimit, parameters.bucketSize,
                parameters.windowSize);
        List<String> tagsList = Arrays.asList(tags);
        var byRateTags = new ArrayList<>(tagsList);
        var byConcurencyTags = new ArrayList<>(tagsList);
        var byQueueTags = new ArrayList<>(tagsList);
        byRateTags.addAll(List.of("reason", "RATE_LIMIT"));
        byConcurencyTags.addAll(List.of("reason", "CONCURRENCY_LIMIT"));
        byQueueTags.addAll(List.of("reason", "QUEUE_LIMIT"));
        passedRequests = Metrics.counter(name + ".passedRequests", tags);
        throttledByRateLimit = Metrics.counter(name + ".throttledRequests", byRateTags.toArray(new String[0]));
        throttledByConcurrecy = Metrics.counter(name  + ".throttledRequests",  byConcurencyTags.toArray(new String[0]));
        throttledByQueue = Metrics.counter(name  + ".throttledRequests", byQueueTags.toArray(new String[0]));
        Gauge.builder(name + ".semaphoreValue", throttler, Throttler::getSemaphoreValue)
                .tags(tags)
                .register(Metrics.globalRegistry);
        Gauge.builder(name + ".semaphoreLimit", throttler, Throttler::getSemaphoreLimit)
                .tags(tags)
                .register(Metrics.globalRegistry);
        if (parameters.queueLimit > 0) {
            queue = new LinkedBlockingQueue<>(parameters.queueLimit);
            Gauge.builder(name + ".queueSize",  queue, LinkedBlockingQueue::size)
                    .tags(tags)
                    .register(Metrics.globalRegistry);
        } else {
            queue = null;
        }
    }

    protected void processBackgroundQueue() {
        Preconditions.checkArgument(queue != null, "Background queue is not enabled");
        log.info("Starting background request queue");
        QueuedRequest<?> request = null;
        while (running.get()) {
            try {
                if (request == null) {
                    request = queue.take();
                }
                switch (throttler.acquire(System.currentTimeMillis())) {
                    case PASS:
                        log.debug("Running background request");
                        passedRequests.increment();
                        request.run();
                        // nullify request so it is re-taken on the next loop iteration
                        request = null;
                        break;
                    case RATE_LIMIT:
                        log.debug("Background request is throttled due to rate limit");
                        Thread.sleep(parameters.getBucketSize().toMillis());
                        throttledByRateLimit.increment();
                        break;
                    case CONCURRENCY_LIMIT:
                        log.debug("Background request is throttled due to concurrency limit");
                        Thread.sleep(parameters.getConcurrencyReschedulePeriod().toMillis());
                        throttledByConcurrecy.increment();
                        break;
                }
            } catch (InterruptedException e) {
                log.info("Interrupted, will exit background processing loop");
                Thread.currentThread().interrupt();
                running.set(false);
            }
        }
        log.info("Background processing loop has ended");
    }

    protected <V> CompletableFuture<V> call(Supplier<CompletableFuture<V>> method) {
        if (queue != null) {
            QueuedRequest<V> request = new QueuedRequest<V>(method);
            if (!queue.offer(request)) {
                throttledByQueue.increment();
                return CompletableFuture.failedFuture(new ErrorException(
                        TError.newBuilder()
                                .setMessage("Call throttled due to queue limit exceeded")
                                .setCode(EErrorCode.EC_RESOURCE_EXHAUSTED)
                                .build()));
            } else {
                log.debug("Queued background request");
                return request.getFuture();
            }
        } else {
            Throttler.EDecision decision = throttler.acquire(System.currentTimeMillis());
            switch (decision) {
                case PASS:
                    passedRequests.increment();
                    return method.get().whenComplete((r, t) -> throttler.release());
                case CONCURRENCY_LIMIT:
                    throttledByConcurrecy.increment();
                    break;
                case RATE_LIMIT:
                    throttledByRateLimit.increment();
                    break;
            }
            return CompletableFuture.failedFuture(new ErrorException(
                    TError.newBuilder()
                            .setMessage("Call throttled due to " + decision.toString())
                            .setCode(EErrorCode.EC_RESOURCE_EXHAUSTED)
                            .build()
            ));
        }
    }

    @Override
    public void destroy() throws Exception {
        running.set(false);
        executor.shutdown();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        if (queue != null) {
            running.set(true);
            executor.submit(this::processBackgroundQueue);
        }
    }

    @Data
    @AllArgsConstructor
    public static class ThrottlingParameters {
        private int rateLimit;
        private int concurrencyLimit;
        private int queueLimit;
        private Duration bucketSize;
        private Duration windowSize;
        private Duration concurrencyReschedulePeriod;
    }

    class QueuedRequest<R> {
        private final Supplier<CompletableFuture<R>> supplier;
        @Getter
        private final CompletableFuture<R> future;

        private QueuedRequest(Supplier<CompletableFuture<R>> supplier) {
            this.supplier = supplier;
            this.future = new CompletableFuture<>();
        }

        public void run() {
            supplier.get().whenComplete((r, t) -> {
                if (t == null) {
                    this.future.complete(r);
                } else {
                    this.future.completeExceptionally(t);
                }
                throttler.release();
            });
        }
    }


}
