package ru.yandex.travel.hotels.searcher.cold;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import lombok.Getter;
import org.springframework.stereotype.Component;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TSearchOffersReq;
import ru.yandex.travel.hotels.searcher.Task;

import java.time.Duration;
import java.time.format.DateTimeFormatter;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.HashMap;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Optional;
import java.util.TreeMap;

@Slf4j
public class ColdService {
    @Getter
    @AllArgsConstructor
    private static class OfferFeatures {
        private EPartnerId partnerId;
        private int daysBefore;
        private int hour;
    }

    private static class FeaturesToLifetimeMapping {
        private HashMap<EPartnerId, TreeMap<Integer, TreeMap<Integer, Duration>>> mapping = new HashMap<>();
        @Getter
        private int size = 0;

        Optional<Duration> getLifetime(OfferFeatures features) {
            TreeMap<Integer, TreeMap<Integer, Duration>> withPartnerId = mapping.get(features.getPartnerId());
            if (withPartnerId == null) {
                return Optional.empty();
            }
            TreeMap<Integer, Duration> withPartnerIdAndDaysBefore =
                getCeilingValue(withPartnerId, features.getDaysBefore());
            if (withPartnerIdAndDaysBefore == null) {
                return Optional.empty();
            }
            return Optional.ofNullable(getCeilingValue(withPartnerIdAndDaysBefore, features.getHour()));
        }

        void setLifetime(OfferFeatures features, Duration lifetime) {
            TreeMap<Integer, TreeMap<Integer, Duration>> withPartnerId = mapping.computeIfAbsent(
                features.getPartnerId(), k -> new TreeMap<>());
            TreeMap<Integer, Duration> withPartnerIdAndDaysBefore = withPartnerId.computeIfAbsent(
                features.getDaysBefore(), k -> new TreeMap<>());
            if (withPartnerIdAndDaysBefore.put(features.getHour(), lifetime) == null) {
                ++size;
            }
        }

        private static <K, V> V getCeilingValue(NavigableMap<K, V> map, K key) {
            Map.Entry<K, V> entry = map.ceilingEntry(key);
            return entry != null ? entry.getValue() : null;
        }
    }

    private final ColdConfigurationProperties config;
    private volatile FeaturesToLifetimeMapping featuresToLifetimeMapping = new FeaturesToLifetimeMapping();

    public ColdService(ColdConfigurationProperties config) {
        this.config = config;
        if (config.isDeducerEnabled()) {
            Executors.newSingleThreadScheduledExecutor().scheduleAtFixedRate(
                    this::readTable, 0, config.getUpdateInterval().getSeconds(), TimeUnit.SECONDS);
        }
    }

    public Duration getOfferLifetime(Task task, Duration maxNonEmptyLifetime) {
        if (task.getOfferCount() == 0) {
            return config.getEmptyLifetime();
        }
        TSearchOffersReq request = task.getRequest();
        LocalDate checkInDate = LocalDate.parse(request.getCheckInDate());
        LocalDateTime now = LocalDateTime.now();
        OfferFeatures features = new OfferFeatures(
            request.getHotelId().getPartnerId(),
            (int)now.toLocalDate().until(checkInDate, ChronoUnit.DAYS),
            now.getHour());
        Duration lifetime = featuresToLifetimeMapping.getLifetime(features).orElse(config.getDefaultLifetime());
        if (maxNonEmptyLifetime != null && lifetime.compareTo(maxNonEmptyLifetime) > 0) {
            lifetime = maxNonEmptyLifetime;
        }
        return lifetime;
    }

    private void readTable() {
        FeaturesToLifetimeMapping newFeaturesToLifetimeMapping = new FeaturesToLifetimeMapping();
        try {
            Yt yt = YtUtils.http(config.getProxy(), config.getToken());
            YPath path = YPath.simple(config.getTablePath());
            yt.tables().read(path, YTableEntryTypes.YSON, (row) -> {
                EPartnerId partnerId = EPartnerId.valueOf(row.getOrThrow("PartnerId").stringValue());
                int daysBeforeUpperBound = row.getOrThrow("DaysBeforeUpperBound").intValue();
                int hourUpperBound = row.getOrThrow("HourUpperBound").intValue();
                Duration lifetime = Duration.ofMinutes(row.getOrThrow("Lifetime").intValue());
                newFeaturesToLifetimeMapping.setLifetime(
                    new OfferFeatures(partnerId, daysBeforeUpperBound, hourUpperBound), lifetime);
            });
        } catch (Exception ex) {
            log.error("An exception occurred during the COLD table processing", ex);
            return;
        }
        log.info("The COLD table was updated, it contains {} records", newFeaturesToLifetimeMapping.getSize());
        featuresToLifetimeMapping = newFeaturesToLifetimeMapping;
    }

}
