package ru.yandex.travel.hotels.searcher.http;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.model.PriceLosRequest;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.GetMealsResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.HotelsWithInfo;
import ru.yandex.travel.hotels.common.partners.bronevik.SearchHotelOffersResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.SalesChannel;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.searcher.http.rsp_models.BnovoResponse;
import ru.yandex.travel.hotels.searcher.http.rsp_models.BronevikResponse;
import ru.yandex.travel.hotels.searcher.http.rsp_models.DolphinResponse;
import ru.yandex.travel.hotels.searcher.http.rsp_models.ExpediaResponse;
import ru.yandex.travel.hotels.searcher.http.rsp_models.TravellineResponse;
import ru.yandex.travel.hotels.searcher.partners.ExpediaTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.RatePlanSearcher;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

@Component
@RequiredArgsConstructor
public class AdminSearcherService {

    private final BNovoClient bnovoClient;
    private final DolphinClient dolphinClient;
    private final ExpediaClient expediaClient;
    private final ExpediaTaskHandlerProperties expediaConfig;
    private final TravellineClient travellineClient;
    private final BronevikClient bronevikClient;

    private final RatePlanSearcher directRatePlanSearcher;

    private final ObjectMapper mapper = new ObjectMapper();

    public CompletableFuture<String> search(EPartnerId partnerId, String hotelId, String requestedCheckin,
                                            String requestedCheckout, String requestedOccupancy, String reqId) {
        LocalDate checkin = LocalDate.parse(requestedCheckin);
        LocalDate checkout = LocalDate.parse(requestedCheckout);
        Occupancy occupancy = Occupancy.fromString(requestedOccupancy);
        String requestId = reqId == null || reqId.isEmpty() ? UUID.randomUUID().toString() : reqId;

        switch (partnerId) {
            case PI_TRAVELLINE:
                return searchTravelline(hotelId, checkin, checkout, requestId);
            case PI_BNOVO:
                return searchBnovo(Long.parseLong(hotelId), checkin, checkout, occupancy, requestId);
            case PI_EXPEDIA:
                return searchExpedia(hotelId, checkin, checkout, occupancy, requestId);
            case PI_DOLPHIN:
                return searchDolphin(hotelId, checkin, checkout, occupancy, requestId);
            case PI_BRONEVIK:
                return searchBronevik(hotelId, checkin, checkout, occupancy, requestId);
            default:
                return CompletableFuture.completedFuture(null);
        }
    }

    private CompletableFuture<String> searchBnovo(Long hotelId, LocalDate checkin, LocalDate checkout,
                                                  Occupancy occupancy, String reqId) {
        int nights = (int) ChronoUnit.DAYS.between(checkin, checkout);
        PriceLosRequest request = PriceLosRequest.builder()
                .checkinFrom(checkin)
                .nights(nights)
                .adults(occupancy.getAdults())
                .children(occupancy.getChildren().size())
                .accounts(List.of(hotelId))
                .requestId(reqId)
                .build();

        var ratePlansFuture = bnovoClient.getRatePlans(hotelId, reqId);
        var servicesFuture = bnovoClient.getServices(hotelId, reqId);
        var roomTypesFuture = bnovoClient.getRoomTypes(hotelId, reqId);
        var hotelInfoFuture = bnovoClient.getHotelByAccountId(hotelId, reqId);
        var pricesFuture = bnovoClient.getPrices(request);

        return CompletableFuture.allOf(ratePlansFuture, servicesFuture, roomTypesFuture, hotelInfoFuture, pricesFuture)
                .thenApply(ignored -> convertToJson(BnovoResponse.builder()
                        .ratePlans(ratePlansFuture.join())
                        .services(servicesFuture.join())
                        .roomTypes(roomTypesFuture.join())
                        .hotelInfo(hotelInfoFuture.join())
                        .prices(pricesFuture.join())
                        .build()));
    }

    private CompletableFuture<String> searchDolphin(String hotelId, LocalDate checkin, LocalDate checkout,
                                                    Occupancy occupancy, String reqId) {
        var responseFuture = dolphinClient.searchCheckins(checkin, checkout, occupancy, List.of(hotelId), reqId);
        var roomsFuture = dolphinClient.getRooms();
        var roomCategoriesFuture = dolphinClient.getRoomCategories();
        var pansionsFuture = dolphinClient.getPansionMap();

        return CompletableFuture.allOf(responseFuture, roomsFuture, roomCategoriesFuture, pansionsFuture)
                .thenApply(ignored -> convertToJson(DolphinResponse.builder()
                        .searchResponse(responseFuture.join())
                        .rooms(roomsFuture.join())
                        .roomCategories(roomCategoriesFuture.join())
                        .pansions(pansionsFuture.join())
                        .build()));
    }

    private CompletableFuture<String> searchExpedia(String hotelId, LocalDate checkin, LocalDate checkout,
                                                    Occupancy occupancy, String reqId) {
        var regularFuture = expediaClient.findAvailabilities(List.of(hotelId), checkin, checkout,
                occupancy.toExpediaString(), expediaConfig.getRequestCurrency(), expediaConfig.isIncludeFencedRates(), expediaConfig.getSearchIpAddress(),
                ProtoUtils.randomId(), reqId, SalesChannel.CACHE);
        var mobileFuture = expediaConfig.isSearchMobileDeals()
                ? expediaClient.findAvailabilities(List.of(hotelId), checkin, checkout, occupancy.toExpediaString(),
                expediaConfig.getRequestCurrency(), expediaConfig.isIncludeFencedRates(), expediaConfig.getSearchIpAddress(),
                ProtoUtils.randomId(), reqId, SalesChannel.MOBILE_WEB)
                : CompletableFuture.completedFuture(new HashMap<String, PropertyAvailability>());

        return CompletableFuture.allOf(regularFuture, mobileFuture)
                .thenApply(ignored -> convertToJson(ExpediaResponse.builder()
                        .regular(regularFuture.join())
                        .mobileWeb(mobileFuture.join())
                        .build()));
    }

    private CompletableFuture<String> searchTravelline(String hotelId, LocalDate checkin, LocalDate checkout, String reqId) {
        CompletableFuture<HotelInfo> hotelInfoFuture = travellineClient.getHotelInfo(hotelId, reqId);
        CompletableFuture<HotelOfferAvailability> hotelOfferAvailabilityFuture = travellineClient.findOfferAvailability(hotelId, checkin, checkout, reqId);

        return CompletableFuture.allOf(hotelInfoFuture, hotelOfferAvailabilityFuture)
                .thenApply(x -> convertToJson(TravellineResponse.builder()
                        .hotelInfo(hotelInfoFuture.join())
                        .offerAvailability(hotelOfferAvailabilityFuture.join())
                        .build()));
    }

    private CompletableFuture<String> searchBronevik(String hotelId, LocalDate checkin, LocalDate checkout, Occupancy occupancy, String reqId) {

        List<Integer> hotelIds = List.of(Integer.parseInt(hotelId));
        CompletableFuture<SearchHotelOffersResponse> searchHotelOffersResponseFuture = bronevikClient.searchHotelOffers(
                    occupancy.getAdults(),
                    occupancy.getChildren(),
                    hotelIds,
                    checkin.toString(),
                    checkout.toString(),
                    Utils.CURRENCY_NAMES.get(ECurrency.C_RUB),
                    reqId);
        CompletableFuture<HotelsWithInfo> getHotelInfoResponseFuture  = bronevikClient.getHotelsInfo(hotelIds, reqId);
        CompletableFuture<GetMealsResponse> getMealsResponseCompletableFuture = bronevikClient.getMeals(reqId);

        return CompletableFuture.allOf(
                searchHotelOffersResponseFuture,
                getHotelInfoResponseFuture,
                getMealsResponseCompletableFuture).thenApply(x -> convertToJson(BronevikResponse.builder()
                    .hotelInfoResponse(getHotelInfoResponseFuture.join())
                    .hotelOffersResponse(searchHotelOffersResponseFuture.join())
                    .mealsResponse(getMealsResponseCompletableFuture.join())
                    .build()));
    }

    private String convertToJson(Object obj) {
        try {
            return mapper.writeValueAsString(obj);
        } catch (JsonProcessingException ex) {
            throw new RuntimeException(ex);
        }
    }
}
