package ru.yandex.travel.hotels.searcher.http;

import java.time.LocalDate;
import java.util.concurrent.CompletableFuture;
import java.util.regex.Pattern;

import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.travel.hotels.proto.EPartnerId;

@RestController
@RequestMapping("/admin")
@RequiredArgsConstructor
public class SearcherAdminController {

    private final AdminSearcherService adminSearcherService;

    private static final Pattern DATE_PATTERN = Pattern.compile("\\d\\d\\d\\d-\\d\\d-\\d\\d");
    private static final Pattern OCCUPANCY_PATTERN = Pattern.compile("\\d{1,2}(?:-\\d{1,2}(?:,\\d{1,2})*)?");

    @ExceptionHandler(AdminSearcherBadRequestException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleAdminSearcherBadRequestException(AdminSearcherBadRequestException e) {
        return ResponseEntity.status(HttpStatus.BAD_REQUEST).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(RuntimeException.class)
    @ResponseStatus(HttpStatus.INTERNAL_SERVER_ERROR)
    public ResponseEntity<String> handleRuntimeException(RuntimeException e) {
        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @RequestMapping(value = {"/search"}, method = RequestMethod.GET, produces = "application/json")
    public CompletableFuture<String> searchPartnerOffers(
            @RequestParam(value = "originalId") String originalHotelId,
            @RequestParam(value = "partnerId") EPartnerId partnerId,
            @RequestParam(value = "checkin", required = false) String requestedCheckin,
            @RequestParam(value = "checkout", required = false) String requestedCheckout,
            @RequestParam(value = "occupancy", required = false, defaultValue = "2") String occupancy,
            @RequestParam(value = "reqId", required = false) String reqId
    ) throws AdminSearcherBadRequestException {
        validateRequest(originalHotelId, requestedCheckin, requestedCheckout, occupancy);
        String checkin = requestedCheckin == null ? LocalDate.now().toString() : requestedCheckin;
        String checkout = requestedCheckout == null ? LocalDate.parse(checkin).plusDays(1).toString() : requestedCheckout;

        return adminSearcherService.search(partnerId, originalHotelId,
                checkin, checkout, occupancy, reqId);
    }

    private void validateRequest(String originalHotelId, String checkin, String checkout, String occupancy) {
        if (originalHotelId.isEmpty()) {
            throw new AdminSearcherBadRequestException("originalHotelId is required");
        }

        if (checkin != null) {
            if (checkin.isEmpty()) {
                throw new AdminSearcherBadRequestException("checkin is required");
            }

            if (!DATE_PATTERN.matcher(checkin).matches()) {
                throw new AdminSearcherBadRequestException("checkin is invalid");
            }
        }

        if (checkout != null) {
            if (checkout.isEmpty()) {
                throw new AdminSearcherBadRequestException("checkout is required");
            }

            if (!DATE_PATTERN.matcher(checkout).matches()) {
                throw new AdminSearcherBadRequestException("checkout is invalid");
            }
        }

        if (occupancy.isEmpty()) {
            throw new AdminSearcherBadRequestException("occupancy is required");
        } else if (!OCCUPANCY_PATTERN.matcher(occupancy).matches()) {
            throw new AdminSearcherBadRequestException("occupancy is invalid");
        }
    }
}
