package ru.yandex.travel.hotels.searcher.management;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import ru.yandex.travel.grpc.GrpcServerRunner;
import ru.yandex.travel.hotels.searcher.PartnerDispatcher;
import ru.yandex.travel.hotels.searcher.SearcherApplicationProperties;
import ru.yandex.travel.hotels.searcher.partners.PartnerTaskHandler;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

@Component
public class ShutdownManager implements ApplicationContextAware, HealthIndicator {
    private final PartnerDispatcher partnerDispatcher;
    private final ScheduledExecutorService executor;
    private final GrpcServerRunner serverRunner;
    private final AtomicBoolean shutdownInitiated;
    private final Duration timeout;
    private final Logger logger;
    private Long shutdownTimestamp;
    private ApplicationContext context;

    public ShutdownManager(PartnerDispatcher partnerDispatcher, GrpcServerRunner serverRunner,
                           SearcherApplicationProperties config) {
        this.partnerDispatcher = partnerDispatcher;
        this.executor = new ScheduledThreadPoolExecutor(1,
                new ThreadFactoryBuilder().setNameFormat("shutdown-thread-%d").build());
        ((ScheduledThreadPoolExecutor) this.executor).setMaximumPoolSize(1);
        this.serverRunner = serverRunner;
        this.timeout = config.getShutdownSequenceTimeout();
        shutdownInitiated = new AtomicBoolean(false);
        this.logger = LoggerFactory.getLogger(this.getClass());
    }

    public void initiateShutdown() {
        if (!shutdownInitiated.getAndSet(true)) {
            logger.info("Shutdown initiated, health check flag switched off");
            shutdownTimestamp = System.currentTimeMillis();
            executor.schedule(this::runShutdownSequence, timeout.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    private void runShutdownSequence() {
        logger.info("Will stop gRPC server");
        try {
            serverRunner.stop();
        } catch (Exception e) {
            logger.error("Unable to shutdown gRPC server", e);
        }
        logger.info("Shutting down partner handlers");
        partnerDispatcher.forEach(PartnerTaskHandler::shutdown);
        logger.info("Waiting for partner handlers to finish running tasks");
        partnerDispatcher.forEach(partnerTaskHandler -> {
            try {
                partnerTaskHandler.awaitTermination();
            } catch (InterruptedException ex) {
                logger.error("Unable to terminate handler {}", partnerTaskHandler);
            }
        });
        logger.info("Shutting down application context");
        ((ConfigurableApplicationContext) context).close();
        logger.info("Dying as a phoenix to burn and be reborn");
        System.exit(0);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }


    @Override
    public Health health() {
        if (shutdownInitiated.get()) {
            return Health.down().withDetail("ShutdownInitiated", shutdownTimestamp).build();
        } else {
            return Health.up().build();
        }
    }
}
