package ru.yandex.travel.hotels.searcher.partners;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.bnovo.BNovoClient;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AccommodationType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.AgeGroup;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Hotel;
import ru.yandex.travel.hotels.common.partners.bnovo.model.HotelStayMap;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Offer;
import ru.yandex.travel.hotels.common.partners.bnovo.model.PriceLosRequest;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.bnovo.model.RoomType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Service;
import ru.yandex.travel.hotels.common.partners.bnovo.model.ServiceType;
import ru.yandex.travel.hotels.common.partners.bnovo.model.Stay;
import ru.yandex.travel.hotels.common.partners.bnovo.model.StayMap;
import ru.yandex.travel.hotels.common.partners.bnovo.model.WarrantyType;
import ru.yandex.travel.hotels.common.partners.bnovo.utils.BNovoPansionHelper;
import ru.yandex.travel.hotels.common.partners.bnovo.utils.BNovoRefundRulesBuilder;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ESearchWarningCode;
import ru.yandex.travel.hotels.proto.TBNovoOffer;
import ru.yandex.travel.hotels.proto.TBNovoUIDMapping;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPartnerSpecificOfferData;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.proto.TRefundRule;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.TravelTokenService;
import ru.yandex.travel.hotels.searcher.services.cache.Actualizable;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.HotelInfoSearcher;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.RatePlanSearcher;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.RoomTypeSearcher;
import ru.yandex.travel.hotels.searcher.services.cache.bnovo.ServicesSearcher;

@PartnerBean(EPartnerId.PI_BNOVO)
@EnableConfigurationProperties(BNovoTaskHandlerProperties.class)
public class BNovoTaskHandler extends AbstractPartnerTaskHandler<BNovoTaskHandlerProperties> {
    private static final int MAX_OFFER_LENGTH = 150;

    private final BNovoClient client;
    private final RatePlanSearcher ratePlanSearcher;
    private final RoomTypeSearcher roomTypeSearcher;
    private final HotelInfoSearcher hotelInfoSearcher;
    private final TravelTokenService travelTokenService;
    private final ServicesSearcher servicesSearcher;

    private final Counter forbiddenOffersCounter = Metrics.counter("searcher.partners.bnovo.droppedOffers", "reason",
            "NegativePrice");
    private final Counter diabledOffersCounter = Metrics.counter("searcher.partners.bnovo.droppedOffers", "reason",
            "Disabled");
    private final Counter invalidCancellationOffersCounter = Metrics.counter("searcher.partners.bnovo.droppedOffers",
            "reason",
            "InvalidCancellation");
    private final Counter unsupportedAccommodationTypeBedCounter = Metrics.counter("searcher.partners.bnovo" +
                    ".droppedOffers", "reason",
            "bed");
    private final Counter unsupportedAccommodationTypeServiceCounter = Metrics.counter("searcher.partners.bnovo" +
                    ".droppedOffers", "reason",
            "service");
    private final Counter emptyPriceBreakdownCounter = Metrics.counter("searcher.partners.bnovo" +
                    ".droppedOffers", "reason",
            "emptyPriceBreakdown");

    private final Counter unfitChildrenOffersCounter = Metrics.counter("searcher.partners.bnovo.droppedOffers",
            "reason",
            "UnfitChildren");

    BNovoTaskHandler(BNovoTaskHandlerProperties config, BNovoClient client, RatePlanSearcher ratePlanSearcher,
                     RoomTypeSearcher roomTypeSearcher, HotelInfoSearcher hotelInfoSearcher,
                     ServicesSearcher servicesSearcher,
                     TravelTokenService travelTokenService) {
        super(config);
        this.client = client;
        this.ratePlanSearcher = ratePlanSearcher;
        this.roomTypeSearcher = roomTypeSearcher;
        this.hotelInfoSearcher = hotelInfoSearcher;
        this.travelTokenService = travelTokenService;
        this.servicesSearcher = servicesSearcher;
    }

    @Override
    CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        List<Long> hotelIds =
                tasks.stream().map(t -> Long.valueOf(t.getRequest().getHotelId().getOriginalId())).distinct().collect(Collectors.toList());
        Map<String, List<Task>> taskByHotel = mapTasksByOriginalId(tasks);

        CompletableFuture<HotelStayMap> pricesFuture = requestPrices(groupingKey,
                tasks.get(0).getCallContext(),
                requestId,
                hotelIds);

        Map<String, CompletableFuture<Actualizable<Map<Long, RatePlan>>>> ratePlanFutures =
                hotelIds.stream().collect(Collectors.toMap(
                        String::valueOf,
                        id -> ratePlanSearcher.getRatePlansForHotel(id,
                                taskByHotel.get(String.valueOf(id)).get(0).getCallContext(),
                                requestId)
                ));

        Map<String, CompletableFuture<Map<Long, Service>>> serviceFutures =
                hotelIds.stream().collect(Collectors.toMap(
                        String::valueOf,
                        id -> servicesSearcher.getServicesForHotel(id,
                                taskByHotel.get(String.valueOf(id)).get(0).getCallContext(),
                                requestId)

                ));
        Map<String, CompletableFuture<Actualizable<Map<Long, RoomType>>>> roomTypeFutures =
                hotelIds.stream().collect(Collectors.toMap(
                        String::valueOf,
                        id -> roomTypeSearcher.getRoomTypesForHotel(id,
                                taskByHotel.get(String.valueOf(id)).get(0).getCallContext(),
                                requestId)
                ));
        Map<String, CompletableFuture<Hotel>> hotelInfoFutures =
                hotelIds.stream().collect(Collectors.toMap(
                        String::valueOf,
                        id -> hotelInfoSearcher.getHotel(id,
                                taskByHotel.get(String.valueOf(id)).get(0).getCallContext(),
                                requestId)
                ));


        List<CompletableFuture<?>> futuresToWait =
                new ArrayList<>(ratePlanFutures.size() + serviceFutures.size() + roomTypeFutures.size() + hotelInfoFutures.size() + 1);
        futuresToWait.add(pricesFuture);
        futuresToWait.addAll(ratePlanFutures.values());
        futuresToWait.addAll(roomTypeFutures.values());
        futuresToWait.addAll(hotelInfoFutures.values());
        futuresToWait.addAll(serviceFutures.values());

        return CompletableFuture.allOf(futuresToWait.toArray(new CompletableFuture[0])).thenCompose(ignored ->
        {
            Map<String, Actualizable<Map<Long, RatePlan>>> ratePlans = ratePlanFutures.entrySet().stream().collect(
                    Collectors.toMap(Map.Entry::getKey, e -> e.getValue().join()));
            Map<String, Actualizable<Map<Long, RoomType>>> roomTypes = roomTypeFutures.entrySet().stream().collect(
                    Collectors.toMap(Map.Entry::getKey, e -> e.getValue().join()));
            Map<String, Hotel> hotelInfos = hotelInfoFutures.entrySet().stream().collect(
                    Collectors.toMap(Map.Entry::getKey, e -> e.getValue().join()));
            Map<String, Map<Long, Service>> services = serviceFutures.entrySet().stream().collect(
                    Collectors.toMap(Map.Entry::getKey, e -> e.getValue().join()));
            return processResults(pricesFuture.join(), ratePlans, roomTypes, hotelInfos, services, taskByHotel,
                    requestId);
        });
    }

    private CompletableFuture<HotelStayMap> requestPrices(Task.GroupingKey groupingKey,
                                                          CallContext callContext,
                                                          String requestId,
                                                          List<Long> hotelIds) {
        LocalDate checkin = LocalDate.parse(groupingKey.getCheckInDate());
        LocalDate checkout = LocalDate.parse(groupingKey.getCheckOutDate());
        int nights = (int) ChronoUnit.DAYS.between(checkin, checkout);
        PriceLosRequest request = PriceLosRequest.builder()
                .checkinFrom(checkin)
                .nights(nights)
                .adults(groupingKey.getOccupancy().getAdults())
                .children(groupingKey.getOccupancy().getChildren().size())
                .accounts(hotelIds)
                .requestId(requestId)
                .build();
        return client.withCallContext(callContext).getPrices(request);
    }

    private CompletableFuture<Void> processResults(HotelStayMap hotelStayMap,
                                                   Map<String, Actualizable<Map<Long, RatePlan>>> ratePlans,
                                                   Map<String, Actualizable<Map<Long, RoomType>>> roomTypes,
                                                   Map<String, Hotel> hotelInfos,
                                                   Map<String, Map<Long, Service>> services,
                                                   Map<String, List<Task>> tasks,
                                                   String httpRequestId) {
        for (var entry : hotelStayMap.entrySet()) {
            if (entry.getValue().size() == 0) {
                logger.info("Empty StayMap for hotel " + entry.getKey());
                return CompletableFuture.completedFuture(null);
            }
            if (entry.getValue().size() > 1) {
                return CompletableFuture.failedFuture(new IllegalStateException("Hotel " + entry.getKey() + " has " +
                        "unexpected number of stay dates"));
            }
        }
        List<CompletableFuture<Void>> futuresToWait = new ArrayList<>(hotelStayMap.entrySet().size());
        for (var entry : hotelStayMap.entrySet()) {
            String hotelId = entry.getKey();
            StayMap stayMap = entry.getValue();
            Stay stay = stayMap.values().iterator().next();
            futuresToWait.add(
                    processResult(
                            hotelInfos.get(hotelId), stay, ratePlans, roomTypes, services,
                            tasks.get(hotelId), httpRequestId
                    )
            );
        }
        return CompletableFuture.allOf(futuresToWait.toArray(new CompletableFuture[0]));
    }


    private CompletableFuture<Void> processResult(Hotel hotel, Stay stay,
                                                  Map<String, Actualizable<Map<Long, RatePlan>>> ratePlans,
                                                  Map<String, Actualizable<Map<Long, RoomType>>> roomTypes,
                                                  Map<String, Map<Long, Service>> services,
                                                  List<Task> tasks,
                                                  String httpRequestId) {
        String hotelId = String.valueOf(hotel.getId());
        return CompletableFuture.allOf(stay.getRates().stream().map(offer -> {
            if (offer.getPrice().intValue() == -1) {
                forbiddenOffersCounter.increment();
                tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_FORBIDDEN_OFFERS));
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture<RatePlan> ratePlanFuture =
                    ratePlans.get(hotelId).getAndFetchActual(rpm -> rpm.get(offer.getPlanId()), httpRequestId);
            CompletableFuture<RoomType> baseRoomTypeFuture = getRecursiveRoomTypeInfo(roomTypes.get(hotelId),
                    offer.getRoomtypeId(), httpRequestId);
            final CompletableFuture<RoomType> actualRoomTypeFuture =
                    roomTypes.get(hotelId).getAndFetchActual(rtm -> rtm.get(offer.getRoomtypeId()), httpRequestId);

            return CompletableFuture.allOf(ratePlanFuture, baseRoomTypeFuture, actualRoomTypeFuture).thenAccept(ignored -> {
                RatePlan ratePlan = ratePlanFuture.join();
                RoomType baseRoomType = baseRoomTypeFuture.join();
                RoomType actualRoomType = actualRoomTypeFuture.join();
                Stay patchedStay = stay.toBuilder().clearRates().rate(offer).build();
                processOffer(tasks, hotel, ratePlan, baseRoomType, actualRoomType, patchedStay, services,
                        resolveAgeGroupsForRoomType(actualRoomType, hotel.getChildrenAges()));
            });
        }).toArray(CompletableFuture[]::new));

    }

    private Map<AgeGroup, Integer> resolveAgeGroupsForRoomType(RoomType roomType, Map<String, AgeGroup> ageGroups) {
        if (ageGroups.isEmpty() || roomType.getExtraArray() == null || roomType.getExtraArray().getChildrenAges() == null || roomType.getExtraArray().getChildrenAges().isEmpty()) {
            return null;
        }
        return roomType.getExtraArray().getChildrenAges().entrySet().stream()
                .map(entry -> Map.entry(ageGroups.get(entry.getKey()), entry.getValue()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    private CompletableFuture<RoomType> getRecursiveRoomTypeInfo(Actualizable<Map<Long, RoomType>> roomTypeMap,
                                                                 long id,
                                                                 String httpRequestId) {
        return roomTypeMap.getAndFetchActual(rtm -> rtm.get(id), httpRequestId).thenCompose(rt -> {
            if (rt.getParentId() == 0) {
                return CompletableFuture.completedFuture(rt);
            } else {
                return getRecursiveRoomTypeInfo(roomTypeMap, rt.getParentId(), httpRequestId);
            }
        });
    }

    private void processOffer(List<Task> tasks, Hotel hotel, RatePlan ratePlan, RoomType baseRoomType,
                              RoomType actualRoomType, Stay stay, Map<String, Map<Long, Service>> services,
                              Map<AgeGroup, Integer> ageGroupsWithCount) {
        Preconditions.checkArgument(stay.getRates().size() == 1, "Unexpected amount of offers");
        Preconditions.checkArgument(stay.getRates().get(0).getPrice().compareTo(BigDecimal.ZERO) > 0, "Unexpected " +
                "offer price");
        Occupancy occupancy = tasks.get(0).getOccupancy();

        if (baseRoomType.getAccommodationType() == AccommodationType.BED && !config.isAllowBeds()) {
            unsupportedAccommodationTypeBedCounter.increment();
            logger.info("RoomType {} of hotel {} has unsupported accommodation type {}", baseRoomType.getId(),
                    hotel.getId(), baseRoomType.getAccommodationType());
            tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_HOSTEL_BEDS));
            return;
        }
        if (baseRoomType.getAccommodationType() == AccommodationType.BED && baseRoomType.getAdults() > 1) {
            unsupportedAccommodationTypeBedCounter.increment();
            logger.info("RoomType {} of hotel {} has accommodation type BED but claims to fit more than 1 person",
                    baseRoomType.getId(), hotel.getId());
            tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_HOSTEL_BEDS));
            return;
        }
        if (baseRoomType.getAccommodationType() == AccommodationType.SERVICE) {
            unsupportedAccommodationTypeServiceCounter.increment();
            logger.info("RoomType {} of hotel {} has accommodation type SERVICE, i.e. is not an accommodation offer",
                    baseRoomType.getId(), hotel.getId());
            tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_SERVICE_OFFERS));
            return;
        }
        if (!ratePlan.isEnabledYandex()) {
            logger.info("Rate plan {} for hotel {} is disabled", ratePlan.getId(), hotel.getId());
            tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_DISABLED_RATE_PLAN));
            diabledOffersCounter.increment();
            return;
        }
        if (ageGroupsWithCount != null) {
            int extraAdultSlots = actualRoomType.getAdults() - occupancy.getAdults();
            var ageCount = occupancy.getChildren().stream().collect(Collectors.groupingBy(Function.identity(),
                    Collectors.counting()));
            int unfitChildren = 0;
            for (var e : ageCount.entrySet()) {
                int age = e.getKey();
                int count = e.getValue().intValue();
                var foundGroup =
                        ageGroupsWithCount.entrySet().stream().filter(entry -> age >= entry.getKey().getMinAge() && age <= entry.getKey().getMaxAge()).findFirst();
                if (foundGroup.isEmpty()) {
                    unfitChildren += count;
                } else {
                    if (foundGroup.get().getValue() < count) {
                        unfitChildren += (count - foundGroup.get().getValue());
                    }
                }
            }
            if (unfitChildren > extraAdultSlots) {
                logger.info("{} children do not fit into appropriate age groups of adult spots, skipping offers",
                        unfitChildren - extraAdultSlots);
                tasks.forEach(t -> t.onWarning(ESearchWarningCode.SW_BN_DROPPED_UNFIT_CHILDREN));
                unfitChildrenOffersCounter.increment();
                return;
            }
        }

        tasks.forEach(task -> {
            String offerId = ProtoUtils.randomId();
            String capacity = Capacity.fromOccupancy(task.getOccupancy()).toString();
            Offer offer = stay.getRates().get(0);
            RefundRules refundRules;
            try {
                refundRules = BNovoRefundRulesBuilder.build(ratePlan, offer,
                        hotel.getCheckinInstantForDate(stay.getCheckin()), Instant.now(), "RUB");
            } catch (IllegalArgumentException ex) {
                logger.warn("Incorrect refund rules", ex);
                task.onWarning(ESearchWarningCode.SW_BN_DROPPED_INVALID_CANCELLATION_PENALTIES);
                invalidCancellationOffersCounter.increment();
                return;
            }
            if (offer.getPricesByDates().isEmpty()) {
                logger.warn("Empty price by date list for hotel {}, room {}, rate plan {}",
                        hotel.getId(), baseRoomType.getId(), ratePlan.getId());
                emptyPriceBreakdownCounter.increment();
                task.onError(TError.newBuilder()
                        .setCode(EErrorCode.EC_GENERAL_ERROR)
                        .setMessage("Empty price by date list")
                        .build());
                return;
            }
            var bNovoOfferBuilder = TBNovoOffer.newBuilder().
                    setPostPayAllowed(ratePlan.getWarrantyType() == WarrantyType.NO_WARRANTY).
                    setBNovoStay(ProtoUtils.toTJson(stay));
            if (config.isRemapIds()) {
                bNovoOfferBuilder.setUIDMapping(TBNovoUIDMapping.newBuilder()
                        .setUID(hotel.getUid())
                        .setOriginalId(String.valueOf(hotel.getId())));
            }
            var hotelServices = services.getOrDefault(String.valueOf(hotel.getId()), Collections.emptyMap());

            String originalRoomId = String.valueOf(baseRoomType.getId());
            EPansionType pansionType = BNovoPansionHelper.getPansionType(ratePlan, hotelServices);
            TPriceWithDetails offerPrice = TPriceWithDetails.newBuilder()
                    .setCurrency(ECurrency.C_RUB)
                    .setAmount(offer.getPrice().intValue())
                    .build();
            List<TRefundRule> protoRefundRules = mapToProtoRefundRules(refundRules);
            var geoData = hotel.getGeoData();
            TCoordinates coordinates = TCoordinates.newBuilder()
                    .setLatitude(geoData.getLatitude())
                    .setLongitude(geoData.getLongitude())
                    .build();
            String travelTokenString = travelTokenService.storeTravelTokenAndGetItsString(offerId,
                    EPartnerId.PI_BNOVO, task,
                    originalRoomId, pansionType,
                    refundRules.isFullyRefundable(), offerPrice, coordinates, protoRefundRules,
                    TOfferData.newBuilder().setBNovoOffer(bNovoOfferBuilder));

            StringBuilder fullNameBuilder = new StringBuilder(baseRoomType.getDefaultName());
            ratePlan.getAdditionalServicesIds().stream()
                    .map(hotelServices::get)
                    .filter(s -> s != null && s.getType() != ServiceType.BOARD)
                    .map(Service::getDefaultName)
                    .sorted()
                    .forEach(name -> {
                        String ext = " • " + name;
                        if (fullNameBuilder.length() + ext.length() <= MAX_OFFER_LENGTH) {
                            fullNameBuilder.append(ext);
                        }
                    });
            TOffer.Builder taskOfferBuilder = TOffer.newBuilder()
                    .setId(offerId)
                    .setDisplayedTitle(StringValue.of(fullNameBuilder.toString()))
                    .setCapacity(capacity)
                    .setSingleRoomCapacity(capacity)
                    .setRoomCount(1)
                    .setOriginalRoomId(originalRoomId)
                    .setPansion(pansionType)
                    .setFreeCancellation(BoolValue.of(refundRules.isFullyRefundable()))
                    .addAllRefundRule(mapToProtoRefundRules(refundRules))
                    .setOperatorId(EOperatorId.OI_BNOVO)
                    .setPrice(offerPrice)
                    .setLandingInfo(TOfferLandingInfo.newBuilder()
                            .setLandingTravelToken(travelTokenString)
                            .build())
                    .setPartnerSpecificData(TPartnerSpecificOfferData.newBuilder()
                            .setBNovoData(TPartnerSpecificOfferData.TBNovoData.newBuilder()
                                    .setAccountId(hotel.getId())
                                    .setRatePlanId(ratePlan.getId())
                                    .build())
                            .build());

            onOffer(task, taskOfferBuilder);
        });
    }


}
