package ru.yandex.travel.hotels.searcher.partners;

import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.Streams;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.web.util.UriBuilder;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.pansions.PansionUnifier;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.partners.booking.BookingClient;
import ru.yandex.travel.hotels.common.partners.booking.BookingUserPlatform;
import ru.yandex.travel.hotels.common.partners.booking.model.Block;
import ru.yandex.travel.hotels.common.partners.booking.model.BlockAvailability;
import ru.yandex.travel.hotels.common.partners.booking.model.Hotel;
import ru.yandex.travel.hotels.common.partners.booking.model.IncrementalPrice;
import ru.yandex.travel.hotels.common.partners.booking.model.Price;
import ru.yandex.travel.hotels.common.partners.booking.model.Result;
import ru.yandex.travel.hotels.common.partners.booking.utils.BookingRefundRulesBuilder;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TOfferRestrictions;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.cache.booking.hotels.BookingHotelSearcher;

@PartnerBean(EPartnerId.PI_BOOKING)
@EnableConfigurationProperties(BookingPartnerTaskHandlerProperties.class)
public class BookingPartnerTaskHandler extends AbstractPartnerTaskHandler<BookingPartnerTaskHandlerProperties> {
    private static class PricePair {
        private String currency;
        private double price;

        public PricePair(String currency, double price) {
            this.currency = currency;
            this.price = price;
        }

        public String getCurrency() {
            return currency;
        }

        public double getPrice() {
            return price;
        }
    }


    private final Counter invalidMainCurrencyCounter;
    private final Counter invalidIncrementalCurrencyCounter;
    private final Counter invalidRefundRulesCounter;
    private final Counter hotelIdNotFoundCounter;
    private final BookingClient bookingClient;
    private final BookingHotelSearcher hotelSearcher;

    public BookingPartnerTaskHandler(BookingPartnerTaskHandlerProperties config, BookingClient bookingClient,
                                     BookingHotelSearcher hotelSearcher) {
        super(config);
        this.bookingClient = bookingClient;
        this.hotelSearcher = hotelSearcher;
        invalidMainCurrencyCounter = Metrics.counter("searcher.partners.booking.invalidMainCurrency");
        invalidIncrementalCurrencyCounter = Metrics.counter("searcher.partners.booking.invalidIncrementalCurrency");
        invalidRefundRulesCounter = Metrics.counter("searcher.partners.booking.invalidRefundRulesCounter");
        hotelIdNotFoundCounter = Metrics.counter("searcher.partners.booking.hotelIdNotFound");
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        Set<String> hotelIds = tasks.stream().map(task -> task.getRequest().getHotelId().getOriginalId()).collect(Collectors.toSet());
        CompletableFuture<Map<String, Hotel>> hotelsFuture = hotelSearcher.getHotels(hotelIds);
        var userPlatforms = List.of(BookingUserPlatform.NOT_SPECIFIED, BookingUserPlatform.MOBILE);
        var responses = userPlatforms.stream()
                .map(userPlatform -> {
                    return bookingClient.getBlockAvailability(
                            groupingKey.getCheckInDate(),
                            groupingKey.getCheckOutDate(),
                            tasks.stream().map(task -> task.getRequest().getHotelId().getOriginalId()).collect(Collectors.toSet()),
                            groupingKey.getOccupancy().toBookingString(),
                            groupingKey.getCurrency(),
                            String.format("%s-%s", requestId, userPlatform.name().toLowerCase(Locale.ROOT)),
                            userPlatform);
                }).collect(Collectors.toList());
        return CompletableFuture.allOf(hotelsFuture, responses.get(0), responses.get(1))
                .thenApply(x -> {
                    Map<String, Hotel> hotels = hotelsFuture.join();
                    Streams.forEachPair(userPlatforms.stream(), responses.stream(), (userPlatform, blockAvailabilityResponseFuture) -> {
                        var blockAvailabilityResponse = blockAvailabilityResponseFuture.join();
                        processResponse(requestId, tasks, blockAvailabilityResponse, hotels, userPlatform);
                    });
                    return x;
                });
    }

    private PricePair getPriceFromBookingPrice(Price price) {
        return new PricePair(price.getCurrency(), price.getPrice());
    }

    private void processResponse(String requestId, List<Task> tasks, BlockAvailability availability,
                                 Map<String, Hotel> hotels, BookingUserPlatform userPlatform) {
        Set<String> foundOriginalIds = new HashSet<>();
        Map<String, List<Task>> tasksByOriginalId = mapTasksByOriginalId(tasks);
        List<Result> results = availability.getResult();
        for (Result result : results) {
            String hotelId = result.getHotelId();
            foundOriginalIds.add(hotelId);
            List<Task> taskList = tasksByOriginalId.get(hotelId);
            if (taskList == null) {
                logger.warn("Req {}, Unknown hotel_id {}", requestId, hotelId);
                continue;
            }
            taskList.forEach(task -> {
                try {
                    result.getBlock().forEach(block -> {
                        var dealTagging = block.getDealTagging();
                        // добавление мобильных предложений тогда и только тогда, когда они участвуют в mobile_rates от букинга
                        if (userPlatform == BookingUserPlatform.MOBILE && (dealTagging == null || !"Mobile".equals(dealTagging.getDealName()))) {
                            return;
                        }
                        String blockId = block.getBlockId();
                        String roomId = block.getRoomId();
                        String expectedCurrency = Utils.CURRENCY_NAMES.get(task.getRequest().getCurrency());
                        Capacity singleRoomCapacity = Capacity.fromRule(
                                block.getMaxOccupancy(),
                                block.getMaxChildrenFree(),
                                block.getMaxChildrenFreeAge());
                        int neededRoomCount = singleRoomCapacity.calculateRoomCount(task.getOccupancy());
                        if (neededRoomCount == 0) {
                            // By default we offer one room, even if its capacity doesn't match requested occupancy
                            neededRoomCount = 1;
                        }
                        PricePair price = null;
                        if (neededRoomCount > 1) {
                            List<IncrementalPrice> priceList = block.getIncrementalPrice();
                            if (neededRoomCount <= priceList.size()) {
                                price = getPriceFromBookingPrice(priceList.get(neededRoomCount - 1));
                                if (!price.getCurrency().equals(expectedCurrency)) {
                                    invalidIncrementalCurrencyCounter.increment();
                                    price = null;// Incremental price has wrong currency sometimes
                                }
                            }
                        }
                        if (price == null) {
                            neededRoomCount = 1;
                            price = getPriceFromBookingPrice(block.getMinPrice());
                            if (!price.getCurrency().equals(expectedCurrency)) {
                                price = getPriceFromBookingPrice(block.getMinPrice().getOtherCurrency());
                            }
                        }
                        if (!price.getCurrency().equals(expectedCurrency)) {
                            invalidMainCurrencyCounter.increment();
                            throw new RuntimeException("Expected currency is " + expectedCurrency + ", but got " + price.getCurrency());
                        }
                        if (!validatePrice(task, price.getPrice())) {
                            return;
                        }
                        String title = block.getName();
                        if (neededRoomCount > 1) {
                            title = String.format("%s × %s", neededRoomCount, title);
                        }
                        boolean fullBoard = block.isFullBoard();
                        boolean halfBoard = block.isHalfBoard();
                        boolean breakfastIncluded = block.isBreakfastIncluded();
                        boolean lunchIncluded = block.isLunchIncluded();
                        boolean dinnerIncluded = block.isDinnerIncluded();
                        boolean allInclusive = block.isAllInclusive();
                        RefundRules refundRules = null;
                        try {
                            refundRules = BookingRefundRulesBuilder.build(block, result.getCheckin().atStartOfDay(), result.getCheckout().atStartOfDay(), price.getPrice());
                        } catch (Exception e) {
                            logger.warn("Req {}: Unable to calculate refund rules for block {}. Empty rules will be used instead", requestId, blockId, e);
                            invalidRefundRulesCounter.increment();
                        }
                        EPansionType pansion = PansionUnifier.get(
                                allInclusive, fullBoard, halfBoard, breakfastIncluded, lunchIncluded, dinnerIncluded);
                        TOffer.Builder o = TOffer.newBuilder()
                                .setOperatorId(EOperatorId.OI_BOOKING)
                                .setCapacity(singleRoomCapacity.multiply(neededRoomCount).toString())
                                .setSingleRoomCapacity(singleRoomCapacity.toString())
                                .setRoomCount(neededRoomCount)
                                .setExternalId(blockId)
                                .setAvailabilityGroupKey(roomId)
                                .setOriginalRoomId(roomId)
                                .setAvailability(block.getNumberOfRoomsLeft())
                                .setPrice(TPriceWithDetails.newBuilder()
                                        .setCurrency(task.getRequest().getCurrency())
                                        .setAmount((int)price.getPrice())
                                        .build())
                                .setDisplayedTitle(StringValue.of(title))
                                .setPansion(pansion)
                                .setWifiIncluded(BoolValue.of(block.isFreeWifi()))
                                .setFreeCancellation(BoolValue.of(block.isRefundable()))
                                .setLandingInfo(buildLandingInfo(result.getHotelUrl(), blockId, neededRoomCount, hotelId, result, hotels.get(hotelId)));
                        if (refundRules != null) {
                            o.addAllRefundRule(mapToProtoRefundRules(refundRules));
                        }
                        if (userPlatform == BookingUserPlatform.MOBILE) {
                            o.setRestrictions(TOfferRestrictions.newBuilder()
                                    .setRequiresMobile(true)
                                    .build());
                        }
                        onOffer(task, o);
                    });
                } catch (Throwable ex) {
                    logger.error(String.format("Task %s: failed to parse offer", task.getId()), ex);
                    task.onError(ProtoUtils.errorFromThrowable(ex, task.isIncludeDebug()));
                }
            });
        }
        for (var originalId: tasksByOriginalId.keySet()) {
            if (!foundOriginalIds.contains(originalId)) {
                hotelIdNotFoundCounter.increment();
                if (config.isFailOnNotFoundHotelId()) {
                    logger.error("Req {}: OriginalId '{}' not found in resp", requestId, originalId);
                    tasksByOriginalId.get(originalId).forEach(task -> {
                        task.onError(TError.newBuilder()
                                .setCode(EErrorCode.EC_GENERAL_ERROR)
                                .setMessage("OriginalId not found in resp"));
                    });
                } else {
                    logger.debug("Req {}: OriginalId '{}' not found in resp", requestId, originalId);
                }
            }
        }
    }

    private TOfferLandingInfo buildLandingInfo(String hotelUrl, String blockId, int neededRoomCount, String hotelId, Result result, Hotel hotel) {
        String hotelLanding = URLDecoder.decode(hotelUrl, StandardCharsets.UTF_8);
        UriBuilder landingBuilder = UriComponentsBuilder.fromUriString(hotelLanding);
        landingBuilder.queryParam("show_room", blockId);

        if (neededRoomCount > 1) {
            landingBuilder.fragment("group_recommendation");
        } else {
            landingBuilder.fragment("RD" + blockId.split("_")[0]);
        }

        UriBuilder searchLandingBuilder = UriComponentsBuilder.fromHttpUrl("https://www.booking.com/searchresults.ru.html");
        searchLandingBuilder.queryParam("aid", "2192269");
        searchLandingBuilder.queryParam("checkin", result.getCheckin());
        searchLandingBuilder.queryParam("checkout", result.getCheckout());
        searchLandingBuilder.queryParam("city_id", hotel.getHotelData().getCityId());
        searchLandingBuilder.queryParam("dest_id", hotel.getHotelData().getCityId());
        searchLandingBuilder.queryParam("dest_type", "city");
        searchLandingBuilder.queryParam("highlighted_hotels", hotelId);

        UriBuilder hotelLandingBuilder = UriComponentsBuilder.fromHttpUrl(hotelLanding);
        hotelLandingBuilder.replaceQueryParam("aid", "2192270");

        return TOfferLandingInfo.newBuilder()
                .setLandingPageUrl(landingBuilder.build().toString())
                .setBookingSearchPageLandingUrl(searchLandingBuilder.build().toString())
                .setBookingHotelPageLandingUrl(hotelLandingBuilder.build().toString())
                .build();
    }
}
