package ru.yandex.travel.hotels.searcher.partners;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.partners.bronevik.AvailableMeals;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikClient;
import ru.yandex.travel.hotels.common.partners.bronevik.BronevikRefundRulesException;
import ru.yandex.travel.hotels.common.partners.bronevik.HotelRoom;
import ru.yandex.travel.hotels.common.partners.bronevik.HotelWithInfo;
import ru.yandex.travel.hotels.common.partners.bronevik.SearchHotelOffersResponse;
import ru.yandex.travel.hotels.common.partners.bronevik.utils.BronevikUtils;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ESearchWarningCode;
import ru.yandex.travel.hotels.proto.TBronevikMeal;
import ru.yandex.travel.hotels.proto.TBronevikOffer;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.proto.TRefundRule;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.TravelTokenService;
import ru.yandex.travel.hotels.searcher.services.cache.bronevik.hotelInfo.BronevikHotelInfoSearcher;

import static java.time.temporal.ChronoUnit.DAYS;

@PartnerBean(EPartnerId.PI_BRONEVIK)
@EnableConfigurationProperties(BronevikTaskHandlerProperties.class)
public class BronevikPartnerTaskHandler extends AbstractPartnerTaskHandler<BronevikTaskHandlerProperties> {

    private final BronevikClient bronevikClient;
    private final BronevikHotelInfoSearcher bronevikHotelInfoSearcher;
    private final TravelTokenService travelTokenService;
    private final Counter missingHotelIdCounter;
    private final Counter missingRoomIdCounter;
    private final Counter failedRefundRulesCounter;
    private final Counter tooManyHotelIdsPerRequestCounter;
    private final Counter checkinCheckoutRangeFailedCounter;

    public BronevikPartnerTaskHandler(BronevikTaskHandlerProperties config, BronevikClient bronevikClient,
                                      BronevikHotelInfoSearcher bronevikHotelInfoSearcher,
                                      TravelTokenService travelTokenService) {
        super(config);
        this.bronevikClient = bronevikClient;
        this.bronevikHotelInfoSearcher = bronevikHotelInfoSearcher;
        this.travelTokenService = travelTokenService;
        missingHotelIdCounter = Metrics.counter("searcher.partners.bronevik.missingHotelId");
        missingRoomIdCounter = Metrics.counter("searcher.partners.bronevik.missingRoomId");
        failedRefundRulesCounter = Metrics.counter("searcher.partners.bronevik.failedRefundRules");
        tooManyHotelIdsPerRequestCounter = Metrics.counter("searcher.partners.bronevik.tooManyHotelIdsPerRequest");
        checkinCheckoutRangeFailedCounter = Metrics.counter("searcher.partners.bronevik.rangeFailedBetweenCheckinCheckout");

    }


    @Override
    protected void checkTask(Task task) {
        var checkin = LocalDate.parse(task.getGroupingKey().getCheckInDate());
        var checkout = LocalDate.parse(task.getGroupingKey().getCheckOutDate());
        if (checkin.plus(config.getCheckinCheckoutRange(), DAYS).compareTo(checkout) < 0) {
            var message = "Maybe crashed, because over 60 days between checkin and checkout";
            checkinCheckoutRangeFailedCounter.increment();
            logger.error(message);
            throw new IllegalArgumentException(message);
        }
    }

    private void validatePreRequest(List<Task> tasks, String requestId) {
        List<Integer> hotelIds =
                tasks.stream().map(task -> Integer.parseInt(task.getRequest().getHotelId().getOriginalId())).collect(Collectors.toList());

        if (hotelIds.size() > 100) {
            var message = String.format("for requestId %s: Maybe crashed, because over 100 hotelIds for one request, pls check batch for bronevik", requestId);
            tooManyHotelIdsPerRequestCounter.increment();
            logger.error(message);
            throw new RuntimeException(message);
        }
    }

    private TBronevikOffer generateTBronevikOffer(
            int adults,
            List<Integer> children,
            String offerCode,
            int hotelId,
            int roomId,
            AvailableMeals meals,
            String checkIn,
            String checkOut,
            int price,
            ECurrency currency
    ) {

        var bronevikOfferBuilder = TBronevikOffer.newBuilder();
        bronevikOfferBuilder.setOccupancy(adults)
                .addAllChildrenAge(children)
                .setOfferCode(offerCode)
                .setHotelId(hotelId)
                .setRoomId(roomId)
                .setCheckin(checkIn)
                .setCheckout(checkOut)
                .setCurrency(currency)
                .setPrice(price);

        if (meals != null) {
            meals.getMeal().stream()
                    .map(meal -> TBronevikMeal.newBuilder()
                            .setId(meal.getId())
                            .setPrice(BronevikUtils.getPriceInt(meal.getPriceDetails()))
                            .setIncluded(meal.isIncluded())
                            .setVatPercent(meal.getVATPercent())
                            .build())
                    .forEach(bronevikOfferBuilder::addMeals);
        }

        return bronevikOfferBuilder.build();
    }

    private void processResponce(
            String requestId,
            List<Task> tasks,
            SearchHotelOffersResponse response,
            Map<Integer, HotelWithInfo> hotelInfoResponse) {
        logger.debug("starting bronevik process response");
        Map<String, List<Task>> tasksByOriginalId = mapTasksByOriginalId(tasks);
        var hotels = response.getHotels().getHotel();
        for (var hotel : hotels) {
            var hotelId = hotel.getId().toString();
            List<Task> taskList = tasksByOriginalId.get(hotelId);
            if (taskList == null) {
                logger.warn("Req {}, Unknown hotel_id {}", requestId, hotelId);
                missingHotelIdCounter.increment();
                continue;
            }

            HotelWithInfo hotelWithInfo = hotelInfoResponse.get(hotel.getId());
            if (hotelWithInfo == null) {
                logger.error("Req {}, Unknown hotelId {}", requestId, hotelId);
                missingHotelIdCounter.increment();
                return;
            }
            taskList.forEach(task -> {
                try {
                    logger.debug("parse bronevik offers task id {}", task.getId());
                    var offers = hotel.getOffers().getOffer();
                    offers.forEach(offer -> {
                        logger.debug("start parsing bronevik offer id {}", offer.getCode());

                        BronevikUtils.generateMealsForOffer(offer.getMeals(), config.getClient().getSoapType())
                                .forEach(availableMeals -> {
                                    logger.debug("start parsing meals");
                                    ru.yandex.travel.hotels.proto.EPansionType pansion = BronevikUtils.parseMeals(
                                            availableMeals, config.getClient().getSoapType());

                                    String offerId = ProtoUtils.randomId();
                                    Integer roomId = offer.getRoomId();

                                    logger.debug("start parsing bronevik hotel info");
                                    logger.debug("start parsing bronevik hotel room");

                                    HotelRoom room = null;
                                    for (var hotelRoom : hotelWithInfo.getRooms().getRoom()) {
                                        if (hotelRoom.getId().compareTo(roomId) == 0) {
                                            room = hotelRoom;
                                        }
                                    }
                                    if (room == null) {
                                        logger.error("Req {}, Unknown roomId {} in hotelId {}", requestId, roomId, hotelId);
                                        missingRoomIdCounter.increment();
                                        return;
                                    }

                                    if (offer.getFreeRooms().compareTo(0) == 0) {
                                        logger.error("Req {}, for offer {} in hotelId {} not free rooms", requestId, offer.getCode(), hotelId);
                                        return;
                                    }
                                    String capacity = room.getRoomCapacity().toString();
                                    String ourCapacity = String.format("<=%s", capacity);
                                    logger.debug("start parsing bronevik refund rules");
                                    RefundRules refundRules;
                                    try {
                                        refundRules = BronevikUtils.parseRefundRules(
                                                offer.getCancellationPolicies(),
                                                BronevikUtils.getPriceInt(offer.getPriceDetails().getClient()));
                                    } catch (BronevikRefundRulesException exception) {
                                        logger.error(
                                                "bronevik refund rules dont build: {}, requestId {}, hotelId {}, offerId {}",
                                                exception.getMessage(),
                                                requestId,
                                                hotelId,
                                                offerId);
                                        failedRefundRulesCounter.increment();
                                        task.onWarning(ESearchWarningCode.SW_BR_DROPPED_INVALID_CANCELLATION_PENALTIES);
                                        return;
                                    }
                                    logger.debug("start parsing bronevik price");
                                    var totalPrice = BronevikUtils.getTotalPrice(
                                            offer.getPriceDetails(),
                                            availableMeals,
                                            task.getOccupancy().getAdults(),
                                            BronevikUtils.getNights(task.getGroupingKey().getCheckInDate(), task.getGroupingKey().getCheckOutDate()));
                                    TPriceWithDetails priceWithDetails = TPriceWithDetails.newBuilder()
                                            .setCurrency(ECurrency.C_RUB)
                                            .setAmount(totalPrice)
                                            .build();

                                    logger.debug("build bronevik KV offer");
                                    TBronevikOffer bronevikOffer = generateTBronevikOffer(
                                            task.getOccupancy().getAdults(),
                                            task.getOccupancy().getChildren(),
                                            offer.getCode(),
                                            hotel.getId(),
                                            roomId,
                                            availableMeals,
                                            task.getRequest().getCheckInDate(),
                                            task.getRequest().getCheckOutDate(),
                                            priceWithDetails.getAmount(),
                                            priceWithDetails.getCurrency());
                                    boolean freeCancellation = refundRules.isFullyRefundable();
                                    List<TRefundRule> protoRefundRules = mapToProtoRefundRules(refundRules);
                                    var descriptionDetails = hotel.getDescriptionDetails();
                                    TCoordinates coordinates = descriptionDetails.getLatitude() != null && descriptionDetails.getLongitude() != null
                                            ? TCoordinates.newBuilder()
                                                .setLatitude(Double.parseDouble(descriptionDetails.getLatitude()))
                                                .setLongitude(Double.parseDouble(descriptionDetails.getLongitude()))
                                                .build()
                                            : null;
                                    logger.debug("push to KV bronevik offer");
                                    String travelTokenString = travelTokenService.storeTravelTokenAndGetItsString(offerId,
                                            partnerId, task, roomId.toString(), pansion, freeCancellation, priceWithDetails,
                                            coordinates, protoRefundRules,
                                            TOfferData.newBuilder().setBronevikOffer(bronevikOffer));
                                    logger.debug("building TOffer");
                                    TOffer.Builder newOffer = TOffer.newBuilder()
                                            .setExternalId(offer.getCode())
                                            .setId(offerId)
                                            .setOperatorId(EOperatorId.OI_BRONEVIK)
                                            .setAvailability(offer.getFreeRooms())
                                            .setOriginalRoomId(roomId.toString())
                                            .setDisplayedTitle(StringValue.of(offer.getName()))
                                            .setRoomCount(1)
                                            .setPrice(priceWithDetails)
                                            .addAllRefundRule(protoRefundRules)
                                            .setFreeCancellation(BoolValue.of(freeCancellation))
                                            .setLandingInfo(TOfferLandingInfo.newBuilder()
                                                    .setLandingTravelToken(travelTokenString)
                                                    .build())
                                            .setCapacity(ourCapacity)
                                            .setPansion(pansion);
                                    logger.debug("bronevik offer was parsed!");
                                    onOffer(task, newOffer);
                                });
                    });
                } catch (Throwable ex) {
                    logger.error(String.format("Task %s: failed to parse offer", task.getId()), ex);
                    task.onError(ProtoUtils.errorFromThrowable(ex, task.isIncludeDebug()));
                }
            });
        }
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {

        try {
            validatePreRequest(tasks, requestId);
        } catch (RuntimeException ex) {
            tasks.forEach(task -> task.onError(ProtoUtils.errorFromThrowable(ex, true)));
            return CompletableFuture.allOf();
        }

        List<Integer> hotelIds =
                tasks.stream().map(task -> Integer.parseInt(task.getRequest().getHotelId().getOriginalId())).collect(Collectors.toList());
        logger.debug("run request to bronevik");

        CompletableFuture<SearchHotelOffersResponse> searchHotelOffersResponseFuture = bronevikClient.searchHotelOffers(
                groupingKey.getOccupancy().getAdults(),
                groupingKey.getOccupancy().getChildren(),
                hotelIds,
                groupingKey.getCheckInDate(),
                groupingKey.getCheckOutDate(),
                Utils.CURRENCY_NAMES.get(groupingKey.getCurrency()),
                requestId);
        var getHotelInfoResponseFuture = bronevikHotelInfoSearcher.getHotelsInfo(hotelIds);
        return CompletableFuture.allOf(
                searchHotelOffersResponseFuture,
                getHotelInfoResponseFuture).thenApply(x -> {
            var searchHotelOffersResponse = searchHotelOffersResponseFuture.join();
            var hotelInfoResponse = getHotelInfoResponseFuture.join();
            logger.debug("requests to bronevik was done");
            processResponce(requestId, tasks, searchHotelOffersResponse, hotelInfoResponse);
            return x;
        });
    }

}
