package ru.yandex.travel.hotels.searcher.partners;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.hotels.common.UpdatableDataHolder;
import ru.yandex.travel.hotels.common.partners.dolphin.DefaultDolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DolphinClient;
import ru.yandex.travel.hotels.common.partners.dolphin.model.FixedPercentFeeRefundParams;
import ru.yandex.travel.hotels.common.partners.dolphin.model.IdNameMap;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Offer;
import ru.yandex.travel.hotels.common.partners.dolphin.model.OfferList;
import ru.yandex.travel.hotels.common.partners.dolphin.model.Pansion;
import ru.yandex.travel.hotels.common.partners.dolphin.model.SearchResponse;
import ru.yandex.travel.hotels.common.partners.dolphin.proto.THotelCoordinates;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.DolphinRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.Formatters;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.RoomNameNormalizer;
import ru.yandex.travel.hotels.common.partners.dolphin.utils.coordinates.DolphinHotelCoordinatesProvider;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ESearchWarningCode;
import ru.yandex.travel.hotels.proto.TDolphinOffer;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPartnerSpecificOfferData;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.proto.TRefundRule;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.TravelTokenService;

@PartnerBean(EPartnerId.PI_DOLPHIN)
@EnableConfigurationProperties(DolphinTaskHandlerProperties.class)
@Slf4j
public class DolphinTaskHandler extends AbstractPartnerTaskHandler<DolphinTaskHandlerProperties> {
    private final ObjectMapper mapper = DefaultDolphinClient.createObjectMapper();
    private final UpdatableDataHolder<Map<Long, Pansion>> pansions;
    private final UpdatableDataHolder<IdNameMap> rooms;
    private final UpdatableDataHolder<IdNameMap> roomCategories;
    private final TravelTokenService travelTokenService;
    private final RoomNameNormalizer roomNameNormalizer;

    private final Counter forbiddenOffersCounter = Metrics.counter("searcher.partners.dolphin.forbiddenOffers");
    private final Counter badPansionOffersCounter = Metrics.counter("searcher.partners.dolphin.badPansionOffers");
    private final DolphinClient dolphinClient;

    private final DolphinHotelCoordinatesProvider hotelCoordinatesProvider;

    DolphinTaskHandler(DolphinTaskHandlerProperties config, TravelTokenService travelTokenService,
                       RoomNameNormalizer roomNameNormalizer, DolphinClient dolphinClient,
                       DolphinHotelCoordinatesProvider hotelCoordinatesProvider) {
        super(config);
        this.travelTokenService = travelTokenService;
        this.roomNameNormalizer = roomNameNormalizer;
        this.dolphinClient = dolphinClient;
        this.rooms = new UpdatableDataHolder<>(config.getAuxUpdateInterval(), config.getAuxRetryInterval(),
                dolphinClient::getRooms);
        this.roomCategories = new UpdatableDataHolder<>(config.getAuxUpdateInterval(), config.getAuxRetryInterval(),
                dolphinClient::getRoomCategories);
        this.pansions = new UpdatableDataHolder<>(config.getAuxUpdateInterval(), config.getAuxRetryInterval(),
                dolphinClient::getPansionMap);
        this.hotelCoordinatesProvider = hotelCoordinatesProvider;
    }

    @Override
    CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        var hotelIds = tasks.stream()
                .map(task -> task.getRequest().getHotelId().getOriginalId())
                .distinct()
                .collect(Collectors.toList());
        CompletableFuture<SearchResponse> responseFuture = dolphinClient
                .withCallContext(tasks.get(0).getCallContext())
                .searchCheckins(
                        LocalDate.parse(groupingKey.getCheckInDate()),
                        LocalDate.parse(groupingKey.getCheckOutDate()),
                        groupingKey.getOccupancy(), hotelIds,
                        requestId);
        CompletableFuture<IdNameMap> roomsFuture;
        CompletableFuture<IdNameMap> roomCategoriesFuture;
        CompletableFuture<Map<Long, Pansion>> pansionsFuture;
        if (tasks.get(0).getCallContext().getTestContext() != null) {
            log.warn("Test context found: will skip caching");
            var client = dolphinClient.withCallContext(tasks.get(0).getCallContext());
            roomsFuture = client.getRooms();
            roomCategoriesFuture = client.getRoomCategories();
            pansionsFuture = client.getPansionMap();
        } else {
            roomsFuture = rooms.get();
            roomCategoriesFuture = roomCategories.get();
            pansionsFuture = pansions.get();

        }
        return CompletableFuture.allOf(responseFuture, roomsFuture, roomCategoriesFuture, pansionsFuture)
                .thenApply(ignored -> {
                    processResponse(
                            tasks,
                            responseFuture.join(),
                            roomsFuture.join(),
                            roomCategoriesFuture.join(),
                            pansionsFuture.join());
                    return null;
                });
    }

    private void processResponse(List<Task> tasks, SearchResponse dolphinResponse, IdNameMap roomMap,
                                 IdNameMap roomCatMap, Map<Long, Pansion> pansionMap) {
        var taskByOriginalId = mapTasksByOriginalId(tasks);
        for (var offerList : dolphinResponse.getOfferLists()) {
            var originalId = String.valueOf(offerList.getHotelId());
            List<Task> hotelTasks = taskByOriginalId.get(originalId);
            var hotelCoordinates = hotelCoordinatesProvider.getCoordinates(originalId);
            addOffersToTasks(hotelTasks, offerList, roomMap, roomCatMap, pansionMap, hotelCoordinates);
        }
    }

    private TDolphinOffer generateDolphinOffer(long hotelId, long tourId, long pansionId, String date, int nights,
                                               Offer innerOffer) {
        @SuppressWarnings("Duplicates")
        TJson shoppingOffer;
        try {
            shoppingOffer = TJson.newBuilder()
                    .setValue(mapper.writer().writeValueAsString(innerOffer))
                    .build();
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Unable to serialize rate offer");
        }
        return TDolphinOffer.newBuilder()
                .setHotelId(hotelId)
                .setPansionId(pansionId)
                .setTourId(tourId)
                .setNights(nights)
                .setDate(date)
                .setShoppingOffer(shoppingOffer)
                .build();
    }

    private void addOffersToTasks(List<Task> tasks, OfferList offerList, IdNameMap roomMap, IdNameMap roomCatMap,
                                  Map<Long, Pansion> pansionMap, THotelCoordinates hotelCoordinates) {
        for (var task : tasks) {
            LocalDate checkinDate = LocalDate.parse(task.getRequest().getCheckInDate());
            LocalDate checkoutDate = LocalDate.parse(task.getRequest().getCheckOutDate());
            try {
                for (Offer offer : offerList.getOffers()) {
                    if (offer.getDiscount() < 0) {
                        forbiddenOffersCounter.increment();
                        continue;
                    }
                    Preconditions.checkArgument(offer.getRate().equals("рб"), "Unexpected rate " + offer.getRate());
                    long priceAmount = Math.round(offer.getPrice());
                    if (!validatePrice(task, priceAmount)) {
                        continue;
                    }
                    EPansionType pansionType = pansionMap.get(offerList.getPansionId()).getPansionType();
                    if (pansionType == EPansionType.PT_UNKNOWN) {
                        log.warn("Unknown pansion with id {}", offerList.getPansionId());
                        badPansionOffersCounter.increment();
                        task.onWarning(ESearchWarningCode.SW_TL_DROPPED_APARTMENTS);
                        continue;
                    }
                    String displayedTitle = roomNameNormalizer.normalize(
                            String.format("Номер %s %s",
                                    roomCatMap.get(offer.getCategoryId()), roomMap.get(offer.getRoomId())));
                    TDolphinOffer dolphinOffer = generateDolphinOffer(offerList.getHotelId(), offerList.getTourId(),
                            offerList.getPansionId(), offerList.getDate().format(Formatters.DOLPHIN_DATE_FORMATTER),
                            offerList.getNights(), offer);
                    BigDecimal price = BigDecimal.valueOf(offer.getPrice());
                    var offerId = ProtoUtils.randomId();
                    String originalRoomId = String.format("%s:%s", offer.getCategoryId(), offer.getRoomId());
                    RefundRules rules = DolphinRefundRulesBuilder.build(
                            Instant.now(),
                            checkinDate.atStartOfDay(ZoneId.of(config.getDefaultTimeZone())).toInstant(),
                            BigDecimal.valueOf(offer.getPrice()),
                            "RUB",
                            new FixedPercentFeeRefundParams(config.getCancellationPenalties()));
                    boolean freeCancellation = rules.isFullyRefundable();
                    TPriceWithDetails offerPrice = TPriceWithDetails.newBuilder()
                            .setCurrency(ECurrency.C_RUB)
                            .setAmount(price.setScale(0, RoundingMode.HALF_UP).intValue())
                            .build();
                    List<TRefundRule> protoRefundsRule = mapToProtoRefundRules(rules);
                    TCoordinates coordinates = hotelCoordinates != null
                            ? TCoordinates.newBuilder()
                                .setLatitude(hotelCoordinates.getLatitude())
                                .setLongitude(hotelCoordinates.getLongitude())
                                .build()
                            : null;
                    String travelTokenString = travelTokenService.storeTravelTokenAndGetItsString(offerId, partnerId,
                            task, originalRoomId, pansionType, freeCancellation, offerPrice,
                            coordinates, protoRefundsRule,
                            TOfferData.newBuilder().setDolphinOffer(dolphinOffer));
                    String capacity = Capacity.fromOccupancy(task.getOccupancy()).toString();
                    TOffer.Builder taskOfferBuilder = TOffer.newBuilder()
                            .setId(offerId)
                            .setDisplayedTitle(StringValue.of(displayedTitle))
                            .setPrice(offerPrice)
                            .setLandingInfo(TOfferLandingInfo.newBuilder()
                                    .setLandingTravelToken(travelTokenString)
                                    .build())
                            .setAvailability(offer.getPlaces())
                            .setCapacity(capacity)
                            .setSingleRoomCapacity(capacity)
                            .setRoomCount(1)
                            .setOriginalRoomId(originalRoomId)
                            .setPansion(pansionType)
                            .setFreeCancellation(BoolValue.of(freeCancellation))
                            .addAllRefundRule(protoRefundsRule)
                            .setOperatorId(EOperatorId.OI_DOLPHIN)
                            .setPartnerSpecificData(TPartnerSpecificOfferData.newBuilder()
                                    .setDolphinData(TPartnerSpecificOfferData.TDolphinData.newBuilder()
                                            .setTour(offerList.getTourId())
                                            .setPansion(offerList.getPansionId())
                                            .setRoom(offer.getRoomId())
                                            .setRoomCat(offer.getCategoryId())
                                            .build())
                                    .build());
                    onOffer(task, taskOfferBuilder);
                }
            } catch (Throwable ex) {
                logger.error(String.format("Task %s: failed to parse offer", task.getId()), ex);
                task.onError(ProtoUtils.errorFromThrowable(ex, task.isIncludeDebug()));
            }
        }
    }
}
