package ru.yandex.travel.hotels.searcher.partners;

import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.hotels.common.partners.expedia.DefaultExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaClient;
import ru.yandex.travel.hotels.common.partners.expedia.ExpediaRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.expedia.Helpers;
import ru.yandex.travel.hotels.common.partners.expedia.KnownAmenity;
import ru.yandex.travel.hotels.common.partners.expedia.Pansions;
import ru.yandex.travel.hotels.common.partners.expedia.model.common.PricingInformation;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.PropertyAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.RoomAvailability;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.SalesChannel;
import ru.yandex.travel.hotels.common.partners.expedia.model.shopping.ShoppingRate;
import ru.yandex.travel.hotels.common.partners.expedia.proto.THotelCoordinates;
import ru.yandex.travel.hotels.common.partners.expedia.utils.coordinates.ExpediaHotelCoordinatesProvider;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TExpediaOffer;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TOfferRestrictions;
import ru.yandex.travel.hotels.proto.TPaymentSchedule;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.TravelTokenService;
import ru.yandex.travel.hotels.searcher.services.cache.expedia.ExpediaPropertyPansions;

@PartnerBean(EPartnerId.PI_EXPEDIA)
@EnableConfigurationProperties(ExpediaTaskHandlerProperties.class)
@Slf4j
public class ExpediaTaskHandler extends AbstractPartnerTaskHandler<ExpediaTaskHandlerProperties> {

    private final ExpediaClient client;
    private final Counter missingRoomNameCounter;
    private final Counter fencedDealCounter;
    private final ObjectMapper expediaMapper;
    private final ExpediaPropertyPansions propertyPansions;
    private final TravelTokenService travelTokenService;
    private final ExpediaHotelCoordinatesProvider hotelCoordinatesProvider;

    ExpediaTaskHandler(ExpediaTaskHandlerProperties config, ExpediaClient client,
                       ExpediaPropertyPansions propertyPansions, TravelTokenService travelTokenService,
                       ExpediaHotelCoordinatesProvider hotelCoordinatesProvider) {
        super(config);
        this.client = client;
        this.propertyPansions = propertyPansions;
        this.travelTokenService = travelTokenService;
        this.hotelCoordinatesProvider = hotelCoordinatesProvider;

        missingRoomNameCounter = Metrics.counter("searcher.partners.expedia.missingRoomName");
        fencedDealCounter = Metrics.counter("searcher.partners.expedia.fencedDeals");
        expediaMapper = DefaultExpediaClient.createObjectMapper();

    }

    @Override
    CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        var regularFuture = executeForChannel(groupingKey, tasks, SalesChannel.CACHE, requestId);
        var mobileFuture = config.isSearchMobileDeals()
                ? executeForChannel(groupingKey, tasks, SalesChannel.MOBILE_WEB, requestId)
                : CompletableFuture.completedFuture(null);
        return CompletableFuture.allOf(regularFuture, mobileFuture);
    }

    private CompletableFuture<Void> executeForChannel(Task.GroupingKey groupingKey, List<Task> tasks,
                                                      SalesChannel channel, String requestId) {
        String sessionId = ProtoUtils.randomId();
        var hotelIds = tasks.stream()
                .map(task -> task.getRequest().getHotelId().getOriginalId())
                .distinct()
                .collect(Collectors.toList());
        return client.findAvailabilities(hotelIds,
                LocalDate.parse(groupingKey.getCheckInDate()),
                LocalDate.parse(groupingKey.getCheckOutDate()), groupingKey.getOccupancy().toExpediaString(),
                config.getRequestCurrency(), config.isIncludeFencedRates(),
                        config.getSearchIpAddress(), sessionId, requestId, channel)
                .thenAccept(availability -> processResponse(tasks, availability, sessionId, channel));
    }

    private void processResponse(List<Task> tasks, Map<String, PropertyAvailability> results,
                                 String sessionId, SalesChannel channel) {
        Map<String, List<Task>> tasksById = mapTasksByOriginalId(tasks);
        String occupancyString = tasks.get(0).getOccupancy().toExpediaString();
        Preconditions.checkArgument(tasks.stream()
                .allMatch(t -> t.getOccupancy().toExpediaString().equals(occupancyString)), "Batched tasks have " +
                "different occupancies");
        for (var entry : results.entrySet()) {
            var av = entry.getValue();
            List<Task> taskList = tasksById.get(entry.getKey());
            if (taskList == null) {
                logger.warn("Unknown property_id '{}'", entry.getKey());
                continue;
            }
            var hotelCoordinates = hotelCoordinatesProvider.getCoordinates(entry.getKey());
            taskList.forEach(task -> entry.getValue().getRooms().forEach(room -> room.getRates()
                    .forEach(rate -> processRate(av, room, rate, task, hotelCoordinates, sessionId, channel))));
        }
    }

    private void processRate(PropertyAvailability propertyAvailability, RoomAvailability room,
                             ShoppingRate rate, Task task, THotelCoordinates hotelCoordinates, String sessionId,
                             SalesChannel channel) {
        try {
            if (Strings.isNullOrEmpty(room.getRoomName())) {
                missingRoomNameCounter.increment();
                logger.error(String.format("Task %s: room without name, room id is %s", task.getId(), room.getId()));
                return;
            }
            String extId = rate.getId();
            int availability = rate.getAvailableRooms();
            String occupancyString = task.getOccupancy().toExpediaString();
            // We assume that answer contains exactly one rate for the occupancy
            // if its incorrect - we want to fail? save record to logging.
            assert rate.getOccupancyPricing().size() == 1;
            PricingInformation pricing = rate.getOccupancyPricing().get(occupancyString);
            PricingInformation rounded = Helpers.round(pricing);

            // TODO(tivelkov): HOTELS-5024 - check whether we need any currency conversion here
            Preconditions.checkState(pricing.getTotals().getInclusive().getBillableCurrency().getCurrency().equals(
                    config.getRequestCurrency()), "Billable currency does not match the requested currency");

            log.info("Rounding {} to {}", pricing, rounded);
            if (rate.hasSomeFencedDeal()) {
                logger.info(String.format("Hotel %s, room %s, rate %s is fenced!", propertyAvailability.getPropertyId(),
                        room.getId(), rate.getId()));
                fencedDealCounter.increment();
            }
            int amount = rounded.getTotals().getInclusive().getBillableCurrency().getValue().intValue();
            if (!validatePrice(task, amount)) {
                return;
            }
            var propertyPansion = propertyPansions.getPropertyPansion(propertyAvailability.getPropertyId());
            var ratePansion = Pansions.getPansionType(KnownAmenity.fromShoppingRate(rate));
            var pansion = Pansions.combinePansionType(propertyPansion, ratePansion);
            TExpediaOffer expOffer = generateExpediaOffer(propertyAvailability, room, rate, task, sessionId);
            var offerId = ProtoUtils.randomId();

            var actualizedRefundRules =
                    ExpediaRefundRulesBuilder.build(rate, occupancyString, config.getSafetyInterval()).actualize();
            boolean freeCancellation = actualizedRefundRules.isFullyRefundable();
            var priceBuilder = TPriceWithDetails.newBuilder()
                    .setCurrency(ECurrency.C_RUB)
                    .setAmount(amount);
            var protoRefundRules = mapToProtoRefundRules(actualizedRefundRules);
            TCoordinates coordinates = hotelCoordinates != null
                    ? TCoordinates.newBuilder()
                        .setLongitude(hotelCoordinates.getLongitude())
                        .setLatitude(hotelCoordinates.getLatitude())
                        .build()
                    : null;
            String travelTokenString = travelTokenService.storeTravelTokenAndGetItsString(offerId, partnerId, task,
                    room.getId(), pansion, freeCancellation, priceBuilder.build(),
                    coordinates, protoRefundRules,
                    TOfferData.newBuilder().setExpediaOffer(expOffer));
            String capacity = Capacity.fromOccupancy(task.getOccupancy()).toString();
            var restrictionBuilder = TOfferRestrictions.newBuilder()
                    .setRequiresMobile(channel.isMobile());
            if (rate.hasSomeFencedDeal()) {
                restrictionBuilder.setRequiresRestrictedUser(true);
            }

            TOffer.Builder builder = TOffer.newBuilder()
                    .setId(offerId)
                    .setDisplayedTitle(StringValue.of(room.getRoomName()))
                    .setPrice(priceBuilder)
                    .setLandingInfo(TOfferLandingInfo.newBuilder()
                            .setLandingTravelToken(travelTokenString)
                            .build())
                    .setRestrictions(restrictionBuilder)
                    .setAvailability(availability)
                    .setCapacity(capacity)
                    .setSingleRoomCapacity(capacity)
                    .setRoomCount(1)
                    .setExternalId(extId)
                    .setFreeCancellation(BoolValue.of(freeCancellation))
                    .addAllRefundRule(mapToProtoRefundRules(actualizedRefundRules))
                    .setOriginalRoomId(room.getId())
                    .setPansion(pansion)
                    .setOperatorId(EOperatorId.OI_EXPEDIA);

            onOffer(task, builder);
        } catch (Throwable ex) {
            logger.error(String.format("Task %s: failed to parse offer", task.getId()), ex);
            task.onError(ProtoUtils.errorFromThrowable(ex, task.isIncludeDebug()));
        }
    }

    private TExpediaOffer generateExpediaOffer(PropertyAvailability property, RoomAvailability room,
                                               ShoppingRate rate, Task task, String sessionId) {
        String occupancyString = task.getOccupancy().toExpediaString();
        var pricingInformation = rate.getOccupancyPricing().get(occupancyString);
        long longAmount = pricingInformation.getTotals().getInclusive().getBillableCurrency().getValue()
                .setScale(2, RoundingMode.HALF_UP).longValue();
        var actualizedRefundRules =
                ExpediaRefundRulesBuilder.build(rate, occupancyString,
                        config.getSafetyInterval()).actualize();
        boolean refundable = actualizedRefundRules.isRefundable();
        String hotelId = property.getPropertyId();
        String roomId = room.getId();
        String rateId = rate.getId();
        TJson shoppingOffer;
        try {
            shoppingOffer = TJson.newBuilder()
                    .setValue(expediaMapper.writer().writeValueAsString(rate))
                    .build();
        } catch (JsonProcessingException e) {
            throw new RuntimeException("Unable to serialize rate offer", e);
        }

        return TExpediaOffer.newBuilder()
                .setPermalink(task.getRequest().getPermalink())
                .setCheckInDate(task.getRequest().getCheckInDate())
                .setCheckOutDate(task.getRequest().getCheckOutDate())
                .setOccupancy(occupancyString)
                .setBillableCurrency(ProtoCurrencyUnit.fromCurrencyCode(pricingInformation.getTotals().getInclusive().getBillableCurrency().getCurrency()).getProtoCurrency())
                .setPayableCurrency(ECurrency.C_RUB)
                .setExchangeRateValidUntil(ProtoUtils.fromInstant(Instant.MAX))
                .setHotelId(hotelId)
                .setRoomId(roomId)
                .setRateId(rateId)
                .setPayablePrice(longAmount)
                .setRefundable(refundable)
                .setShoppingOffer(shoppingOffer)
                .setSearchRequestId(sessionId)
                .setApiVersion(config.getDefaultApiVersion().getValue())
                .build();
    }
}
