package ru.yandex.travel.hotels.searcher.partners;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.pansions.PansionUnifier;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;

@PartnerBean(EPartnerId.PI_HOTELSCOMBINED)
@EnableConfigurationProperties(HotelsCombinedTaskHandlerProperties.class)
public class HotelsCombinedTaskHandler extends AbstractPartnerTaskHandler<HotelsCombinedTaskHandlerProperties> {

    private final static String NUMBER_REGEX = "^[0-9]+$";
    private final Counter unknownProviderCounter;

    public HotelsCombinedTaskHandler(HotelsCombinedTaskHandlerProperties config) {
        super(config);
        // HTC counters
        unknownProviderCounter = Metrics.counter("searcher.partners.hotelscombined.unknownProvider");
    }

    static EPansionType getPansion(JsonNode jResultNode) {
        Set<Inclusions> inclusions = new HashSet<>();
        for (JsonNode inclusion : jResultNode.get("inclusions")) {
            inclusions.add(Inclusions.values()[inclusion.asInt()]);
        }

        boolean breakfast = inclusions.contains(Inclusions.BREAKFAST);
        boolean lunch = inclusions.contains(Inclusions.LUNCH);
        boolean dinner = inclusions.contains(Inclusions.DINNER);
        boolean fb = inclusions.contains(Inclusions.MEALS);
        boolean ai = inclusions.contains(Inclusions.ALL_INCLUSIVE);

        return PansionUnifier.get(ai, fb, false, breakfast, lunch, dinner);
    }

    private static boolean isNumericId(String id) {
        return id.matches(NUMBER_REGEX);
    }

    private CompletableFuture<Void> await(int iteration) {
        int delay = 0;
        if (iteration > 0) {
            delay = 50;
        }
        if (iteration > 3) {
            delay = 200;
        }
        CompletableFuture<Void> future = new CompletableFuture<>();
        executor.schedule(() -> future.complete(null), delay, TimeUnit.MILLISECONDS);
        return future;
    }

    @Override
    protected void checkTask(Task task) {
        super.checkTask(task);
        if (!isNumericId(task.getRequest().getHotelId().getOriginalId())) {
            throw new IllegalArgumentException("Original ID should be a number");
        }
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        // Reusing the global request timeout; make as much work as possible within the time limit.
        Instant deadline = Instant.now().plus(config.getHttpRequestTimeout());
        String sessionId = String.valueOf(tasks.get(0).getRequest().getId());
        return runMhsHandleIteration(groupingKey, tasks, sessionId, deadline, 0, requestId);
    }

    private CompletableFuture<Void> runMhsHandleIteration(Task.GroupingKey groupingKey, List<Task> batch,
                                                          String sessionId, Instant deadline, int iteration,
                                                          String requestId) {
        if (Instant.now().isAfter(deadline)) {
            CompletableFuture<Void> failedFuture = new CompletableFuture<>();
            failedFuture.completeExceptionally(new IllegalStateException("Request timed out"));
            return failedFuture;
        }
        return CompletableFuture
                .supplyAsync(() -> prepareMhsRequest(groupingKey, batch, sessionId, iteration, requestId), executor)
                .thenCompose(request -> runHttpRequest(request, false, iteration <= 5 ? "call_" + String.valueOf(iteration) : "call_tail", groupingKey.getRequestClass()))
                .thenCompose(response -> {
                    switch (response.getStatusCode()) {
                        case 202:
                            logger.debug("Batch with sessionId {}: Search is in progress; sleeping", sessionId);
                            return await(iteration).thenCompose(aVoid -> runMhsHandleIteration(
                                    groupingKey, batch,sessionId, deadline, iteration + 1, requestId));
                        case 200:
                            logger.debug("Batch with sessionId {}: Search completed", sessionId);
                            processMhsResponse(groupingKey, batch, response);
                            return CompletableFuture.completedFuture(null);
                        default:
                            logger.error("Batch {}: Search failed", sessionId);
                            logger.error(response.getResponseBody());
                            batch.forEach(task -> task.onError(TError.newBuilder().setMessage("Bad HTTP status code: " + String.valueOf(response.getStatusCode())).build()));
                            return CompletableFuture.completedFuture(null);
                    }
                });
    }

    private Request prepareMhsRequest(Task.GroupingKey groupingKey, List<Task> batch, String sessionId, int iteration,
                                      String requestId) {
        RequestBuilder builder = buildHttpRequest(batch, requestId);
        builder = prepareBaseRequest(groupingKey, builder, iteration);
        builder.setUrl(config.getBaseMhsUrl());
        String ids = batch.stream().map(task -> task.getRequest().getHotelId().getOriginalId()).collect(Collectors.joining(","));
        builder.addQueryParam("sessionID", sessionId);
        builder.addQueryParam("destination", "hotels:" + ids);
        builder.addQueryParam("responseOptions", "MultipleHotelsAllRates");
        return builder.build();
    }

    private RequestBuilder prepareBaseRequest(Task.GroupingKey groupingKey, RequestBuilder builder, int iteration) {
        return builder
                .setHeader("accept", "application/json")
                .setHeader("user-agent", "ning/1.x (x86_64-pc-linux-gnu)")
                .setHeader("x-ya-hotels-searcher-attempt", String.valueOf(iteration))
                .addQueryParam("apiKey", config.getApiKey())
                .addQueryParam("onlyIfComplete", "true")
                .addQueryParam("checkin", groupingKey.getCheckInDate())
                .addQueryParam("checkout", groupingKey.getCheckOutDate())
                .addQueryParam("rooms", groupingKey.getOccupancy().toHotelsCombinedString())
                .addQueryParam("languageCode", "RU")
                .addQueryParam("currencyCode", Utils.CURRENCY_NAMES.get(groupingKey.getCurrency()));
    }

    private JsonNode parseResponse(Response response) {
        ObjectMapper mapper = new ObjectMapper();
        JsonNode jNode;
        try {
            jNode = mapper.readTree(response.getResponseBodyAsBytes());
        } catch (IOException e) {
            throw new CompletionException("Parsing failed", e);
        }
        return jNode;
    }

    private void processMhsResponse(Task.GroupingKey groupingKey, List<Task> batch, Response response) {
        JsonNode jNode = parseResponse(response);
        Map<Integer, EOperatorId> providerIndexToOperator = computeProviderIndexToOperatorMap(jNode);
        Map<String, List<Task>> originalIdToTasks = new HashMap<>();
        batch.forEach(task -> {
            String originalId = task.getRequest().getHotelId().getOriginalId();
            List<Task> taskList = originalIdToTasks.get(originalId);
            if (taskList == null) {
                taskList = new ArrayList<>();
                originalIdToTasks.put(originalId, taskList);
            }
            taskList.add(task);
        });
        for (JsonNode jResultNode : jNode.path("results")) {
            List<Task> tasks = originalIdToTasks.get(jResultNode.get("id").asText());
            if (tasks == null || tasks.size() == 0) {
                logger.warn("Task not found for result with id {}", jResultNode.get("id").asText());
                continue;
            }
            for (JsonNode jRateNode : jResultNode.path("rates")) {
                TOffer.Builder builder = getOffer(groupingKey, providerIndexToOperator, jRateNode, tasks);
                if (builder == null) {
                    continue;
                }
                tasks.forEach(task -> onOffer(task, TOffer.newBuilder(builder.build())));
            }
        }
    }

    private TOffer.Builder getOffer(Task.GroupingKey groupingKey, Map<Integer, EOperatorId> providerIndexToOperator,
                                    JsonNode jRateNode, List<Task> tasks) {
        if (jRateNode.get("isBundledRate").asBoolean(false)) {
            return null;
        }
        int providerIndex = jRateNode.get("providerIndex").asInt();
        if (!providerIndexToOperator.containsKey(providerIndex)) {
            return null;
        }

        long priceAmount = jRateNode.get("totalRate").asLong();
        if (!validatePrice(tasks.get(0), priceAmount)) {
            return null;
        }
        String capacity = Capacity.fromOccupancy(groupingKey.getOccupancy()).toString();
        String landingUrl = getBookUri(jRateNode);
        return TOffer.newBuilder()
                .setDisplayedTitle(StringValue.of(jRateNode.get("roomName").asText()))
                .setPrice(TPriceWithDetails.newBuilder()
                        .setAmount((int) priceAmount)
                        .setCurrency(groupingKey.getCurrency())
                        .build()
                )
                .setFreeCancellation(BoolValue.of(jRateNode.get("hasFreeCancellation").asBoolean()))
                .setOperatorId(providerIndexToOperator.get(providerIndex))
                .setPansion(getPansion(jRateNode))
                .setCapacity(capacity)
                .setSingleRoomCapacity(capacity)
                .setRoomCount(1)
                .setLandingInfo(TOfferLandingInfo.newBuilder()
                        .setLandingPageUrl(landingUrl)
                        .build())
                ;
    }

    private String getBookUri(JsonNode jRateNode) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder.fromUriString(jRateNode.get("bookUri").asText());
        UriComponents components = uriBuilder.build();
        List<String> valueList = components.getQueryParams().get("splash");
        if (valueList != null) {
            if (valueList.size() != 1 || !valueList.get(0).equals("false")) {
                logger.warn("URI contains unexpected 'splash' parameter: '{}'", String.join(", ", valueList));
            }
        } else {
            uriBuilder.queryParam("splash", "false");
        }
        return uriBuilder.toUriString();
    }

    private Map<Integer, EOperatorId> computeProviderIndexToOperatorMap(JsonNode jNode) {
        Map<Integer, EOperatorId> providerIndexToOperator = new HashMap<>();
        int providerIndex = 0;
        for (JsonNode jProviderNode : jNode.path("providers")) {
            String code = jProviderNode.get("code").asText();
            switch (code) {
                case "AGD":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_AGODA);
                    break;
                case "ACC":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_ACCORHOTELSCOM);
                    break;
                case "HDE":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_HOTELINFO);
                    break;
                case "IAN":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_HOTELSCOM);
                    break;
                case "PRS":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_PRESTIGIACOM);
                    break;
                case "OLO":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_AMOMA);
                    break;
                case "BOO":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_HTC_ONSITE);
                    break;
                case "OTE":
                    providerIndexToOperator.put(providerIndex, EOperatorId.OI_OTELCOM);
                    break;
                case "HOA":
                    // Silently ignore Hotelopia.com (HOTELS-3756).
                    break;
                default:
                    logger.warn(String.format("Unknown provider code %s", code));
                    unknownProviderCounter.increment();
                    break;
            }
            ++providerIndex;
        }
        return providerIndexToOperator;
    }

    @Override
    protected List<String> getHttpCallPurposes() {
        return Arrays.asList("call_0", "call_1", "call_2", "call_3", "call_4", "call_5", "call_tail");
    }

    enum Inclusions {
        BREAKFAST,
        LUNCH,
        DINNER,
        MEALS,
        ALL_INCLUSIVE
    }
}
