package ru.yandex.travel.hotels.searcher.partners;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;

@PartnerBean(EPartnerId.PI_OSTROVOK)
@EnableConfigurationProperties(OstrovokTaskHandlerProperties.class)
public class OstrovokTaskHandler extends AbstractPartnerTaskHandler<OstrovokTaskHandlerProperties> {
    private static final ImmutableMap<String, EPansionType> PANSION_TYPE_CONVERSIONS = ImmutableMap.<String,
            EPansionType>builder()
            .put("", EPansionType.PT_RO)
            .put("RO", EPansionType.PT_RO)
            .put("BB", EPansionType.PT_BB)
            .put("HB", EPansionType.PT_HB)
            .put("FB", EPansionType.PT_FB)
            .put("AI", EPansionType.PT_AI)
            .build();
    private final Counter duplicateOfferCounter;

    OstrovokTaskHandler(OstrovokTaskHandlerProperties config) {
        super(config);
        duplicateOfferCounter = Metrics.counter("searcher.partners.ostrovok.duplicateOffers");
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        CompletableFuture<Response> responseFuture = CompletableFuture
                .supplyAsync(() -> buildRequest(groupingKey,
                        tasks,
                        Utils.createBasicAuthHeader(config.getUsername(), config.getPassword()),
                        requestId
                ), executor)
                .thenCompose(request -> runHttpRequest(request, true, "main", groupingKey.getRequestClass()));
        return responseFuture.thenAccept(response -> processResponses(tasks, response));
    }

    private Request buildRequest(Task.GroupingKey key, List<Task> tasks, String secret, String requestId) {
        JsonObject jData = new JsonObject();
        Occupancy occupancy = key.getOccupancy();
        jData.addProperty("checkin", key.getCheckInDate());
        jData.addProperty("checkout", key.getCheckOutDate());
        jData.addProperty("adults", occupancy.getAdults());
        if (!occupancy.getChildren().isEmpty()) {
            JsonArray jChildren = new JsonArray();
            for (Integer age : occupancy.getChildren()) {
                jChildren.add(age);
            }
            jData.add("children", jChildren);
        }
        jData.addProperty("currency", Utils.CURRENCY_NAMES.get(key.getCurrency()));
        jData.addProperty("limit", config.getMaxBatchSize());
        JsonArray ids = new JsonArray();
        tasks.stream().map(t -> t.getRequest().getHotelId().getOriginalId()).distinct().forEach(ids::add);
        jData.add("ids", ids);
        return buildHttpRequest(tasks, requestId)
                .setHeader("Authorization", secret)
                .addQueryParam("data", jData.toString())
                .build();
    }

    private void processResponses(List<Task> tasks, Response response) {
        Map<String, Map<MatchingKey, TOffer.Builder>> offers = parseOffers(tasks,
                Objects.requireNonNull(response).getResponseBody());
        Map<String, Map<MatchingKey, TOffer.Builder>> fencedOffers = null;
        for (Task task : tasks) {
            if (!offers.containsKey(task.getId())) {
                continue;
            }
            for (Map.Entry<MatchingKey, TOffer.Builder> entry : offers.get(task.getId()).entrySet()) {
                TOffer.Builder offer = entry.getValue();
                onOffer(task, TOffer.newBuilder(offer.build()));
            }
        }
    }

    private Map<String/*taskId*/, Map<MatchingKey, TOffer.Builder>> parseOffers(List<Task> tasks, String responseBody) {
        Map<String, List<Task>> tasksByOriginalId = mapTasksByOriginalId(tasks);
        Map<String, Map<MatchingKey, TOffer.Builder>> resultMap = new HashMap<>();
        JsonParser parser = new JsonParser();
        JsonObject jBodyObj = parser.parse(responseBody).getAsJsonObject();
        JsonElement jResultEl = jBodyObj.get("result");
        if (jResultEl.isJsonNull()) {
            String errorText;
            JsonElement jErrorEl = jBodyObj.get("error");
            if (jErrorEl != null) {
                errorText = jErrorEl.getAsJsonObject().get("message").getAsString();
            } else {
                errorText = "?";
            }
            throw new RuntimeException("Response with error: '" + errorText + "'");
        }
        JsonArray jHotels = jResultEl.getAsJsonObject().get("hotels").getAsJsonArray();
        for (JsonElement jHotelEl : jHotels) {
            String originalId = jHotelEl.getAsJsonObject().get("id").getAsString();
            List<Task> taskList = tasksByOriginalId.get(originalId);
            if (taskList == null) {
                logger.warn("Unknown hotel_id '{}'", originalId);
                continue;
            }
            taskList.forEach(task -> jHotelEl.getAsJsonObject().getAsJsonArray("rates").forEach(jRateEl -> {
                Map<MatchingKey, TOffer.Builder> result = resultMap.computeIfAbsent(task.getId(),
                        k -> new LinkedHashMap<>());
                JsonObject jRate = jRateEl.getAsJsonObject();
                String expectedCurrency = Utils.CURRENCY_NAMES.get(task.getRequest().getCurrency());
                String actualCurrency = jRate.get("rate_currency").getAsString();
                if (!actualCurrency.equals(expectedCurrency)) {
                    throw new RuntimeException("Expected currency is " + expectedCurrency + ", but got " + actualCurrency);
                }

                long priceAmount = Math.round(Double.parseDouble(jRate.get("rate_price").getAsString()));
                if (!validatePrice(task, priceAmount)) {
                    return;
                }

                TOffer.Builder o = TOffer.newBuilder();
                String capacity = Capacity.fromOccupancy(task.getOccupancy()).toString();
                o.setCapacity(capacity);
                o.setSingleRoomCapacity(capacity);
                o.setRoomCount(1);
                o.setOperatorId(EOperatorId.OI_OSTROVOK);
                o.setExternalId(jRate.get("availability_hash").getAsString());
                o.setPrice(TPriceWithDetails.newBuilder()
                        .setCurrency(task.getRequest().getCurrency())
                        .setAmount((int) priceAmount)
                        .build());
                o.setLandingInfo(TOfferLandingInfo.newBuilder()
                        .setLandingPageUrl(jRate.get("hotelpage").getAsString())
                        .build());
                o.setOriginalRoomId(Integer.toString(jRate.get("room_group_id").getAsInt()));
                o.setAvailabilityGroupKey(jRate.get("room_type_id").getAsString());
                o.setDisplayedTitle(StringValue.of(jRate.get("room_name").getAsString()));
                String ptStr = jRate.get("mealcode").getAsString();
                EPansionType pt = PANSION_TYPE_CONVERSIONS.get(ptStr);
                if (pt == null) {
                    pt = EPansionType.PT_UNKNOWN;
                    logger.error("Unknown pansion '{}'", ptStr);
                }
                o.setPansion(pt);

                boolean freeCancel = false;
                JsonElement jCancInfoEl = jRate.get("cancellation_info");
                RefundRules refundRules = null;
                if (jCancInfoEl != null) {
                    // Completely copied from old tours
                    JsonElement jBeforeEl = jCancInfoEl.getAsJsonObject().get("free_cancellation_before");
                    if (jBeforeEl != null && !jBeforeEl.isJsonNull()) {
                        String beforeStr = jBeforeEl.getAsString();
                        Instant freeCancelBefore = LocalDateTime.parse(beforeStr).toInstant(ZoneOffset.of(config.getDefaultZoneOffset()));
                        freeCancel = freeCancelBefore.isAfter(Instant.now());
                        if (freeCancel) {
                            refundRules = RefundRules.builder()
                                    .rule(RefundRule.builder()
                                            .type(RefundType.FULLY_REFUNDABLE)
                                            .endsAt(freeCancelBefore)
                                            .build())
                                    .build();
                        }
                    }
                }
                MatchingKey matchingKey = new MatchingKey(jRate);
                o.setFreeCancellation(BoolValue.of(freeCancel));
                if (refundRules != null) {
                    o.addAllRefundRule(mapToProtoRefundRules(refundRules));
                }
                if (result.get(matchingKey) != null) {
                    logger.warn(String.format("Offer with matching key '%s' is duplicated for task '%s'", matchingKey.toString(), task.getId()));
                    duplicateOfferCounter.increment();
                    if (result.get(matchingKey).getPrice().getAmount() > o.getPrice().getAmount()) {
                        logger.warn("Duplicate offer is cheaper than the one which is already known");
                        result.put(matchingKey, o);
                    }
                } else {
                    result.put(matchingKey, o);
                }
            }));
        }
        return resultMap;
    }

    @Override
    protected List<String> getHttpCallPurposes() {
        return Arrays.asList("main", "fenced");
    }

    private static class MatchingKey {
        private final String availabilityHash;
        private final String mealcode;
        private final int roomGroupId;
        private final String roomTypeId;
        private int childCotCount = 0;
        private int extraCount = 0;
        private int mainCount = 0;
        private int sharedWithChildrenCount = 0;
        private boolean refundable = false;
        private String freeCancellationDate = "";
        private Set<String> bedTypes = new HashSet<>();
        private Set<String> valueAdds = new HashSet<>();

        public MatchingKey(JsonObject rate) {
            availabilityHash = rate.get("availability_hash").getAsString();
            mealcode = rate.get("mealcode").getAsString();
            roomGroupId = rate.get("room_group_id").getAsInt();
            roomTypeId = rate.get("room_type_id").getAsString();
            rate.get("bed_types").getAsJsonArray().forEach(je -> bedTypes.add(je.getAsString()));

            if (rate.has("bed_places")) {
                JsonObject bedPlaces = rate.get("bed_places").getAsJsonObject();
                childCotCount = bedPlaces.get("child_cot_count").getAsInt();
                extraCount = bedPlaces.get("extra_count").getAsInt();
                mainCount = bedPlaces.get("main_count").getAsInt();
                sharedWithChildrenCount = bedPlaces.get("shared_with_children_count").getAsInt();
            }

            if (rate.has("cancellation_info")) {
                JsonObject cancellationInfo = rate.get("cancellation_info").getAsJsonObject();
                freeCancellationDate = cancellationInfo.get("free_cancellation_before").isJsonNull() ? null :
                        cancellationInfo.get("free_cancellation_before").getAsString();
                refundable = cancellationInfo.get("refundable").getAsBoolean();
            }

            if (rate.has("value_adds")) {
                rate.get("value_adds").getAsJsonArray().forEach(je ->
                        valueAdds.add(je.getAsJsonObject().get("code").getAsString()));
            }
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            MatchingKey that = (MatchingKey) o;
            return childCotCount == that.childCotCount &&
                    extraCount == that.extraCount &&
                    mainCount == that.mainCount &&
                    sharedWithChildrenCount == that.sharedWithChildrenCount &&
                    refundable == that.refundable &&
                    roomGroupId == that.roomGroupId &&
                    Objects.equals(availabilityHash, that.availabilityHash) &&
                    Objects.equals(freeCancellationDate, that.freeCancellationDate) &&
                    Objects.equals(mealcode, that.mealcode) &&
                    Objects.equals(roomTypeId, that.roomTypeId) &&
                    Objects.equals(bedTypes, that.bedTypes) &&
                    Objects.equals(valueAdds, that.valueAdds);
        }

        @Override
        public int hashCode() {
            return Objects.hash(
                    availabilityHash,
                    childCotCount,
                    extraCount,
                    mainCount,
                    sharedWithChildrenCount,
                    refundable,
                    freeCancellationDate,
                    mealcode,
                    roomGroupId,
                    roomTypeId,
                    bedTypes,
                    valueAdds);
        }

        @Override
        public String toString() {
            return "MatchingKey{" +
                    "availabilityHash='" + availabilityHash + '\'' +
                    ", childCotCount=" + childCotCount +
                    ", extraCount=" + extraCount +
                    ", mainCount=" + mainCount +
                    ", sharedWithChildrenCount=" + sharedWithChildrenCount +
                    ", refundable=" + refundable +
                    ", freeCancellationDate='" + freeCancellationDate + '\'' +
                    ", mealcode='" + mealcode + '\'' +
                    ", roomGroupId=" + roomGroupId +
                    ", roomTypeId='" + roomTypeId + '\'' +
                    ", bedTypes=" + bedTypes +
                    ", valueAdds=" + valueAdds +
                    '}';
        }
    }
}
