package ru.yandex.travel.hotels.searcher.partners;

import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.metrics.MetricsUtils;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.hotels.common.partners.base.exceptions.PartnerException;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineRefundRulesBuilder;
import ru.yandex.travel.hotels.common.partners.travelline.exceptions.CacheNotFoundException;
import ru.yandex.travel.hotels.common.partners.travelline.model.AgeGroup;
import ru.yandex.travel.hotels.common.partners.travelline.model.BookingMode;
import ru.yandex.travel.hotels.common.partners.travelline.model.GuestPlacementKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInfo;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.Placement;
import ru.yandex.travel.hotels.common.partners.travelline.model.PlacementRate;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlan;
import ru.yandex.travel.hotels.common.partners.travelline.model.RatePlanKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomStay;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomType;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomTypeQuota;
import ru.yandex.travel.hotels.common.partners.travelline.model.RoomTypeUnitKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.Service;
import ru.yandex.travel.hotels.common.partners.travelline.model.ServiceKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.StayUnitKind;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.HotelDto;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.HotelInfoMappings;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.OfferDto;
import ru.yandex.travel.hotels.common.partners.travelline.model.dto.ServiceDto;
import ru.yandex.travel.hotels.common.partners.travelline.placements.Allocation;
import ru.yandex.travel.hotels.common.partners.travelline.placements.InvalidPlacementAllocationException;
import ru.yandex.travel.hotels.common.partners.travelline.placements.PlacementGenerator;
import ru.yandex.travel.hotels.common.partners.travelline.placements.PlacementSet;
import ru.yandex.travel.hotels.common.partners.travelline.utils.RoomMatchingHelper;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ESearchWarningCode;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPartnerSpecificOfferData;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.proto.TTravellineOffer;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;
import ru.yandex.travel.hotels.searcher.services.TravelTokenService;
import ru.yandex.travel.hotels.searcher.services.cache.Actualizable;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.TravellineAvailabilitySearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.hotels.TravellineHotelDataSearcher;

@PartnerBean(EPartnerId.PI_TRAVELLINE)
@EnableConfigurationProperties(TravellineTaskHandlerProperties.class)
@Slf4j
class TravellineTaskHandler extends AbstractPartnerTaskHandler<TravellineTaskHandlerProperties> {

    public static final String WIFI_AMENITY_KIND = "wifi_internet";
    public static final String WIFI_FEATURE_NAME = "WiFi в номере";
    public static final String WIFI = "wifi";
    public static final String WIFIRUS = "вайфай";
    private static final int MAX_OFFER_LENGTH = 150;
    private static final String HOTEL_INFO = "hotelInfo";
    private static final String HOTEL_OFFER_AVAILABILITY = "hotelOfferAvailability";
    private final TravelTokenService travelTokenService;
    private final Counter incorrectCancellationPenaltyCounter;
    private final Counter incorrectRoomPlacementsCounter;
    private final Counter missingServiceInfo;
    private final Counter droppedApartments;
    private final Counter droppedSharedRatePlans;
    private final Counter droppedUnknownRatePlans;
    private final Counter droppedNonYandexRatePlans;
    private final Counter droppedRequiresChildren;
    private final Timer allocationGenerationTimer;
    private final Timer placementMappingTimer;
    private final TravellineAvailabilitySearcher offerAvailabilitySearcher;
    private final TravellineHotelDataSearcher hotelInfoSearcher;


    TravellineTaskHandler(TravellineTaskHandlerProperties config, TravelTokenService travelTokenService,
                          TravellineAvailabilitySearcher offerAvailabilitySearcher,
                          TravellineHotelDataSearcher hotelInfoSearcher) {
        super(config);
        this.travelTokenService = travelTokenService;
        this.offerAvailabilitySearcher = offerAvailabilitySearcher;
        this.hotelInfoSearcher = hotelInfoSearcher;

        incorrectCancellationPenaltyCounter = Metrics.counter("searcher.partners.travelline" +
                ".incorrectCancellationPenalties");
        incorrectRoomPlacementsCounter = Metrics.counter("searcher.partners.travelline" +
                ".incorrectRoomPlacements");
        missingServiceInfo = Metrics.counter("searcher.partners.travelline.missingServiceInfo");
        droppedApartments = Metrics.counter("searcher.partners.travelline.droppedApartments");
        droppedSharedRatePlans = Metrics.counter("searcher.partners.travelline.droppedRatePlans", "reason", "shared");
        droppedUnknownRatePlans = Metrics.counter("searcher.partners.travelline.droppedRatePlans", "reason", "unknown");
        droppedRequiresChildren = Metrics.counter("searcher.partners.travelline.droppedRatePlans", "reason",
                "children");
        droppedNonYandexRatePlans = Metrics.counter("searcher.partners.travelline.droppedRatePlans",
                "reason", "non-yandex");
        allocationGenerationTimer = Timer.builder("searcher.partners.travelline.placements.allocations")
                .serviceLevelObjectives(MetricsUtils.smallDurationSla())
                .publishPercentiles(MetricsUtils.higherPercentiles()).register(Metrics.globalRegistry);
        placementMappingTimer = Timer.builder("searcher.partners.travelline.placements.mapping")
                .serviceLevelObjectives(MetricsUtils.smallDurationSla())
                .publishPercentiles(MetricsUtils.higherPercentiles()).register(Metrics.globalRegistry);

    }

    private OfferDto getOfferDto(Task task, HotelInfo hotelInfo,
                                 RoomStay roomStay,
                                 RoomStay.RoomTypeRef roomTypeRef,
                                 RoomStay.RatePlanRef ratePlanRef,
                                 Occupancy occupancy,
                                 HotelInfoMappings hotelInfoMappings,
                                 Map<String, HotelOfferAvailability.ServiceConditions> rphToServiceConditionsMap,
                                 Map<String, Integer> rphToRoomTypeQuotaMap,
                                 boolean dataMayBeStale) {
        HotelDto hotelDto = HotelDto.builder()
                .code(hotelInfo.getHotel().getCode())
                .name(hotelInfo.getHotel().getName())
                .description(hotelInfo.getHotel().getDescription())
                .images(hotelInfo.getHotel().getImages())
                .stars(hotelInfo.getHotel().getStars())
                .contactInfo(hotelInfo.getHotel().getContactInfo())
                .policy(hotelInfo.getHotel().getPolicy())
                .timezone(hotelInfo.getHotel().getTimezone())
                .stayUnitKind(hotelInfo.getHotel().getStayUnitKind())
                .build();

        RatePlan ratePlan = hotelInfoMappings.getRatePlan(ratePlanRef.getCode());
        RoomType roomType = hotelInfoMappings.getRoomType(roomTypeRef.getCode());
        if (roomType.getKind() == RoomTypeUnitKind.APARTMENT && (config.getApartmentExceptions() == null ||
                !config.getApartmentExceptions().contains(hotelInfo.getHotel().getCode()))) {
            if (!RoomMatchingHelper.checkRoomMatchesHotel(roomType, hotelInfo.getHotel())) {
                droppedApartments.increment();
                task.onWarning(ESearchWarningCode.SW_TL_DROPPED_APARTMENTS);
                return null;
            }
        }
        if (ratePlan.getKind() == RatePlanKind.SHARED && ratePlan.isTreatment()) {
            log.warn("Dropping shared offer to hotel {}, rate plan {}: " +
                            "rate plan has treatment",
                    hotelInfo.getHotel().getCode(), ratePlan.getCode());
            droppedSharedRatePlans.increment();
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_RATEPLAN_WITH_TREATMENT);
            return null;
        }
        if (ratePlan.getKind() == RatePlanKind.UNKNOWN) {
            log.warn("Dropping offer to hotel {}, rate plan {}: rate plan is unknown", hotelInfo.getHotel().getCode(),
                    ratePlan.getCode());
            droppedUnknownRatePlans.increment();
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_UNKNOWN_RATE_PLAN);
            return null;
        }
        if (ratePlan.getSources() == null || !ratePlan.getSources().contains("yandex")) {
            log.warn("Dropping offer to hotel {}, rate plan {}: rate plan is not enabled for Yandex",
                    hotelInfo.getHotel().getCode(),
                    ratePlan.getCode());
            droppedNonYandexRatePlans.increment();
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_NON_YANDEX_RATE_PLAN);
            return null;
        }
        if (ratePlan.getKind() == RatePlanKind.SHARED && ratePlan.isWithChildrenOnly() && occupancy.getChildren().size() == 0) {
            droppedRequiresChildren.increment();
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_REQUIRES_CHILDREN);
            return null;
        }
        Map<String, ServiceDto> services = roomStay.getServices().stream()
                .map(serviceRef -> {
                    HotelOfferAvailability.ServiceConditions conditions =
                            rphToServiceConditionsMap.get(serviceRef.getRph());
                    Preconditions.checkNotNull(conditions,
                            "ServiceCondition are not set for rph " + serviceRef.getRph());
                    try {
                        Service serviceByCode = hotelInfoMappings.getServiceByCode(conditions.getCode());
                        return ServiceDto.builder()
                                .conditions(conditions)
                                .info(serviceByCode)
                                .build();
                    } catch (HotelInfoMappings.MissingHotelDataException ex) {
                        if (!dataMayBeStale) {
                            missingServiceInfo.increment();
                        }
                        if (conditions.isInclusive() || dataMayBeStale) {
                            throw ex;
                        } else {
                            return null;
                        }
                    }
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(dto -> dto.getInfo().getCode(), Function.identity()));

        Preconditions.checkNotNull(roomStay.getPlacementRates(), "Placement rates are not set");
        Map<Tuple2<String, GuestPlacementKind>, PlacementRate> placementRatesByCodeAndKind =
                roomStay.getPlacementRates().stream()
                        .filter(pr -> pr.getRatePlanCode().equals(ratePlan.getCode()) && pr.getRoomTypeCode().equals(roomType.getCode()))
                        .collect(Collectors.toMap(pr ->
                                        Tuple2.tuple(pr.getPlacement().getCode(), pr.getPlacement().getKind()),
                                Function.identity()));

        List<Placement> placementsWithDailyRate = roomTypeRef.getPlacements().stream()
                .map(placement -> placement.toBuilder().rates(placementRatesByCodeAndKind.computeIfAbsent(
                        Tuple2.tuple(placement.getCode(), placement.getKind()),
                        c -> {
                            throw new RuntimeException("PlacementRate is not available for placement " + c);
                        }
                ).getRates()).build())
                .collect(Collectors.toList());

        long numDates = ChronoUnit.DAYS.between(roomStay.getStayDates().getStartDate().toLocalDate(),
                roomStay.getStayDates().getEndDate().toLocalDate());
        if (hotelInfo.getHotel().getStayUnitKind() == StayUnitKind.DAY) {
            numDates += 1;
        }
        final long finalNumDates = numDates;
        placementsWithDailyRate.forEach(p -> {
            if (p.getRates().size() != finalNumDates) {
                throw new RuntimeException("Unexpected PlacementRate breakdown for placement " + p.getCode());
            }
        });

        List<AgeGroup> noBedAgeGroups = placementsWithDailyRate.stream()
                .filter(p -> p.getKind() == GuestPlacementKind.CHILD_BAND_WITHOUT_BED)
                .map(p -> hotelInfoMappings.getAgeGroupsByCode(String.valueOf(p.getAgeGroup())))
                .collect(Collectors.toList());

        log.debug("Task {}: Allocating placements for occupancy {} for room {}", task.getId(), occupancy.toString(),
                roomType.getName());
        Timer.Sample started = Timer.start(Metrics.globalRegistry);
        Map<Allocation.Key, List<Allocation>> allocationsByKey =
                PlacementGenerator.generateAllocation(occupancy,
                                roomType.getMaxAdultOccupancy(),
                                roomType.getMaxExtraBedOccupancy(),
                                roomType.getMaxWithoutBedOccupancy(),
                                noBedAgeGroups)
                        .stream().collect(Collectors.groupingBy(Allocation::getKey,
                                Collectors.toList()));
        started.stop(allocationGenerationTimer);

        if (allocationsByKey.isEmpty()) {
            log.debug("Unable to allocate placements for occupancy {} in room {}", occupancy.toString(),
                    roomType.getName());
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_INVALID_ALLOCATION);
            return null;
        }
        List<PlacementSet> possiblePlacements = new ArrayList<>();
        Double bestPrice = null;
        log.debug("Task {}: mapping placements for generated {} allocations for room {}", task.getId(),
                allocationsByKey.size(), roomType.getName());
        for (List<Allocation> allocations : allocationsByKey.values()) {
            try {
                var bestPlacement = allocations.stream()
                        .map(allocation -> {
                            var placementTimer = Timer.start(Metrics.globalRegistry);
                            var ps = PlacementGenerator.mapPlacementsForAllocation(
                                    placementsWithDailyRate,
                                    allocation,
                                    hotelInfoMappings.getAgeGroupsByAge(),
                                    occupancy,
                                    ratePlan.getKind() == RatePlanKind.SHARED,
                                    hotelInfo.getHotel().getBookingModes().contains(BookingMode.CALCULATE_ADULTS_SEPARATELY_FROM_CHILDREN),
                                    config.getAlwaysAllowChildrenAsAdults());
                            placementTimer.stop(placementMappingTimer);
                            return ps;
                        })
                        .min(Comparator.comparingDouble(PlacementSet::getTotalCost))
                        .orElseThrow();
                try {
                    var refundRules =
                            TravellineRefundRulesBuilder.build(ratePlanRef.getCancelPenaltyGroup(),
                                    bestPlacement.getPlacements(), ratePlan.getCode(), roomStay.getStayDates(),
                                    ZoneOffset.of(hotelInfo.getHotel().getTimezone().getOffset()),
                                    (int) Math.round(bestPlacement.getTotalCost()));
                    bestPlacement = bestPlacement.toBuilder()
                            .refundRules(refundRules)
                            .build();

                } catch (PartnerException e) {
                    incorrectCancellationPenaltyCounter.increment();
                    task.onWarning(ESearchWarningCode.SW_TL_DROPPED_INVALID_CANCELLATION_PENALTIES);
                    log.warn("CancelPenaltyGroup is incorrect", e);
                    continue;
                }
                if (bestPrice == null) {
                    bestPrice = bestPlacement.getTotalCost();
                } else {
                    bestPrice = Math.min(bestPrice, bestPlacement.getTotalCost());
                }
                possiblePlacements.add(bestPlacement);
            } catch (InvalidPlacementAllocationException ex) {
                log.warn(ex.getMessage());
                continue;
            }
        }
        if (possiblePlacements.size() == 0) {
            log.warn("Incorrect room placements for hotel {} room {} : " +
                            "allocations {} were found for occupancy {}, " +
                            "but non of them match the available places of the room stay ({})",
                    hotelInfo.getHotel().getCode(), roomTypeRef.getCode(), allocationsByKey.keySet(),
                    occupancy.toString(),
                    roomTypeRef.getPlacements());
            incorrectRoomPlacementsCounter.increment();
            task.onWarning(ESearchWarningCode.SW_TL_DROPPED_NO_PLACEMENTS);
            return null;
        }
        return OfferDto.builder()
                .ratePlan(ratePlan)
                .roomType(roomType)
                .stayDates(roomStay.getStayDates())
                .services(services)
                .quota(rphToRoomTypeQuotaMap.get(roomTypeRef.getRoomTypeQuotaRph()))
                .hotel(hotelDto)
                .possiblePlacements(possiblePlacements)
                .build();
    }

    private List<OfferDto> getAllOffers(Task task, HotelOfferAvailability offerAvailability, HotelInfo hotelInfo,
                                        boolean dataMayBeStale) {
        log.info("Task {}: building all offers out of availability response and hotel info", task.getId());
        var results = new ArrayList<OfferDto>();
        if (offerAvailability.getRoomStays().size() == 0) {
            return Collections.emptyList();
        }
        var hotelInfoMappings = new HotelInfoMappings(hotelInfo);
        Map<String, HotelOfferAvailability.ServiceConditions> rphToServiceConditionsMap =
                offerAvailability.getServices()
                        .stream()
                        .collect(Collectors.toMap(
                                HotelOfferAvailability.ServiceConditions::getRph,
                                Function.identity())
                        );
        Map<String, Integer> rphToRoomTypeQuotaMap =
                offerAvailability.getRoomTypeQuotas()
                        .stream()
                        .collect(Collectors.toMap(
                                RoomTypeQuota::getRph, RoomTypeQuota::getQuantity));

        for (var roomStay : offerAvailability.getRoomStays()) {
            Preconditions.checkArgument(roomStay.getRatePlans().size() == 1, "Too many rate plans");
            Preconditions.checkArgument(roomStay.getRoomTypes().size() == 1, "Too many room types");
            for (RoomStay.RoomTypeRef roomTypeRef : roomStay.getRoomTypes()) {
                for (RoomStay.RatePlanRef ratePlanRef : roomStay.getRatePlans()) {
                    try {
                        OfferDto offerDto = getOfferDto(task, hotelInfo, roomStay,
                                roomTypeRef, ratePlanRef, task.getOccupancy(), hotelInfoMappings,
                                rphToServiceConditionsMap, rphToRoomTypeQuotaMap, dataMayBeStale);
                        if (offerDto == null) {
                            continue;
                        }
                        results.add(offerDto);
                    } catch (HotelInfoMappings.MissingHotelDataException ex) {
                        if (dataMayBeStale) {
                            throw ex;
                        }
                        log.error(ex.getMessage());
                        task.onWarning(ESearchWarningCode.SW_TL_DROPPED_MISSING_HOTEL_DATA);
                        continue;
                    }
                }
            }
        }
        log.info("Task {}: done building {} offers", task.getId(), results.size());
        return results;
    }

    private void processResults(Task task, HotelOfferAvailability offerAvailability,
                                Actualizable<HotelInfo> hotelInfo, boolean dataMayBeStale) {
        var all = getAllOffers(task, offerAvailability, hotelInfo.getCached(), dataMayBeStale);
        all.forEach(offerDto -> {
            var travellineOffer = TTravellineOffer.newBuilder()
                    .setTravellineOfferDTO(ProtoUtils.toTJson(offerDto, "2"))
                    .build();

            String offerId = ProtoUtils.randomId();
            TPriceWithDetails offerPrice = TPriceWithDetails.newBuilder()
                    .setCurrency(ECurrency.C_RUB)
                    .setAmount((int) Math.round(offerDto.getBestPrice()))
                    .build();
            var hotelAddress = offerDto.getHotel().getContactInfo().getAddresses().get(0);
            var protoRefundRules = mapToProtoRefundRules(offerDto.getRefundRules());
            TCoordinates coordinates = TCoordinates.newBuilder()
                    .setLongitude(hotelAddress.getLongitude())
                    .setLatitude(hotelAddress.getLatitude())
                    .build();
            String travelTokenString = travelTokenService.storeTravelTokenAndGetItsString(offerId, partnerId, task,
                    offerDto.getRoomType().getCode(), offerDto.getPansionType(),
                    offerDto.getRefundRules().actualize().isFullyRefundable(), offerPrice,
                    coordinates, protoRefundRules,
                    TOfferData.newBuilder().setTravellineOffer(travellineOffer));
            String capacity = Capacity.fromOccupancy(task.getOccupancy()).toString();
            List<String> services = new ArrayList<>();

            boolean hasWifi = offerDto.getRoomType().getAmenities().stream()
                    .anyMatch(a -> WIFI_AMENITY_KIND.equals(a.getKind()));
            AtomicBoolean wifiServiceFromHotel = new AtomicBoolean(false);

            offerDto.getServices().values().stream()
                    .filter(s -> s.getConditions().isInclusive() && s.getInfo().getKind() != ServiceKind.MEAL)
                    .map(s -> s.getInfo().getName())
                    .forEach(serviceName -> {
                        String ns = serviceName.toLowerCase().replaceAll("-", "");
                        if (ns.contains(WIFI) || ns.contains(WIFIRUS)) {
                            wifiServiceFromHotel.set(true);
                        }
                        services.add(serviceName);
                    });
            if (hasWifi && !wifiServiceFromHotel.get()) {
                services.add(WIFI_FEATURE_NAME);
            }
            StringBuilder fullNameBuilder = new StringBuilder(offerDto.getRoomType().getName());
            for (String service : services.stream().sorted().collect(Collectors.toList())) {
                String ext = " • " + service;
                if (fullNameBuilder.length() + ext.length() <= MAX_OFFER_LENGTH) {
                    fullNameBuilder.append(ext);
                }
            }

            TOffer.Builder taskOfferBuilder = TOffer.newBuilder()
                    .setId(offerId)
                    .setDisplayedTitle(StringValue.of(fullNameBuilder.toString()))
                    .setAvailability(offerDto.getQuota())
                    .setCapacity(capacity)
                    .setSingleRoomCapacity(capacity)
                    .setRoomCount(1)
                    .setOriginalRoomId(offerDto.getRoomType().getCode())
                    .setPansion(offerDto.getPansionType())
                    .setFreeCancellation(BoolValue.of(offerDto.getRefundRules().actualize().isFullyRefundable()))
                    .addAllRefundRule(protoRefundRules)
                    .setWifiIncluded(BoolValue.of(hasWifi))
                    .setOperatorId(EOperatorId.OI_TRAVELLINE)
                    .setPartnerSpecificData(TPartnerSpecificOfferData.newBuilder()
                            .setTravellineData(TPartnerSpecificOfferData.TTravellineData.newBuilder()
                                    .setHotelCode(offerDto.getHotel().getCode())
                                    .setRatePlanCode(offerDto.getRatePlan().getCode())
                                    .build())
                            .build())
                    .setPrice(offerPrice)
                    .setLandingInfo(TOfferLandingInfo.newBuilder()
                            .setLandingTravelToken(travelTokenString)
                            .build());
            onOffer(task, taskOfferBuilder);
        });
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        Preconditions.checkArgument(tasks.size() == 1,
                "It is expected to have a single task, actually: " + tasks.size());
        Task task = tasks.get(0);
        CompletableFuture<HotelOfferAvailability> availabilityFuture = FutureUtils.handleExceptionOfType(
                getHotelOfferAvailability(groupingKey, task, requestId),
                CacheNotFoundException.class, t -> {
                    throw new ErrorException(TError.newBuilder()
                            .setCode(EErrorCode.EC_NO_HOTEL_CACHE)
                            .setMessage("Cache not found on partner side")
                            .build());
                });

        CompletableFuture<Actualizable<HotelInfo>> hotelInfoFuture = getHotelInfo(task, requestId);
        return CompletableFuture.allOf(availabilityFuture, hotelInfoFuture)
                .thenComposeAsync(ignored -> {
                    var offerAvailability = availabilityFuture.join();
                    var hotelInfo = hotelInfoFuture.join();
                    try {
                        processResults(task, offerAvailability, hotelInfo, true);
                        return CompletableFuture.completedFuture(null);
                    } catch (HotelInfoMappings.MissingHotelDataException ex) {
                        log.warn("Task {}: Some data is missing in hotel info: {}, will invalidate cached item",
                                task.getId(), ex.getMessage());
                        return hotelInfo.actualize(requestId)
                                .thenAccept(hi -> processResults(task, offerAvailability, hi,
                                        false));
                    }
                }, executor);
    }

    @Override
    protected List<String> getHttpCallPurposes() {
        return List.of(HOTEL_INFO, HOTEL_OFFER_AVAILABILITY);
    }

    private CompletableFuture<HotelOfferAvailability> getHotelOfferAvailability(Task.GroupingKey groupingKey,
                                                                                Task task,
                                                                                String requestId) {
        try {
            LocalDate checkin = LocalDate.parse(groupingKey.getCheckInDate());
            LocalDate checkout = LocalDate.parse(groupingKey.getCheckOutDate());
            return offerAvailabilitySearcher.lookupOffers(task.getId(), task.getRequest().getHotelId().getOriginalId(),
                    checkin, checkout, task.getRequest().getRequestClass(), task.getCallContext(),
                    requestId);
        } catch (DateTimeParseException ex) {
            return CompletableFuture.failedFuture(ex);
        }
    }

    private CompletableFuture<Actualizable<HotelInfo>> getHotelInfo(Task task, String requestId) {
        return hotelInfoSearcher.getHotelData(task.getId(), task.getRequest().getHotelId().getOriginalId(),
                task.getCallContext(), requestId);
    }
}
