package ru.yandex.travel.hotels.searcher.partners;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.partners.tvil.TvilClient;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilOffer;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilSearchRequest;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilSearchRequest.TvilGuests;
import ru.yandex.travel.hotels.common.partners.tvil.model.TvilSearchResponse;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.EOperatorId;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TOffer;
import ru.yandex.travel.hotels.proto.TOfferLandingInfo;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.searcher.Capacity;
import ru.yandex.travel.hotels.searcher.PartnerBean;
import ru.yandex.travel.hotels.searcher.Task;

import static java.util.stream.Collectors.toSet;

@PartnerBean(EPartnerId.PI_TVIL)
public class TvilTaskHandler extends AbstractPartnerTaskHandler<TvilTaskHandlerProperties> {
    private static final ImmutableMap<String, EPansionType> PANSION_TYPE_CONVERSIONS = ImmutableMap.<String,
                    EPansionType>builder()
            .put("Завтрак, обед и ужин", EPansionType.PT_FB)
            .put("Завтрак и ужин", EPansionType.PT_HB)
            .put("Завтрак и обед", EPansionType.PT_BB)
            .put("Обед и ужин", EPansionType.PT_BD)
            .put("Завтрак", EPansionType.PT_BB)
            .put("Обед", EPansionType.PT_RO)
            .put("Ужин", EPansionType.PT_BD)
            .build();

    private static final String ONLY_SUPPORTED_CURRENCY = "RUB";
    private static final String DEFAULT_LANGUAGE = "ru";
    private static final String DEFAULT_USER_COUNTRY = "RU";

    private TvilClient client;

    TvilTaskHandler(TvilClient client, TvilTaskHandlerProperties config) {
        super(config);
        this.client = client;
    }

    @Override
    CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        return client.searchOffers(createRequest(groupingKey, tasks, requestId))
                .thenAccept(response -> processResponse(groupingKey, tasks, response));
    }

    private TvilSearchRequest createRequest(Task.GroupingKey key, List<Task> tasks, String requestId) {
        Occupancy occupancy = key.getOccupancy();
        return TvilSearchRequest.builder()
                .hotelIds(tasks.stream().map(t -> t.getRequest().getHotelId().getOriginalId()).collect(toSet()))
                .checkIn(LocalDate.parse(key.getCheckInDate()))
                .checkOut(LocalDate.parse(key.getCheckOutDate()))
                .guests(List.of(TvilGuests.builder()
                        .adults(occupancy.getAdults())
                        .childrenAges(occupancy.getChildren())
                        .build()))
                .language(DEFAULT_LANGUAGE)
                .currency(ONLY_SUPPORTED_CURRENCY)
                .userCountry(DEFAULT_USER_COUNTRY)
                .requestId(requestId)
                .build();
    }

    private void processResponse(Task.GroupingKey groupingKey, List<Task> tasks, TvilSearchResponse response) {
        Map<String, List<Task>> tasksByOriginalId = mapTasksByOriginalId(tasks);
        for (Map.Entry<String, List<TvilOffer>> responseEntry : response.entrySet()) {
            String hotelId = responseEntry.getKey();
            for (TvilOffer offer : responseEntry.getValue()) {
                List<Task> taskList = tasksByOriginalId.get(hotelId);
                if (taskList == null) {
                    logger.warn("Unknown hotel_id '{}'", hotelId);
                    continue;
                }
                try {
                    TOffer.Builder convertedOffer = convertOffer(groupingKey, taskList, offer);
                    if (convertedOffer == null) {
                        // invalid price
                        continue;
                    }
                    for (Task task : taskList) {
                        onOffer(task, TOffer.newBuilder(convertedOffer.build()));
                    }
                } catch (Throwable ex) {
                    taskList.forEach(t -> {
                        logger.error(String.format("Task %s: failed to convert offer", t.getId()), ex);
                        t.onError(ProtoUtils.errorFromThrowable(ex, t.isIncludeDebug()));
                    });
                }
            }
        }
    }

    private TOffer.Builder convertOffer(Task.GroupingKey groupingKey, List<Task> tasks, TvilOffer offer) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(offer.getUrl()),
                "Empty url for task %s", tasks.get(0).getId());
        int price = (int) Math.ceil(offer.getPrice().floatValue());
        if (!validatePrice(tasks.get(0), price)) {
            return null;
        }
        String capacity = Capacity.fromOccupancy(groupingKey.getOccupancy()).toString();
        String roomId = offer.getRoomId() != null ? String.valueOf(offer.getRoomId()) : null;
        String landingUrl = UriComponentsBuilder.fromUriString(offer.getUrl())
                .queryParam("utm_content", config.getClickOutPartnerId())
                .build().toString();
        String eating = offer.getEating();
        EPansionType pansion = eating == null ? EPansionType.PT_RO : PANSION_TYPE_CONVERSIONS.get(eating);
        if (pansion == null) {
            pansion = EPansionType.PT_UNKNOWN;
            logger.error("Unknown pansion '{}'", eating);
        }
        return TOffer.newBuilder()
                .setOperatorId(EOperatorId.OI_TVIL)
                .setCapacity(capacity)
                .setSingleRoomCapacity(capacity)
                .setRoomCount(1)
                .setDisplayedTitle(StringValue.of(offer.getName()))
                .setPrice(TPriceWithDetails.newBuilder()
                        .setCurrency(ECurrency.C_RUB)
                        .setAmount(price)
                        .build())
                .setLandingInfo(TOfferLandingInfo.newBuilder()
                        .setLandingPageUrl(landingUrl)
                        .build())
                .setPansion(pansion)
                .setAvailability(offer.getAvailable())
                // todo(tlg-13): should be removed when decide what to do with the missing data (TRAVELBACK-732)
                .setFreeCancellation(BoolValue.of(false))
                .setOriginalRoomId(Strings.nullToEmpty(roomId));
    }
}
