package ru.yandex.travel.hotels.searcher.partners

import com.google.protobuf.BoolValue
import com.google.protobuf.StringValue
import org.springframework.boot.context.properties.EnableConfigurationProperties
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit
import ru.yandex.travel.commons.proto.ProtoUtils
import ru.yandex.travel.hotels.common.partners.vipservice.VipserviceClient
import ru.yandex.travel.hotels.common.partners.vipservice.model.Offer
import ru.yandex.travel.hotels.proto.*
import ru.yandex.travel.hotels.searcher.Capacity
import ru.yandex.travel.hotels.searcher.PartnerBean
import ru.yandex.travel.hotels.searcher.Task
import ru.yandex.travel.hotels.searcher.services.TravelTokenService
import java.math.BigDecimal
import java.time.LocalDate
import java.time.ZoneOffset
import java.util.concurrent.CompletableFuture

@PartnerBean(EPartnerId.PI_VIPSERVICE)
@EnableConfigurationProperties(VipserviceTaskHandlerProperties::class)
class VipserviceTaskHandler(
    private val config: VipserviceTaskHandlerProperties,
    private val client: VipserviceClient,
    private val travelTokenService: TravelTokenService,
) : AbstractPartnerTaskHandler<VipserviceTaskHandlerProperties>(config) {

    override fun execute(
        groupingKey: Task.GroupingKey,
        tasks: MutableList<Task>,
        requestId: String,
    ): CompletableFuture<Void> {
        val tasksByHotelId = HashMap<Int, List<Task>>()
        for ((originalId, hotelTasks) in mapTasksByOriginalId(tasks)) {
            val hotelId = try {
                originalId.toInt()
            } catch (_: NumberFormatException) {
                logger.warn("Req $requestId, Invalid hotel_id $originalId")
                continue
            }
            tasksByHotelId[hotelId] = hotelTasks
        }
        return client.findOffers(
            hotelIds = tasksByHotelId.keys.toList(),
            checkIn = LocalDate.parse(groupingKey.checkInDate),
            checkOut = LocalDate.parse(groupingKey.checkOutDate),
            adults = groupingKey.occupancy.adults,
            children = groupingKey.occupancy.children,
            currency = groupingKey.currency,
        ).thenApply { offers ->
            processResponse(requestId, tasksByHotelId, offers)
            return@thenApply null
        }
    }

    private fun processResponse(requestId: String, tasksByHotelId: Map<Int, List<Task>>, offers: Array<Offer>) {
        for (offer in offers) {
            val tasks = tasksByHotelId[offer.hotelId]
            if (tasks == null) {
                logger.warn("Req {}, Unknown hotel_id {}", requestId, offer.hotelId)
                continue
            }

            val price = TPriceWithDetails.newBuilder().also { priceBuilder ->
                priceBuilder.amount = offer.price.toInt()
                priceBuilder.currency = offer.currency.protoCurrency
            }.build()
            val refundRules = extractRefundRules(offer)
            val freeCancellation = refundRules.any { it.type == ERefundType.RT_FULLY_REFUNDABLE }

            for (task in tasks) {
                val offerId = ProtoUtils.randomId()
                val capacity = Capacity.fromOccupancy(task.occupancy).toString()

                val offerBuilder = TOffer.newBuilder()
                offerBuilder.id = offerId
                offerBuilder.externalId = offer.hash
                offerBuilder.operatorId = EOperatorId.OI_VIPSERVICE
                offerBuilder.price = price
                offerBuilder.availability = offer.availableRooms ?: 0
                offerBuilder.capacity = capacity
                offerBuilder.roomCount = 1
                offerBuilder.singleRoomCapacity = capacity
                offerBuilder.landingInfoBuilder.landingTravelToken =
                    makeLandingToken(offerId, task, offer, freeCancellation, price, refundRules)
                offerBuilder.displayedTitle = StringValue.of(offer.roomName)
                offerBuilder.freeCancellation = BoolValue.of(freeCancellation)
                offerBuilder.addAllRefundRule(refundRules)

                // TODO: offerBuilder.pansion

                onOffer(task, offerBuilder)
            }
        }
    }

    private fun makeLandingToken(
        offerId: String,
        task: Task,
        offer: Offer,
        freeCancellation: Boolean,
        price: TPriceWithDetails,
        refundRules: List<TRefundRule>,
    ): String {
        val offerDataBuilder = TOfferData.newBuilder()
            .addAllRefundRule(refundRules)

        val partnerOfferBuilder = offerDataBuilder.vipServiceOfferBuilder
        partnerOfferBuilder.hotelId = offer.hotelId
        partnerOfferBuilder.checkIn = task.groupingKey.checkInDate
        partnerOfferBuilder.checkOut = task.groupingKey.checkOutDate
        partnerOfferBuilder.adults = task.groupingKey.occupancy.adults
        partnerOfferBuilder.addAllChildren(task.groupingKey.occupancy.children)
        partnerOfferBuilder.currency = ProtoCurrencyUnit.fromProtoCurrencyUnit(task.groupingKey.currency).currencyCode
        partnerOfferBuilder.searchItemBuilder.providerId = offer.providerId
        partnerOfferBuilder.searchItemBuilder.hash = offer.hash

        return travelTokenService.storeTravelTokenAndGetItsString(
            offerId,
            EPartnerId.PI_VIPSERVICE,
            task,
            "", // TODO
            EPansionType.PT_UNKNOWN, // TODO
            freeCancellation,
            price,
            null,
            refundRules,
            offerDataBuilder,
        )
    }

    private fun extractRefundRules(offer: Offer): List<TRefundRule> {
        val refundRules = ArrayList<TRefundRule>()
        offer.cancelConditions.freeCancellationBefore?.let {
            refundRules += TRefundRule.newBuilder().also { refundRuleBuilder ->
                refundRuleBuilder.type = ERefundType.RT_FULLY_REFUNDABLE
                refundRuleBuilder.endsAt = ProtoUtils.timestamp(it.toInstant(ZoneOffset.UTC).toEpochMilli())
            }.build()
        }
        for (policy in offer.cancelConditions.policies ?: emptyList()) {
            val refundRuleBuilder = TRefundRule.newBuilder()
            val penalty = policy.penalty

            if (penalty.amount == BigDecimal.ZERO && penalty.percent == 0) {
                if (policy.startAt == null && policy.endAt == offer.cancelConditions.freeCancellationBefore) {
                    continue
                }
                refundRuleBuilder.type = ERefundType.RT_FULLY_REFUNDABLE
            } else if (penalty.amount == offer.price && penalty.currencyCode == offer.currency) {
                refundRuleBuilder.type = ERefundType.RT_NON_REFUNDABLE
            } else {
                // TODO: penalty.percent
                refundRuleBuilder.type = ERefundType.RT_REFUNDABLE_WITH_PENALTY
                refundRuleBuilder.penaltyBuilder.amount = penalty.amount.toLong()
                refundRuleBuilder.penaltyBuilder.currency = penalty.currencyCode.protoCurrency
            }

            policy.startAt?.let {
                refundRuleBuilder.startsAt = ProtoUtils.timestamp(it.toInstant(ZoneOffset.UTC).toEpochMilli())
            }
            policy.endAt?.let {
                refundRuleBuilder.endsAt = ProtoUtils.timestamp(it.toInstant(ZoneOffset.UTC).toEpochMilli())
            }

            refundRules += refundRuleBuilder.build()
        }
        return refundRules
    }
}
