package ru.yandex.travel.hotels.searcher.partners;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;
import com.google.protobuf.BoolValue;
import com.google.protobuf.StringValue;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import org.asynchttpclient.Request;
import org.asynchttpclient.Response;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.pansions.PansionUnifier;
import ru.yandex.travel.hotels.common.partners.Utils;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.proto.*;
import ru.yandex.travel.hotels.searcher.*;

import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

abstract class YandexTravelTaskHandler<T extends YandexTravelTaskHandlerProperties> extends AbstractPartnerTaskHandler<T> {
    private EOperatorId operatorId;
    private final Counter unmatchedPansionOffersCounter;

    YandexTravelTaskHandler(T config, EOperatorId operatorId) {
        super(config);
        this.operatorId = operatorId;
        unmatchedPansionOffersCounter = Metrics.counter("searcher.partners.tasks.unmatchedPansionOffers", "partner", partnerId.toString());
    }

    @Override
    protected CompletableFuture<Void> execute(Task.GroupingKey groupingKey, List<Task> tasks, String requestId) {
        return CompletableFuture
                .supplyAsync(() -> prepareRequest(groupingKey, tasks, requestId), executor)
                .thenCompose(request -> runHttpRequest(request, true, "main", groupingKey.getRequestClass()))
                .thenAccept(response -> processResponse(groupingKey, tasks, response));
    }

    private Request prepareRequest(Task.GroupingKey key, List<Task> tasks, String requestId) {
        Occupancy occupancy = key.getOccupancy();
        String adults = Integer.toString(occupancy.getAdults());
        String children = occupancy.getChildrenAsString(",");
        String nights = getNights(key.getCheckInDate(), key.getCheckOutDate());
        return buildHttpRequest(tasks, requestId)
                .setHeader("Authorization", Utils.createBasicAuthHeader(config.getUsername(), config.getPassword()))
                .addQueryParam("hotel_ids", tasks.stream().map(t -> t.getRequest().getHotelId().getOriginalId()).distinct().collect(Collectors.joining(",")))
                .addQueryParam("date_from", key.getCheckInDate())
                .addQueryParam("nights", nights)
                .addQueryParam("adults", adults)
                .addQueryParam("children", children)
                .build();
    }

    private void processResponse(Task.GroupingKey groupingKey, List<Task> tasks, Response response) {
        Map<String, List<Task>> tasksByOriginalId = mapTasksByOriginalId(tasks);
        JsonParser parser = new JsonParser();
        JsonArray offers = parser.parse(response.getResponseBody()).getAsJsonArray();
        for (JsonElement offer : offers) {
            String hotelId = offer.getAsJsonObject().get("hotel_id").getAsString();
            List<Task> taskList = tasksByOriginalId.get(hotelId);
            if (taskList == null) {
                logger.warn("Unknown hotel_id '{}'", hotelId);
                continue;
            }
            try {
                boolean free_cancellation = offer.getAsJsonObject().get("free_cancellation").getAsBoolean();
                String meal = offer.getAsJsonObject().get("meal").getAsString();
                long price = offer.getAsJsonObject().get("total_price").getAsLong();
                String room = offer.getAsJsonObject().get("room").getAsString();
                String url = offer.getAsJsonObject().get("agent_url").getAsString();
                EPansionType pansionType;
                try {
                    pansionType = PansionUnifier.get(meal);
                } catch (PansionUnifier.UnmatchedPansionException ex) {
                    unmatchedPansionOffersCounter.increment();
                    logger.warn("Unmatched pansion '{}'", meal);
                    pansionType = EPansionType.PT_UNKNOWN;
                }
                if (!validatePrice(taskList.get(0), price)) {
                    continue;
                }
                Capacity capacity;
                if (offer.getAsJsonObject().has("max_occupancy")) {
                    // This is explained here https://HOTELS-3374
                    int maxAdults = offer.getAsJsonObject().get("max_occupancy").getAsInt();
                    capacity = Capacity.fromOccupancyAndAdults(groupingKey.getOccupancy(), maxAdults);
                } else {
                    capacity = Capacity.fromOccupancy(groupingKey.getOccupancy());
                }
                TOffer.Builder builder = TOffer.newBuilder()
                        .setOperatorId(operatorId)
                        .setCapacity(capacity.toString())
                        .setSingleRoomCapacity(capacity.toString())
                        .setRoomCount(1)
                        .setDisplayedTitle(StringValue.of(room))
                        .setFreeCancellation(BoolValue.of(free_cancellation))
                        .setPrice(TPriceWithDetails.newBuilder()
                                .setCurrency(ECurrency.C_RUB)
                                .setAmount((int)price)
                                .build())
                        .setPansion(pansionType)
                        .setLandingInfo(TOfferLandingInfo.newBuilder()
                                .setLandingPageUrl(url)
                                .build());
                if (offer.getAsJsonObject().has("id")) {
                    builder = builder.setOriginalRoomId(offer.getAsJsonObject().get("id").getAsString());
                }
                for (Task task : taskList) {
                    onOffer(task, TOffer.newBuilder(builder.build()));
                }
            } catch (Throwable ex) {
                taskList.forEach(t -> {
                    logger.error(String.format("Task %s: failed to parse offer", t.getId()), ex);
                    t.onError(ProtoUtils.errorFromThrowable(ex, t.isIncludeDebug()));
                });
            }
        }
    }

    private static String getNights(String checkInDate, String checkOutDate) {
        LocalDate ci = LocalDate.parse(checkInDate);
        LocalDate co = LocalDate.parse(checkOutDate);
        return Long.toString(ChronoUnit.DAYS.between(ci, co));
    }
}
