package ru.yandex.travel.hotels.searcher.services;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.FixedListChannelSupplier;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.orders.client.yp.YpChannelSupplier;
import ru.yandex.travel.orders.proto.EServerState;
import ru.yandex.travel.orders.proto.ExchangeRateInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.HADiscoveryInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TPingRpcReq;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@Slf4j
public class ExchangeRateConfiguration {
    @EnableConfigurationProperties(ExchangeRateConfigurationProperties.class)
    @ConditionalOnProperty("exchange.enabled")
    @Configuration
    public static class GrpcExchangeRateConfiguration {
        @Autowired
        private ExchangeRateConfigurationProperties properties;

        @Bean
        public ExchangeRateClientFactory exchangeRateClientFactory(@Qualifier("ExchangeRateChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
            return () -> ExchangeRateInterfaceV1Grpc.newBlockingStub(haGrpcChannelFactory.getMasterChannel());
        }

        @Bean
        public ExchangeRateService exchangeRateService(ExchangeRateClientFactory exchangeRateClientFactory) {
            return new GrpcExchangeRateService(properties, exchangeRateClientFactory);
        }

        @Bean
        @ConditionalOnProperty("exchange.targets")
        @Qualifier("ExchangeRateSupplier")
        public ChannelSupplier exchangeRateFixedListChannelSupplier() {
            log.info("Will use fixed list grpc channel supplier");
            return new FixedListChannelSupplier(properties.getTargets());
        }

        @Bean
        @ConditionalOnMissingBean(name = "ExchangeRateSupplier")
        @Qualifier("ExchangeRateSupplier")
        public ChannelSupplier exchangeRateYpChannelSupplier() {
            log.info("Will use YP discovery grpc channel supplier");
            return new YpChannelSupplier(properties.getYp());
        }

        @Bean
        @Qualifier("ExchangeRateChannelFactory")
        public HAGrpcChannelFactory exchangeRateHAGrpcChannelFactory(@Qualifier("ExchangeRateSupplier") ChannelSupplier channelSupplier, @Autowired(required = false) TvmWrapper tvm) {
            HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
            return builder.withPingProducer(
                    channel -> FutureUtils.buildCompletableFuture(HADiscoveryInterfaceV1Grpc
                            .newFutureStub(channel)
                            .ping(TPingRpcReq.newBuilder().build())
                    ).thenApply((rsp) -> rsp.getState() == EServerState.SS_MASTER ? ChannelState.READY_MASTER : ChannelState.READY))
                    .withFailureDetectorProperties(properties.getFailureDetection())
                    .withChannelSupplier(channelSupplier)
                    .withChannelBuilder(label -> this.createChannel(label, getTvmHelper(tvm)))
                    .build();
        }

        @SneakyThrows
        private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
            String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
            LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                    clientFqdn, target, Set.of(HADiscoveryInterfaceV1Grpc.getPingMethod().getFullMethodName())
            );
            DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                    properties.getTimeout()
            );
            List<ClientInterceptor> interceptors = new ArrayList<>(3);
            interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor));
            if (tvmHeaderClientHelper != null) {
                interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
            }
            return new LabeledChannel(target,
                    ManagedChannelBuilder
                            .forTarget(target)
                            .intercept(interceptors)
                            .usePlaintext()
                            .build());
        }

        private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
            if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
                return new TvmHeaderClientHelper(tvm);
            } else {
                return null;
            }
        }
    }

    @Configuration
    @ConditionalOnMissingBean(GrpcExchangeRateConfiguration.class)
    @Slf4j
    public static class StubbedExchangeRateConfiguration {
        @Bean
        public ExchangeRateService exchangeRateService() {
            log.warn("Will use stubbed exchange rate service");
            return new StubbedExchangeRateService();
        }
    }
}
