package ru.yandex.travel.hotels.searcher.services;

import java.time.Instant;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.searcher.services.entities.ExchangeRate;
import ru.yandex.travel.orders.proto.TGetExchangeRateReq;
import ru.yandex.travel.orders.proto.TGetExchangeRateRsp;


@Slf4j
public class GrpcExchangeRateService implements ExchangeRateService, DisposableBean {

    private final ExchangeRateClientFactory clientFactory;
    private AtomicReference<ExchangeRate> exchangeRate;
    private ScheduledExecutorService refreshService;

    @Autowired
    public GrpcExchangeRateService(ExchangeRateConfigurationProperties config,
                                   ExchangeRateClientFactory clientFactory) {
        this.clientFactory = clientFactory;

        this.exchangeRate = new AtomicReference<>(null);

        this.refreshService = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("exchange-rate-refresh-thread-%d")
                        .build());

        this.refreshService.scheduleAtFixedRate(this::refreshExchangeRate,
                ThreadLocalRandom.current().nextLong(1000L),
                config.getRefreshInterval().toMillis(),
                TimeUnit.MILLISECONDS);
    }

    private void refreshExchangeRate() {

        TGetExchangeRateReq grpcRequest = TGetExchangeRateReq.newBuilder()
                .setFromCurrency(ECurrency.C_USD)
                .setToCurrency(ECurrency.C_RUB)
                .build();
        try {
            TGetExchangeRateRsp grpcResponse = clientFactory.createBlockingStub().getExchangeRate(grpcRequest);
            this.exchangeRate.set(ExchangeRate.builder()
                    .rateValue(grpcResponse.getRateValue())
                    .rateValidUntil(ProtoUtils.toInstant(grpcResponse.getRateValidUntil()))
                    .build());
            log.debug("Successfully updated exchange rate");
        } catch (Exception e) {
            log.error("Failed to update exchange rate", e);
        }
    }

    public Optional<ExchangeRate> getExchangeRate(ECurrency fromCurrency, ECurrency toCurrency) {
        if (fromCurrency == toCurrency) {
            return Optional.of(ExchangeRate.builder().rateValue(1.00f).rateValidUntil(Instant.MAX).build());
        }
        Preconditions.checkArgument(fromCurrency == ECurrency.C_USD && toCurrency == ECurrency.C_RUB, "Currently only USD-RUB exchanges is supported");
        ExchangeRate rate = exchangeRate.get();
        if (rate == null) {
            return Optional.empty();
        }
        return Instant.now().isBefore(rate.getRateValidUntil()) ? Optional.of(rate) : Optional.empty();
    }

    @Override
    public void destroy() throws Exception {
        refreshService.shutdown();
    }
}
