package ru.yandex.travel.hotels.searcher.services;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.List;

import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.TCoordinates;
import ru.yandex.travel.hotels.common.token.Occupancy;
import ru.yandex.travel.hotels.common.token.TokenCodec;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.TOfferData;
import ru.yandex.travel.hotels.proto.TPriceWithDetails;
import ru.yandex.travel.hotels.proto.TRefundRule;
import ru.yandex.travel.hotels.searcher.Task;

@Service
@RequiredArgsConstructor
@Slf4j
public class TravelTokenService {
    private final TokenCodec tokenEncoder;

    public String storeTravelTokenAndGetItsString(String offerId, EPartnerId partnerId, Task task,
                                                  String originalRoomId,
                                                  EPansionType pansionType, Boolean freeCancellation,
                                                  TPriceWithDetails offerPrice, TCoordinates coordinates,
                                                  List<TRefundRule> refundRules,
                                                  TOfferData.Builder offerDataBuilder) {
        if (task.getCallContext().getTestContext() != null) {
            offerDataBuilder.setTestContext(task.getCallContext().getTestContext());
        }

        offerDataBuilder.setPrice(offerPrice);
        offerDataBuilder.addAllRefundRule(refundRules);

        if (coordinates != null) {
            offerDataBuilder.setHolelCoordinates(coordinates);
        }

        return storeTravelTokenAndGetItsString(offerId, partnerId, task, originalRoomId, pansionType, freeCancellation,
                offerDataBuilder.build());
    }

    private String storeTravelTokenAndGetItsString(String offerId, EPartnerId partnerId, Task task,
                                                   String originalRoomId, EPansionType pansionType,
                                                   Boolean freeCancellation, Message payload) {
        var tt = TravelToken.builder()
                .setTokenIdBytes(getTokenIdBytes(payload))
                .setOfferId(offerId)
                .setGeneratedAt(LocalDateTime.now(ZoneOffset.UTC))
                .setCheckInDate(LocalDate.parse(task.getRequest().getCheckInDate()))
                .setCheckOutDate(LocalDate.parse(task.getRequest().getCheckOutDate()))
                .setOccupancy(Occupancy.fromString(task.getRequest().getOccupancy()))
                .setPermalink(task.getRequest().getPermalink())
                .setPartnerId(partnerId)
                .setOriginalId(task.getRequest().getHotelId().getOriginalId())
                .setOriginalRoomId(originalRoomId)
                .setFreeCancellation(freeCancellation)
                .setPansion(pansionType)
                .build();
        log.info("Task {}: will put OfferData {} to KV", task.getId(), tt.getTokenId());
        task.storeMessage(tt.getTokenId(), payload);
        return tokenEncoder.encode(tt);
    }


    private static byte[] getTokenIdBytes(Message message) {
        byte[] bytes = message.toByteArray();
        byte[] digest = new byte[0];
        try {
            digest = MessageDigest.getInstance("SHA-1").digest(bytes);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Unable to get SHA-1 hasher");
        }
        return digest;
    }
}
