package ru.yandex.travel.hotels.searcher.services.cache;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.function.BiFunction;

import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;

import ru.yandex.bolts.function.Function;

public class ActualizableCache<K, T> {
    private final AsyncLoadingCache<K, T> cache;
    private final Counter getCounter;
    private final Counter putCounter;
    private final Counter actualizeCounter;

    private final Function<String, Function<K, T>> supplierBuilder;

    public ActualizableCache(int maxItemsToCache, Duration cacheDuration,
                             BiFunction<K, String, T> supplierBuilder,
                             String counterPrefix, String name) {
        this(maxItemsToCache, cacheDuration,
                (requestId) -> (k) -> supplierBuilder.apply(k, requestId),
                counterPrefix, name);
    }

    public ActualizableCache(int maxItemsToCache, Duration cacheDuration,
                             Function<String, Function<K, T>> supplierBuilder,
                             String counterPrefix, String name) {
        getCounter = Metrics.counter(String.format("%s.%s.get", counterPrefix, name));
        putCounter = Metrics.counter(String.format("%s.%s.put", counterPrefix, name));
        actualizeCounter = Metrics.counter(String.format("%s.%s.actualize", counterPrefix, name));
        cache = Caffeine.newBuilder()
                .maximumSize(maxItemsToCache)
                .expireAfterWrite(cacheDuration)
                .buildAsync(key -> {
                    putCounter.increment();
                    return supplierBuilder.apply(null).apply(key);
                });
        this.supplierBuilder = supplierBuilder;
    }


    public CompletableFuture<Actualizable<T>> get(K key, String httpRequestId) {
        getCounter.increment();
        return cache
                .get(key, supplierBuilder.apply(httpRequestId))
                .thenApply(
                        CachedActualizable.wrap(requestId -> {
                            actualizeCounter.increment();
                            cache.synchronous().invalidate(key);
                            return get(key, requestId);
                        })
                );
    }

    public CompletableFuture<Actualizable<T>> get(K key) {
        return this.get(key, null);
    }
}
