package ru.yandex.travel.hotels.searcher.services.cache;

import java.time.Duration;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.function.BiFunction;
import java.util.function.Function;

import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;

public class SimpleAsyncCache<K, T> {
    private final AsyncLoadingCache<K, T> cache;
    private final Counter getCounter;
    private final Counter putCounter;

    public SimpleAsyncCache(int maxItemsToCache, Duration cacheDuration, Function<K, T> supplier,
                            String counterPrefix, String name) {
        getCounter = Metrics.counter(String.format("%s.%s.get", counterPrefix, name));
        putCounter = Metrics.counter(String.format("%s.%s.put", counterPrefix, name));
        cache = Caffeine.newBuilder()
            .maximumSize(maxItemsToCache)
            .expireAfterWrite(cacheDuration)
            .buildAsync(key -> {
                putCounter.increment();
                return supplier.apply(key);
            });
    }

    public CompletableFuture<T> get(K key) {
        getCounter.increment();
        return cache.get(key);
    }

    public CompletableFuture<Map<K, T>> getAll(Set<K> keys, BiFunction<Iterable<? extends K>, Executor, CompletableFuture<Map<K, T>>> mappingFunction) {
        getCounter.increment(keys.size());

        return cache.getAll(keys, mappingFunction);
    }
}
