package ru.yandex.travel.hotels.searcher.services.cache.bnovo;

import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import ru.yandex.travel.hotels.common.partners.bnovo.BNovoUidMap;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

public class YtBNovoUidMap implements BNovoUidMap, HealthIndicator {
    private static final int STRUCTURE_VERSION = 1;
    private static final String FIELD_ORIGINAL_ID = "o";
    private static final String FIELD_UID = "u";

    private final YtLuceneIndex index;

    public YtBNovoUidMap(YtLuceneIndexParams params) {
        index = new YsonYtLuceneIndex(params, "bnovo-uuid-mapping", STRUCTURE_VERSION, (row) -> {
            Document document = new Document();
            document.add(new StringField(FIELD_ORIGINAL_ID, row.getString("originalId"), Field.Store.NO));
            document.add(new StoredField(FIELD_UID, row.getString("uid")));
            return Collections.singletonList(document);
        });
    }

    @PostConstruct
    public void init() {
        index.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        index.stop();
    }

    @Override
    public Health health() {
        if (index.isReady()) {
            return Health.up().build();
        } else {
            return Health.down().withDetail("MappingNotLoaded", this.getClass()).build();
        }
    }
    @Override
    public String getUidByOriginalId(String originalId) {
        return index.search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_ORIGINAL_ID, originalId));
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                return document.getField(FIELD_UID).stringValue();
            }
            return null;
        });
    }
}
