package ru.yandex.travel.hotels.searcher.services.cache.expedia;

import java.util.Collections;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import ru.yandex.travel.hotels.proto.EPansionType;
import ru.yandex.travel.yt_lucene_index.YsonYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

@Slf4j
public class YtExpediaPropertyPansions implements ExpediaPropertyPansions, HealthIndicator {
    private static final int STRUCTURE_VERSION = 1;
    private static final String FIELD_PROPERTY_ID = "i";
    private static final String FIELD_PANSION = "p";

    private final YtLuceneIndex index;

    public YtExpediaPropertyPansions(YtLuceneIndexParams params) {
        index = new YsonYtLuceneIndex(params, "expedia-property-pansions", STRUCTURE_VERSION, (row) -> {
            Document document = new Document();
            document.add(new StringField(FIELD_PROPERTY_ID, row.getString("propertyId"), Field.Store.NO));
            document.add(new StoredField(FIELD_PANSION, row.getString("pansion")));
            return Collections.singletonList(document);
        });
    }

    @PostConstruct
    public void init() {
        index.start();
    }

    @PreDestroy
    public void destroy() {
        index.stop();
    }

    @Override
    public Health health() {
        if (index.isReady()) {
            return Health.up().build();
        } else {
            return Health.down().withDetail("MappingNotLoaded", this.getClass()).build();
        }
    }

    public EPansionType getPropertyPansion(String propertyId) {
        return index.search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_PROPERTY_ID, propertyId));
            TopDocs topDocs = searcher.search(query, 1);
            if (topDocs.totalHits > 0) {
                Document document = searcher.doc(topDocs.scoreDocs[0].doc);
                var pansion = document.getField(FIELD_PANSION).stringValue();
                try {
                    return EPansionType.valueOf(pansion);
                } catch (IllegalArgumentException | NullPointerException e) {
                    log.error("Failed to parse pansion value: {}", pansion, e);
                    return null;
                }
            }
            return null;
        });
    }
}
