  # Кеширование данных в Серчере Travelline

Формирование офферов для травеллайн на Серчере идет из двух источников: статических данных об отеле, его тарифах и
услугах (возвращаемых методом `getHotelInfo` TL-клиента) и данных о доступности офферов (возвращаемых методом
`findOfferAvailability` TL-клиента). При этом метод `findOfferAvailability` параметризован только айдишником отеля и
датами проживания, но не occupancy: его ответ содержит "сырые" данные о доступном инвентаре отеля на заданные даты, а
"расселением" запрошенного количества гостей по этому инвентарю занимается наш код в Серчере. Поэтому один и тот же
ответ `findOfferAvailability` может переиспользоваться для ответов на запросы на одни и те же даты, но с разными
occupancy.

Travelline чувствителен к rps'у на свои API, поэтому просит кешировать результаты ответов как `getHotelInfo` (как nice
to have), так и `findOfferAvailability` (этот прямо must).

Этот документ коротко описывает принципы этого кеширования. Так как `getHotelInfo` и `findOfferAvailability` кешируются
совершенно независимо друг от друга, рассмотрим их по отдельности.


## Кеширование статической информации HotelInfo

За получение HotelInfo отвечает интерфейс `TravellineHotelDataSearcher`. У него две имплементации:
`DirectTravellineHotelDataSearcher` и `CachedTravellineHotelDataSearcher`.

Первый класс по сути просто оборачивает в себе `TravellineClient` и напрямую вызывает его `getHotelInfo`. В продакшн
окружении он не используется: эта имплементация `TravellineHotelDataSearcher` подставляется в спринг-контекст только при
выключенном кешировании (настройка `partners.travelline.cache.hotels.enabled=false`) и используется для тестирования и
отладки.

Боевой же `CachedTravellineHotelDataSearcher` устроен сложнее. Он кеширует объекты `HotelInfo`, чтобы переиспользовать
их при обработке повторных поисков в этот же отель (в т.ч. и на другие даты). Для этого используется класс
`AsyncLoadingCache` из библиотеки Caffeine, позволяющий сохранять объекты по ключу в inmemory-кеше c ограничением на
количество объектов и время жизни обекта.

Иначе говоря, при первом поиске `HotelInfo` по заданному коду отеля будет выполнен запрос к партнеру и его результат
будет помещен в кеш. Последующие запросы в этот отель будут доставать объект `HotelInfo` из кеша, пока не истечет время
его жизни (в проде эта настрока равна 10-минутам) или пока он не будет выкинут из кеша из-за превышения количества
элементов (сейчас настройки поддерживают 1000 отелей в памяти одновременно, при превышении начинается эвикшен по
принципу Least Frequently Used; число 1000, кстати взято "с потолка", кажется его можно свободно увеличить, память
серчера позволяет).

При этом важно понимать: если на стороне партнера поменяются какие-то значимые статические данные для отеля, чей
`HotelInfo` закеширован, то возможна ситуация, что могут начать приходить офферы, ссылающиеся на статические данные,
которых нет в закешированном обекте (к примеру, могут приходить офферы с `room_type.code` или `rate_plan.code`, для
которых нет описаний в `HotelInfo`), и это будет продолжаться, пока `HotelInfo` не будет заэвикчен из кеша. Чтобы не
ждать 10 минут (все это время офферы в такой отель будут теряться), предусмотрен механизм инвалидации закешированных
данных.

Работает он следующем образом: в caffeine-кеше лежат не сами `HotelInfo`-объекты, а их обертки —
`Actualizable<HotelInfo>`. Из такой обертки можно получить закешированный объект (методом `getCached`), но если
обнаружится, что в нем недостаточно данных, его можно актуализивать, вызвав для обертки метод `actualize`, который
вернет новый экземляр `Actualizable<HotelInfo>`, в котором будет находится обновленный объект, при этом будет обновлен и
сам объект в кеше.

Этот механизм используется в методе `execute` класса `TravellineTaskHandler`, который, передав полученный из кеша
`Actualizable<HotelInfo>` в метод-обработчик processResults, ловит исключение `MissingHotelDataException`, которое
выбрасывается из недр `processResults` в случае, если закешированный HotelInfo не содержит нужных данных. Поймав это
исключение, код вызывает `actualize` на полученную обертку - и повторно вызывает `processResults`, но уже с новым,
актуализированным значением, при этом так же передавая в processResults специальный флаг о том, что снова выбрасывать
`MissingHotelDataException` больше не надо: если нужных данных не найдется и в актуализированном объекте тоже, то значит
рассинхрон данных случился на стороне партнера, мы с этим ничего сделать не можем, а проблемный оффер надо просто
пропустить.

За поведением этого класса можно следить в соломоне по метрикам с префиксом сенсора
[`searcher.partners.travelline.hotelCache.*.rate`](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=searcher.partners.travelline.hotelCache.*.rate&graph=auto).
Cенсор `get` отвечает за чтение статической информации из кеша, сенсор `put` — за ее сохранение, сенсор `loadDisabled`
указывает на ошибки попытки загрузки данных для отключенных отелей (см раздел "Отключенные отели" ниже), сенсор
`actualize` указывает на ситуации, при которы в данных, сохраненные в кеше, не нашлось какой-то информации и был
задействован механизм актуализации, а сенсор list инкрементируется каждый 10 минут при перезагрузке списка подключенных
отелей (опять таки см. раздел "Отключенные отели" ниже).

Так же имеет смысл следить за метриками http-вызовов к Travelline по сенсору `http.client.calls.count.rate` с
`destination=travelline`. В контексте этого раздела важны вызовы метода [HotelInfo](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=http.client.calls.count.rate&l.destination=travelline&l.method=HotelInfo&graph=auto&b=1h&e=).



## Кеширование доступности инвентаря OfferAvailability

###  Общая логика
Общая логика здесь следующая: мы можем на нашей стороне кешировать сырые ответы метода `findOfferAvailability`, сохраняя
их по комбинации `<hotel_code, checkin_date, checkout_date>` для последующего использования, как в полностью совпадающих
поисковых запросах, так и в запросах в тот же отель на те же даты, но с другой occupancy, так как на формировании этого
ответа запрошенная occupancy не влияет.<br>
Самое сложное здесь — понять, когда такой кеш необходимо инвалидировать. Вся остальная логика в этом разделе посвящена
этому вопросу.

На стороне Travelline поддерживается _версионирование инвентаря_ на двух уровнях: уровне конкретной даты внутри отеля и
уровне всего отеля в целом.<br>
Так, для каждого отеля в Travelline хранится список дат, на которые в нем в принципе возможны продажи через них. Обычно
в каждый момент времени это все даты начиная от текущей и на несколько месяцев вперед (до 2 лет в будущее). С каждой
датой связано положительное целое 64-битное число (*версия инвентаря даты*). Это число увеличивается каждый раз, когда
меняется стомость или существенно уменьшается доступность номеров в данном отеле на данную дату. Это может происходить
когда отельер меняет цены, добавляет или изменяет тарифные планы, так и просто при бронировании номеров.<br>
Скажем, если отельер добавил номер тарифный план и открыл его к продаже с 10 сентября 2020 года, то версии инвентаря
всех дат, начиная с 2020-09-10 включительно и до конца горизонта будут увеличены.<br>
Другой пример: клиент бронирует номер (не важно: через нас, в модуле на сайте отеля или через любую другую систему) на
период с 15 по 18 октября 2020 года. Предположим, что на 16, 17 и 18 октября других номеров в этом отеле нет, а 15го —
есть еще много. Тогда версия инвентаря 2020-10-15 изменена не будет (так как доступность существенно не изменилась), а
на 2020-10-16 и 2020-10-17 версия будет увеличена. Дата выезда — 2020-10-18 — не будет изменена, так как обычно номер
считается свободным в день выезда.<br>
В любом случае, при изменении версии инвентаря какой-то даты, необходимо инвалидировать все объекты `OfferAvailability`,
закешированные на даты, включающие в себя данную (включая дату заезда, исключая дату выезда).<br>
Так же Travelline хранит глобальную версию инвентаря всего отеля: это тоже неотрицательное 64-битное целое,
увеличивающееся, когда увеличивается версия какой-либо даты этого отеля.<br>
У этой версии есть еще одна, дополнительная функция: она равна `0` для отелей, поиск предложений в которых для нас не
доступен (чаще всего это либо отели, которые еще не приняли оферту и поэтому не подключились к Яндексу, или временно
отключенные от Travelline: для технического обслуживания, за неуплату комиссий и тп). У нас есть договоренность с
Travelline, что для таких отелей мы не будет вызывать поиск предложений через `findOfferAvailability`: таковой поиск
гарантированно завершится ошибкой, но мигания мониторингов никому не нужно, так что даже вызовов лучше не делать.

Для чтения версий инвентаря Travelline предоставляется два метода API:

- метод `/hotels`, возвращающий глобальные версии инвентаря всех отелей, в виде списка пар `<hotel_code, version>` для
всех отелей travelline (их порядка 5000, соответственно размер ответа этого метода относительно большой, ~200Kb)
- метод `/hotel_inventory?hotel.code=`, возвращающий версии всех дат заданного отеля, в виде списка пар
`<date, version>`. При горизонте бронирования в 2 года (максимум на данный момент), этот список будет содержать около
730 пар, размер ответа будет порядка 30KB.

Travelline рекомендует следующий достаточно простой паттерн работы с этими методами:
1. На регулярной основе (раз в несколько минут) вызывать метод `/hotels` и обнаруживать изменения в его выдаче:
новые отели, пропавшие отели и отели, чья глобальная версия увеличилась с момента предыдущего вызова
2. Для каждого изменившегося отеля из п1 вызывать метод `/hotel_inventory` и сравнивать его выдачу с результатом
предыдущего его вызова для данного отеля, обнаруживать изменения: новые даты, пропавшие даты и даты, чья версия
увеличилась с момента предыдущего вызова.
3. Инвалидировать все сохраненные в нашем кеше объекты `OfferAvailability`, в чьи интервалы дат `[checkin...checkout)`
попадают изменившиеся и удаленные даты из пункта 2.


Получается, что для реализации вышеописанного подхода необходимо поддерживать два кеша:

- Кеш версий инвентаря, хранящий ответы методов `/hotels` и `/hotel_inventory`;
- Кеш предложений, хранящий объекты `OfferAvailability`

При этом была предложена следующая логика:
- регулярный процесс, следящий за изменениями версий инвентаря, обновляет кеш версий инвентаря (т.е. выполняет пункты
1-2 из вышеописанного алгоритма), но НЕ выполняет инвалидации кеша предложений, т.е. не делает п3.
- при сохранении объекта `OfferAvailability` в кеше предложений, вместе с ним сохраняются так же соотвествующие ему
версии инвентаря, которые были актуальны на момент получения этого объекта от партнера.
- при поиске предложений поисковый запрос достает не только закешированное предложение из кеша предложений, но и читает
текущие актуальные версии инвентаря из кеша версий инвентаря, после чего сравнивает эти версии с теми, что были
сохранены вместе с закешированным предложением. Если хоть для одной из дат текущая версия оказывается новее, чем версия
этой же даты, сохраненная вместе с предложением, то предложение считается устаревшим и подлежащим инвалидации, а значит
поисковый запрос отправяяется в метод `findOfferAvailability` с целью актуализации, и его ответ используется в качестве
результата поиска, а так же сохраняется в кеше ответов вместе с текущими актуальными версиями инвентаря,
соответствующими датам запроса.

Таким образом, инвалидация объектов `OfferAvailability` получается "ленивой": она происходит не в тот момени, когда мы
обнаруживаем изменение версий инвентаря, а позднее: в момент, когда поиск пытается получить к ним доступ после такового
изменения. Это позволяется разграничить потоки записи: в кеш версий инвентаря данные пишет только регулярный фоновый
процесс,следящйи за версиями инвентаря, а в кеш предложений пишут только обработчики поисковых запросов.

> ####Пример:
> Предположим, при первоначальном наполнении кеша для некоего отеля c кодом `10` в методе `/hotels` возвращается
> глобальная версия `100`, а метод `/hotel_inventory` для этого отеля возвращает одинаковые версии `50` для дат
> `2020-10-01`, `2020-10-02` и `2020-10-03`. Процесс, поддерживающий версии инвентаря, запускается вместе с запуском
> Серчера и сохраняет эти значения в кеше версий инвентаря.<br>
> Далее поступает поисковый запрос для этого отеля с заездом `2020-10-01` и выездом `2020-10-02`. Обработчик запросы
> извлекает версии дат для этого запроса из кеша версий инвентаря (в данном случае - только для даты `2020-10-01`, т.к.
> дата выезда не учитывается) и пытается извлечь `OfferAvailability` из кеша предложений (там по условиям задачи пока
> пусто). Так как в кеше предложений ничего не находится, то выполняется поисковый запрос в Api партнера, его результат
> используется в качестве поискового ответа для генерации офферов, а так же кешируется в кеше предложений по ключу
> `<hotel_code=100, checkin = 2020-10-01, checkout = 2020-10-02>` вместе с соответствующи ему версиями инвентаря дат —
> массивом `[50]`, полученным чуть ранее из кеша версий инвентаря.<br>
> Пусть так же придет запрос в этот же отель на даты `2020-10-01 - 2020-10-03` - будет точно так же выполнен запрос в
> API партнера, а его результат будет аналогичным образом закеширован в кеше предложений по ключу `<hotel_code=100,
> checkin = 2020-10-01, checkout = 2020-10-03>` вместе с массивом версий дат `[50, 50]`. <br>
> Предположим теперь, что на стороне партнера произошло радикальное сокращение инвентаря этого отеля на 2 октября 2020
> года: к примеру, забронировали много номеров на эту дату, и все или почти все номера закончились. Тогда при очередном
> вызове метода `/hotels` мы заметим, что глобальная версия этого отеля возрасла и стала, к примеру, равна `150`.
> Процесс, следящий за акутальностью кеша версий инвентаря обнаружит это и вызовет метод `/hotel_inventory` для нашего
> отеля. Этот метод вернет старую версию `50` для дат `2020-10-01` и `2020-10-03`, но новую версию `75` для даты
>`2020-10-02`. Процесс, поддерживающий версии инвентаря, просто сохранит эти даты в своем кеше, не делая ничего с кешем
> предложений. <br>
> Далее, к примеру, поступают два поисковых запроса, аналогичные двум предыдущим: один на даты `2020-10-01 — 2020-10-02`,
> другой на даты `2020-10-01 — 2020-10-03`. Для обоих этих запросов в кеше предложений есть сохраненные ответы, один
> сохранен с массивом версий `[50]`, другой — с массивом версий `[50, 50]`. <br>
> Для первого ответа обработчик поискового запроса прочитает из кеша версий инвентаря актуальные версии равные массиву
> `[50]`, увидит, что они совпадают с сохраненными вместе с ответом в кеше предложений, из чего сделает вывод о
> валидности кеша и вернет закешированный объект без похода к партнеру.<br>
> Для второго ответа обработчик поискового запроса прочитает из кеша версий инвентаря актуальные версии равные массиву
> `[50, 75]`, обнаружит, что эти версии новее сохраненных вместе с ответом версий `[50, 50]`, сделает из этого вывод о
> невалидности ответа, после чего сделает новый запрос к API партнера на эти даты, и использует полученный ответ
> в качестве поискового ответа для генерации офферов, а так же пересохранит их в кеше ответов поверх старой записи с тем
> же самым ключом `<hotel_code=100, checkin = 2020-10-01, checkout = 2020-10-03>`, но уже с новым массивом версий
> `[50, 75]`.

> ####Note:
> В примере выше в качестве версий использованы простые двух-трех значные числа. В реальности Travelline использует для
> версионирования длинные числа, получаемые как Unix timestamps (с секундной точностью) на момент перестраивания
> кеша инвентаря на их стороне, вроде `1600690315`. Вышеописанная логика не зависит от этой детали имплементации, но ее
> стоит иметь в виду в качестве sanity check: слишком маленькие или слишком большие значения версий (соответствующие
> далекому прошлому или будущему) могут свидетельствовать о какой-то ошибке.

###  Имплементация логики кеширования

Для поиска доступности офферов используется интерфейс `TravellineAvailabilitySearcher`, декларирующий один единственный
метод `lookupOffers`. Этот интерфейс реализуется двумя классами: `DirectTravellineAvailabilitySearcher` и
`CachedTravellineAvailabilitySearcher`. Первый — простой: он просто перенаправляет запросы напрямую в API клиента,
используя для этого два разных экземпляра `TravellineClient` (один — для Interactive, другой — для Background запросов,
о причинах такого разделения см. ниже в разделе "ThrottledTravellineClient"). Этот класс используется только в
unit-test'ах или при отключенном кешировании предложений (настройка
`partners.travelline.cache.availability.enabled=false`), но не используется в production. Второй же класс-наследник —
`CachedTravellineAvailabilitySearcher` — является основным и реализует вышеописанную логику поддержания и обновления
кешей версий инвентаря и ответов партнера.

Логика регулярного обновления кеша инвентаря реализуется в нем методом `updateCache`, который запускается по расписанию
(в конфиге регулярность определяется параметром `partners.travelline.cache.availability.full-cache-update-interval`, по
умолчанию — раз в 5 минут) в отдельном треде при помощи `SingleThreadScheduledExecutor`'а, инстанциируемого при старте
приложения. Это метод вызывает клиентский метод `listHotels` (соответствующий методу `/hotels` API TL, возвращающему
список глобальных версий всех отелей), после чего выполняется основные действия по апдейту инвентаря в методе
`updateInventoryForHotels`: именно здесь определяется сет отелей, у которых обновилась глобальная версия инвентаря
(`itemsToUpdate`), сет новых отелей (`itemsToAdd`) и сет удаленных отелей (`itemsToRemove`). Для новых и обновившихся
отелей вызывается метод `updateHotelInventory`, который вызывает клиентский метод `getHotelInventory` (соответствующий
методу `/hotel_inventory` API TL, возвращающему список версий дат данного отеля), а так же читает из кеша сохраненный
кеш этого отеля, после чего обнаруживает изменившиеся, добавившиеся и удалившиеся версии дат. Для всех таковых версий
дат генерируется событие `TTravellineCacheEvent`, которое отправляется в шину результатов серчера и может быть
использовано на стороне OfferСache для автоматической инвалидации офферов уже на более низком уровне. После отправки
этого события обновленный инвентарь каждого отеля сохраняется в кеше. Аналогичный метод `removeHotelInventory`
вызывается для отелей из сета `itemsToRemove`, т.е. пропаших из ответа `listHotels` или таковых, чья глобальная версия в
этом ответе стала равна `0` (о таких отелях см. ниже, в разделе "Отключенные отели"). Для этих отелей запросов к
`getHotelInventory` не осуществляется, но событие `TTravellineCacheEvent` генерируется, а сохраненные версии инвентаря
удаляются из кеша.

Логика поиска и "ленивой" инвалидации предложений реализована в методе `lookupOffersImpl`, который параллельно загружает
кеш версий соответствующего отеля (во фьючу `loadCachedInventoryFuture`) и кеш ответов на соответствующую дату (во фьючу
`loadCachedAvailabilityFuture`), и, дождавшись ответов по обоим фьючам, сравнивает массивы версий дат в обоих кешах с
использованием хелпер-методов `Helpers.checkVersionsAreValid` и `Helpers.checkCacheIsValid`. При отсутствии или
невалидности кеша ответов, выполняется клиентский вызов `findOfferAvailability` (соответствующий, собственно, поисковому
вызову `/hotel_offer_availability` в API TL), чей результат сохраняется в кеше предложений вместе с версиями дат
(которые, в свою очередь, генерятся из ответа фьючи `loadCachedInventoryFuture` при помощи хелпер-метода
`Helpers.generateCachedVersions`).


###  Имплементация самих кешей

Вышеописанная логика использует интерфейс `L2Cache` для абстракции непосредственно хранилищ кешей версий инвентаря и
предложений. Этот интерфейс декларирует следующие методы работы с кешем инвентаря:
- `getInventoryVersions` — получить список закешированных глобальных версий инвентарей отелей;
- `getInventory` — получить закешированные версии дат конкретного отеля;
- `putInventory` — сохранить в кеше версии инвентаря конкретного отеля (как версии дат, так и глобальную);
- `removeInventories` — удалить закешированные версии инвентаря для списка отелей;

...и следующие методы работы с кешем предложений:
- `getAvailability` — получить закешированный ответ метода поиска предложений по отелю и датам;
- `putAvailability` — сохранить в кеше ответ метода поиска предложений по отелю и датам;
- `removeAvailabilities` — удалить из кеша ответы метода поиску предложений для заданного набора отелей и дат;

... а так же вспомогательные методы для поиска старых записей в кеше (пока логика очистки не реализована, так что эти
методы не используются в production-коде):
- `listHotelsActualizedBefore` — получить список id отелей, чей инвентарь был последний раз актуализирован до заданной
даты;
- `listHotelsActualizedBefore` — получить список id отелей, чей инвентарь был последний раз актуализирован до заданной
даты;
- `listAvailabilityKeysForOffersBeforeDate` — получить список id отелей и дат, для которых есть закешированные ответы
метода поиска предложений, у которых checkoutDate меньше или равна заданной даты.

Все методы этого интерфейса в качестве одного из параметров принимают объект, реализующий интерфейс `CacheTransaction`,
что позволяет выполнять несколько методов подряд в рамках одной транзакции, если конкретная имплементация `L2Cache`
поддерживает транзакционность.

Сами хранилища кешей версий инвентаря и предложений абстрагированы двумя интерфейсами: `InventoryRepository` и
`AvailabilityRepository`, каждый из которых является уточнением общего generic-интерфейса `CacheRepository`, который по
сути определяет базовый интерфейс KV-хранилища; а для поддержки транзакционного поведения добавлен интерфейс
`TransactionSupplier`. Эти три интерфейса композируются в классе `L2CacheImplementation`, который делегирует на них
вышеперечисленные методы интерфейса `L2Cache`. Соответственно, сама по себе логика хранения кешей попадает в
имплементации интерфейсов `InventoryRepository`, `AvailabilityRepository` и `TransactionSupplier`.

Наборов этих имплементаций два: один тестово-отладочный, для хранения кешей просто в хэш-мапах в памяти. Другой —
основной, для хранения кешей в YT. Живут они в java-пакетах
`ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory` и
`ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt` соответственно.

> **Важно**: inmemory-классы не предназначены для production'а. Они созданы исключеительно для отладочных целей и
> написания юнит-тестов, могут содержать баги в изоляции квази-транзакций и тп. Никогда не включайте InMemory режим на
> production-окружении.

Основной интерес представляет собой YT-имплементация: классы `YtInventoryRepository`, `YtAvailabilityRepository` и их
базовый класс `YtCacheRepository`, в которй помещен базовый функционал KV-хранилища: get/put/remove методы. Все их
методы выполняются в контексте YT-транзакции, которую, завернутой в интерфейс `CacheTransaction`, предоставляет класс
`YtTransactionSupplier`.

Оба YT-репозитория хранят свои данные в реплицированных динамических YT-таблицах на кластере markov, у каждой таблички
есть по одной синхронной реплике и по две асинхронных, реплики живут на кластерах seneca-(sas/man/vla), синхронная
реплика переключается средствами YT автоматически, с использованием `enable_replicated_table_tracker=%true`.

Для создания табличек можно воспользоваться инструментом [l2_tool](https://a.yandex-team.ru/arc/trunk/arcadia/travel/hotels/devops/l2_tool),
позволяющим создать мастер-таблицы, их реплики и тп по нужным путям на нужных кластерах.

`YtInventoryRepository` хранит кеш версий инвентаря в табличке `inventory`, ключом в ней является строковый `HotelId`, в
других колонках лежат `Version` (глобальная версия инвентаря отеля), `ActualizationTimestamp` (таймстемп последней
актуализации версий инвентаря) и `Inventory` (string). В последнем поле сохраняется бинарное представление
протобуф-объекта `TInventoryList`, являющегося списком протобуф-объектов `TInventoryItem`, каждый из которых содержит в
себе дату (в виде целого числа, равного epoch day) и версию инвентаря на эту дату.

`YtAvailabilityRepository` хранит кеш предложений в табличке `offer_availability`, ключ в ней составной, состоит из
колонок `HotelId` (строка), `CheckIn` (epoch day от даты), `CheckOut` (epoch dat от даты). В остальных колонках лежат
`ActualizationTimestamp` (таймстемп последней актуализации предложения), `Versions` (строка с бинарным представлением
протобуф-объекта `TInventoryList`, содержащего версии дат, соответствующих предложению) и `Response` (строка с бинарным
представлением протобуф-объекта `TJson`, содержащим в себе json-ответ ручки `/hotel_offer_availability`).

Транзакции, порождаемые классом `YtTransactionSupplier`, содержат в себе обычные YT-транзакции, таким образом в рамках
одной транзакции проходят следующие действия:

При обновление инвентаря отелей:
1. чтение всех глобальных версий всех отелей
2. получение версий всех дат всех изменившихся отелей
3. сохранение версий дат всех новых и изменившихся отелей
4. удаления информации для удалившихся отелей

При поиске предложений:
1. чтение инвентаря данного отеля
2. чтения кеша предложений на данные даты в данный отель
3. запись обновленного предложения, если прочитанное в п2 устарело

### Ретрай транзакций

Поисковые таски выполняются паралельно, поэтому возможна ситуация, что одновременно выполняется несколько поисковых
запросов в один отель на одни и те же даты. Это может приводить к тому, что паралельно будет осуществляться работа с
одной и той же строчкой таблицы `offer_availability`. Если предложение в ней актуально, то все паралельные задачи
успешно прочитают данные и сформируют ответ из кеша. Однако, если предложение устарело, то все задачи могут осуществить
запрос в API партнера, после чего попытаются записать в нее обновленный ответ партнера. Так как операции выполняются
транзакционно, только одна из этих задач преуспеет, остальные же поймают исключение на попытке закоммитить транзакцию.
Из YT-клиента это исключение будет выброшено как rpc-ошибка с кодом `ROW_CONFLICT = 1700`. В классе YtTransaction она
будет переобернута в более общее независящее от YT исключение `ConcurrentUpdateException`. Исключения этого типа ловятся
в `CachedTravellineAvailabilitySearcher` при помощи специальной Retry-стратегии `L2RetryStrategy`, которая оборачивает
собой весь вызов `lookupOffersImpl`. Поэтому поиски, сломавшиеся на соответствующем исключении, будут перезапущенны, а
так как к моменту этого перезапуска первый, успешный поиск уже успеет закоммитить свою транзакцию, то перезапущенные
поиски прочитают результат из кеша и он уже будет актуальным, поэтому повторных вызовов API партнера и обновления кеша
не случится.

Аналогичным образом ловится и ретраится другая периодчиески всплывающая из YT ошибка: rpc-ошибка с кодом
`MOUNT_INFO_NOT_READY = 1707`: подобные ошибки периодически случаются при чтении/записи в реплицированные таблицы,
команда YT рекомендует ретраить их на клиенте, что и делается внутри Retry-стратегии `L2RetryStrategy`.


### Дедупликация процессов обновления инвентаря
Серчер травеллайн доступен на всех хостах, на которых запущен Серчер, однако в ситуации, когда сам кеш версий инвентаря
расположен во внешнем хранилище (YT), нет необходимости поддерживать процесс обновления этого кеша на всех хостах,
достаточно только одного, любого.<br>
Реализовано это следующем образом. В классе `CachedTravellineAvailabilitySearcher` объявлен атомарный булевый флаг
`inventoryWasTouched`. После каждого обновления кеша версий (которые происходят редко, раз в 5 минут) инвентаря этот
флаг взводится, перед каждой попытки чтения кеша предложений (которых в нормальных условиях - множество каждую секунду)
— опускается. При этом следующий процесс обновления кеша версий не будет запущен, если флаг поднят, т.е. если после
предыдущего обновления этого кеша, сделанного на данном хосте, на этом же хосте не было выполнено ни одного поиска
Travelline.


### ThrottledTravellineClient

В ситуациях, когда надо выполнять живые запросы к API партнера, важно не превысить жесткие ограничения по RPS. Однако,
использовать для этого стандартный механизм ограничений Searcher'а нельзя: стандартный механизм ограничивает rate и
concurrency поисковых задач, а не запросов к партнеру. При хорошей работе кеша, бОльшая часть поисковых задач партнера
затраигивать не должна, поэтому ограничивать надо не поисковые задачи, а именно запросы к API партнера.

Для этой цели создан `ThrottledTravellineClient`: класс, реализующий клиентский интерфейс `TravellineClient`,
оборачивающий его стандартную имплементацию, но при этом добавляющий механизм троттлинга поверх него. Он использует
точно ту же логику троттлинга, что и механизм, ограничивающий поисковые задачи, но поверх только клиентскихв вызовов.
В `CachedTravellineAvailabilitySearcher` передаются два экземпляра этого класса, с разными настройками лимитов, один для
интерактивных запросов (лимит побольше, очереди запросов нет), другой для фоновых (лимит поменьше, запросы могут
складываться в очередь при превышении rate и concurrency лимитов). Оборачивают они при этом один и тот же экземпляр
`DefaultTravellineClient`а, поэтому лишней траты ресурсов не происходит.

### Метрики

За поведением этого класса можно следить в соломоне по метрикам с префиксом сенсора
[`searcher.partners.travelline.availabilityCache.*.rate`](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=searcher.partners.travelline.availabilityCache.*.rate&graph=auto&checks=&b=1d&e=)
для которых сделан отдельный график [searcher-l2-cache](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&host=cluster&service=searcher&graph=searcher-l2-cache),
отображающий эффективность работы кеша предложений. Здесь есть несколько высокоуровневых метрик:

- Метрика "Cache Hit" отвечает за поиски, для которых в кеше нашлись актуальные данные. Это самая хорошая метрика: она
означает, что для поиска сразу будет возврашен результат без похода в API партнера.

- Метрики "Cache Miss", "Outdated" и "Invalid" означают, что в кеше либо не нашлось данных (первая), либо нашлись, но они
либо устаревшие (вторая), либо по еще каким-то причинам плохие (третья). В поисках, которые увеличивают эти метрики,
последует вызов API партнера. В этом нет ничего страшного, но важно, чтобы таких запросов не становилось большинство,
иначе партнер может не выдержать или может начаться троттлинг запросов (см раздел ThrottledTravellineClient) ниже.

- Метрика "Error" означает, что при чтении кешей возникла ошибка, из-за которой не удалось получить закешированные
версии инвентаря, или закешированный ответ или и то, и другое. Чаще всего эти ошбики в production означают проблемы с YT
кластерами. Эти ошибки не фатальны для поиска: как и в случае с "Cache Miss", "Outdated" и "Invalid" здесь будет
осуществлен вызов API партнера. Однако при этом не случится сохранения результата в кеш, поэтому если такие ошибки
продолжаются достаточно долго (что может быть вызвано, например, downtime'ом кластера markov), то потом сильно растет
доля "Cache Miss" / "Outdated" результатов, так как кеши "отстают" от партнера.

- Метрика "Forbidden" означает, что поиск осуществляется в отключенный (т.е. запрещенный для поиска) отель (об
отключенных отелях - см ниже). Вызова к партенеру направлено не будет, но поиск завершится ошибкой (searcher-таск
пофейлится). Наличие относительно небольшого фона этой метрики нормально, главное чтобы внезапно она не становилась
равна 100% от всех - тогда будут фейлится все задачи Travelline.

- Метрика "Fatal" отвечает за ошибки, случившиеся при обращении к партнеру (которое, в свою очередь, было вызвано чем-то,
что нашло отражение в метриках "Cache Miss"/"Outdated"/"Invalid"/"Error" - при успешных "Cache Hit" и при "Forbidden"
обращений к партнеру не бывает). Это самый неприятный сорт ошибки: он приводит к ошибке поиска в ситуации, когда от него
ожидается результат, эту метрику стоит минимизировать.

Есть другой ряд метрик, которые обновляет фоновый процесс обновления кеша версий инвентаря.
Это метрики с префиксом сенсора [`searcher.partners.travelline.l2.*.rate`](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=searcher.partners.travelline.l2.*.rate&graph=auto&b=1d&e=)
Здесь доступны следующие сенсоры:
- newHotels - новые отели, для которых появился инвентарь с глобальной версией большей нуля
- removedHotels - отели, пропавшие из списка travelline или чья глобальная версия инвентаря стала равна нулю
- updatedHotels - отели, у которых поменялась (с ненулевого на ненулевое значения) глоабльная версия инвентаря
- newInventory - новые даты, для которых в отелях появился инвентарь (сильнее всего метрика инкрементируется при
подключении новых отелей, а так же в полночь, когда у всех отелей появляется новый день в их инвентаре)
- removedInventory - даты, пропавшие из инвентаря отелей (сильнее всего метрика инкрементируется при
отключении отелей, а так же в полночь, когда у всех отелей пропадает текущий день из инвентаря)
- updatedInventory - основная метрика обновления инвентаря, соответствует датам, для которых поменялась версия инвентаря
- updates — общая метрика, инкрементируемая при каждом полном обновлении кеша инвентаря, успешные апдейты инкрементят
эту метрику с полем`result=success`, завершившиеся ошибкой - с полем `result=error`. Последние представляют особенный
интерес с точки зрения operations. Ошибки чаще всего случаются при разовых флапах API партнера или YT кластера. Важно
следить, чтобы несколько операций обновления подряд не завершались ошибками.


Так же имеет смысл следить за метриками http-вызовов к Travelline по сенсору `http.client.calls.count.rate` с
`destination=travelline`. В контексте этого раздела важны вызовы методов [Hotels](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=http.client.calls.count.rate&l.destination=travelline&l.method=Hotels&graph=auto&b=1h&e=),
[HotelInventory](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=http.client.calls.count.rate&l.destination=travelline&l.method=HotelInventory&graph=auto&b=1h&e=)
и [OfferAvailability](https://solomon.yandex-team.ru/?project=travel&cluster=searcher_prod&service=searcher&l.host=cluster&l.sensor=http.client.calls.count.rate&l.destination=travelline&l.method=OfferAvailability&graph=auto&b=1h&e=)


## Отключенные отели

Метод `/hotels` Travelline возвращает вообще все отели в базе travelline, а не только принявшие оферту Яндекса и
предоставляющие нам офферы для поиска. В отелях, не подключенных к яндексу искать предложения нельзя: вызов методов
`/hotel_offer_availability` и `hotel_inventory` для них приведет к ошибке. В теории, поисковых запросов в них
приходить не должно, так как белый список отелей в offer_cache включает только подключенные отели. Но на практике белый
список обновляется достаточно редко: вновь подключившиеся отели добавляются в него автоматически, в то время как
отключившиеся отели из него просто так не пропадают и ждут ручного вмешательства. Кроме того, отели могут отключаться на
стороне Travelline, причем временно: из-за задолжности отеля перед TL, во время технического обслужевания сайта отеля
и тп. Поэтому может быть так, что с точки зрентия API Travelline отель отключен, но мы об этом не знаем и продолжаем
посылать в searcher предложения для поиска в нем.

Travelline сообщает нам о том, что отель отключен, указывая глобальную версию его инвентаря в ответе метода `/hotels`
равной `0`. Для таких отелей не разрешен поиск предложений в `CachedTravellineAvailabilitySearcher`, а так же поиск
статической информации про отель в `CachedTravellineHotelDataSearcher`, последний для обеспечения этого функционала
поддерживает периодически обновляемый сет `enabledHotels`. Входящий запрос на поиск информации про такой отель
завершается исключением `Unable to load hotel info for disabled hotel` в `CachedTravellineHotelDataSearcher`, и
исключением `hotel .. is not allowed by partner for searching` в `CachedTravellineAvailabilitySearcher`. Небольшой фон
этих ошибок существует всегда (по вышеописанной причине), но если таковых становится очень много (вплоть до 100%), то
это признак того, что у Travelline слетел кеш отелей на их стороне, и все отели считаются отключенными. До исправления
ошибки [TRAVELBACK-1178](https://st.yandex-team.ru/TRAVELBACK-1178) такой "пустой" ответа метода `/hotels` кешировался
на нашей стороне, что приводило к тому, что до следующего обновления инвентаря (и на стороне
`CachedTravellineAvailabilitySearcher`, и на стороне `CachedTravellineHotelDataSearcher`) фейлились абсолютно все
запросы. В данный момент эта ошибка исправлена, но за фоном исключений все равно стоит следить.
