package ru.yandex.travel.hotels.searcher.services.cache.travelline;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.messaging.MessageBus;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.yt.ConnectionFactory;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.searcher.partners.TravellineTaskHandlerProperties;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.AvailabilityRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CachedTravellineAvailabilitySearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.DirectTravellineAvailabilitySearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.InventoryRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.L2Cache;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.L2CacheImplementation;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.TransactionSupplier;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.TravellineAvailabilitySearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory.InmemoryAvailabilityRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory.InmemoryInventoryRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory.InmemoryTransactionSupplier;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt.YtAvailabilityRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt.YtInventoryRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.yt.YtTransactionSupplier;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.hotels.CachedTravellineHotelDataSearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.hotels.DirectTravellineHotelDataSearcher;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.hotels.TravellineHotelDataSearcher;
import ru.yandex.travel.hotels.searcher.yt.YtL2Properties;

@Configuration
@RequiredArgsConstructor
@Slf4j
public class TravellineCacheConfiguration {
    private final TravellineTaskHandlerProperties travellineProperties;

    @Bean
    @ConditionalOnProperty({"partners.travelline.cache.availability.enabled", "partners.travelline.cache.in-memory"})
    public InventoryRepository inmemoryInventoryRepository() {
        return new InmemoryInventoryRepository();
    }

    @Bean
    @ConditionalOnProperty("partners.travelline.cache.availability.enabled")
    @ConditionalOnMissingBean(InventoryRepository.class)
    public InventoryRepository ytInventoryRepository(YtL2Properties ytL2Properties) {
        return new YtInventoryRepository(ytL2Properties.getBasePath());
    }

    @Bean
    @ConditionalOnProperty({"partners.travelline.cache.availability.enabled", "partners.travelline.cache.in-memory"})
    public AvailabilityRepository inmemoryAvailabilityRepository() {
        return new InmemoryAvailabilityRepository();
    }

    @Bean
    @ConditionalOnProperty("partners.travelline.cache.availability.enabled")
    @ConditionalOnMissingBean(AvailabilityRepository.class)
    public AvailabilityRepository ytAvailabilityRepository(YtL2Properties ytL2Properties) {
        return new YtAvailabilityRepository(ytL2Properties.getBasePath());
    }

    @Bean
    @ConditionalOnProperty({"partners.travelline.cache.availability.enabled", "partners.travelline.cache.in-memory"})
    public TransactionSupplier inmemoryTransactionSupplier() {
        return new InmemoryTransactionSupplier();
    }

    @Bean
    @ConditionalOnProperty("partners.travelline.cache.availability.enabled")
    @ConditionalOnMissingBean(TransactionSupplier.class)
    public TransactionSupplier ytTransactionSupplier(YtL2Properties ytL2Properties) {
        var connectionFactory = new ConnectionFactory(ytL2Properties);
        return new YtTransactionSupplier(connectionFactory, ytL2Properties.getCluster());
    }


    @Bean
    @ConditionalOnProperty("partners.travelline.cache.availability.enabled")
    public L2Cache l2Cache(InventoryRepository inventoryRepository, AvailabilityRepository availabilityRepository,
                           TransactionSupplier transactionSupplier) {
        return new L2CacheImplementation(inventoryRepository, availabilityRepository, transactionSupplier);
    }

    @Bean
    @ConditionalOnBean(L2Cache.class)
    public TravellineAvailabilitySearcher cachedTravellineAvailabilitySearcher(L2Cache l2Cache,
                                                                               TravellineClient backgroundThrottledTravellineClient,
                                                                               TravellineClient interactiveThrottledTravellineClient,
                                                                               TravellineClient inventoryThrottledTravellineClient,
                                                                               MessageBus messageBus,
                                                                               Retry retryHelper) {
        log.info("Travelline l2 cache is enabled, will use CachedTravellineAvailabilitySearcher");
        return new CachedTravellineAvailabilitySearcher(l2Cache, backgroundThrottledTravellineClient,
                interactiveThrottledTravellineClient, inventoryThrottledTravellineClient, messageBus,
                travellineProperties.getCache().getAvailability(), retryHelper);
    }

    @Bean
    @ConditionalOnMissingBean(TravellineAvailabilitySearcher.class)
    public TravellineAvailabilitySearcher directTravellineAvailabilitySearcher(TravellineClient backgroundThrottledTravellineClient,
                                                                               TravellineClient interactiveThrottledTravellineClient) {
        log.warn("Travelline L2 cache is disabled, will use DirectTravellineOfferSearcher");
        return new DirectTravellineAvailabilitySearcher(backgroundThrottledTravellineClient,
                interactiveThrottledTravellineClient);
    }

    @Bean
    @ConditionalOnProperty("partners.travelline.cache.hotels.enabled")
    public TravellineHotelDataSearcher cachedTravellineHotelDataSearcher(TravellineClient travellineClient) {
        log.info("Travelline HotelInfo cache is enabled, will use CachedTravellineHotelDataSearcher");
        return new CachedTravellineHotelDataSearcher(travellineClient, travellineProperties.getCache().getHotels());
    }

    @Bean
    @ConditionalOnMissingBean(TravellineHotelDataSearcher.class)
    public TravellineHotelDataSearcher directTravellineHotelDataSearcher(TravellineClient travellineClient) {
        log.warn("Travelline HotelInfo cache is disabled, will use DirectTravellineHotelDataSearcher");
        return new DirectTravellineHotelDataSearcher(travellineClient);
    }

}
