package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability;

import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.util.concurrent.MoreExecutors;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import ru.yandex.misc.thread.factory.ThreadNameThreadFactory;
import ru.yandex.travel.commons.messaging.MessageBus;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.retry.DefaultRetryStrategy;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.hotels.common.partners.base.CallContext;
import ru.yandex.travel.hotels.common.partners.travelline.TravellineClient;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventoryResponse;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelListItem;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelOfferAvailability;
import ru.yandex.travel.hotels.common.partners.travelline.model.ListHotelsResponse;
import ru.yandex.travel.hotels.proto.EOfferInvalidationSource;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.hotels.proto.ERequestClass;
import ru.yandex.travel.hotels.proto.TFilter;
import ru.yandex.travel.hotels.proto.THotelId;
import ru.yandex.travel.hotels.proto.TInventoryItem;
import ru.yandex.travel.hotels.proto.TInventoryList;
import ru.yandex.travel.hotels.proto.TOfferInvalidationEvent;
import ru.yandex.travel.hotels.proto.TOfferInvalidationMessage;
import ru.yandex.travel.hotels.proto.TTargetIntervalFilter;

@Slf4j
@RequiredArgsConstructor
public class CachedTravellineAvailabilitySearcher implements InitializingBean, DisposableBean,
        TravellineAvailabilitySearcher {
    private static final int UNAVAILABLE_INVENTORY_VERSION = 0;
    private static final int NOT_FOUND_IN_HOTELS_INVENTORY = -1;

    private final L2Cache l2Cache;
    private final TravellineClient backgroundThrottledTravellineClient;
    private final TravellineClient interactiveThrottledTravellineClient;
    private final TravellineClient inventoryThrottledTravellineClient;
    private final MessageBus messageBus;
    private final CachedTravellineAvailabilitySearcherProperties properties;
    private final Counter cacheHitCounter = Metrics.counter("searcher.partners.travelline.availabilityCache.hit");
    private final Counter cacheMissCounter = Metrics.counter("searcher.partners.travelline.availabilityCache.miss");
    private final Counter cacheForbiddenCounter = Metrics.counter("searcher.partners.travelline.availabilityCache" +
            ".forbidden");
    private final Counter cacheOutdatedCounter = Metrics.counter("searcher.partners.travelline.availabilityCache" +
            ".outdated");
    private final Counter cacheErrorCounter = Metrics.counter("searcher.partners.travelline.availabilityCache.error");
    private final Counter cacheFatalCounter = Metrics.counter("searcher.partners.travelline.availabilityCache.fatal");
    private final Counter cacheInvalidCounter = Metrics.counter("searcher.partners.travelline.availabilityCache" +
            ".invalid");
    private final Counter fullUpdateSuccesses = Metrics.counter("searcher.partners.travelline.l2.updates", "result",
            "success");
    private final Counter fullUpdateErrors = Metrics.counter("searcher.partners.travelline.l2.updates", "result",
            "error");
    private final Counter newHotelsSuccessCounter = Metrics.counter("searcher.partners.travelline.l2.newHotels",
            "result", "success");
    private final Counter updatedHotelsSuccessCounter = Metrics.counter("searcher.partners.travelline.l2.updatedHotels",
            "result", "success");
    private final Counter removedHotelsSuccessCounter = Metrics.counter("searcher.partners.travelline.l2.removedHotels",
            "result", "success");
    private final Counter newHotelsErrorCounter = Metrics.counter("searcher.partners.travelline.l2.newHotels",
            "result", "error");
    private final Counter updatedHotelsErrorCounter = Metrics.counter("searcher.partners.travelline.l2.updatedHotels",
            "result", "error");
    private final Counter removedHotelsErrorCounter = Metrics.counter("searcher.partners.travelline.l2.removedHotels",
            "result", "error");

    private final Counter newInventoryCounter = Metrics.counter("searcher.partners.travelline.l2.newInventory");
    private final Counter updatedInventoryCounter = Metrics.counter("searcher.partners.travelline.l2.updatedInventory");
    private final Counter removedInventoryCounter = Metrics.counter("searcher.partners.travelline.l2.removedInventory");
    private final Retry retryHelper;
    private final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor(
            new ThreadNameThreadFactory("L2-cache-updater"));

    private AtomicBoolean inventoryWasTouched = new AtomicBoolean(true);

    private static TInventoryList fromInventoryItemList(List<TInventoryItem> list) {
        var builder = TInventoryList.newBuilder();
        builder.addAllItems(list);
        return builder.build();
    }

    public CompletableFuture<Void> updateCache() {
        if (inventoryWasTouched.get()) {
            log.info("Starting full Inventory cache update");
            var listHotelFuture = inventoryThrottledTravellineClient.listHotels();
            return retryHelper.withRetry("TravellineInventoryCacheUpdate",
                            () -> l2Cache.transactionally(t -> updateInventoryForHotels(t, listHotelFuture)),
                            new L2RetryStrategy<>())
                    .thenAccept(result -> {
                        if (!result.contains(null)) {
                            // if result contains null it means we didn't update the cache for some hotels, so we do
                            // not need to set inventory as ready (non-touched), so the next update will update it
                            inventoryWasTouched.compareAndSet(true, false);
                        }
                    })
                    .whenComplete((r, t) -> {
                        if (t != null) {
                            fullUpdateErrors.increment();
                        } else {
                            fullUpdateSuccesses.increment();
                        }
                    });
        } else {
            log.info("Full inventory cache update was not performed as the previous update result was not used");
            return CompletableFuture.completedFuture(null);
        }
    }

    public void updateCacheSync() {
        try {
            updateCache().join();
        } catch (Exception ex) {
            log.error("Unable to updated inventory cache", ex);
        }
    }

    @Override
    public CompletableFuture<HotelOfferAvailability> lookupOffers(String taskId, String hotelId, LocalDate checkin,
                                                                  LocalDate checkout, ERequestClass requestClass,
                                                                  CallContext callContext, String requestId) {
        if (callContext.getTestContext() != null) {
            log.warn("Test context found: will skip caching");
            TravellineClient client = clientForRequestClass(requestClass);
            return client.withCallContext(callContext)
                    .findOfferAvailability(hotelId, checkin, checkout, requestId);
        }
        inventoryWasTouched.set(true);
        return retryHelper.withRetry("TravellineOfferAvailabilityL2Search",
                () -> lookupOffersImpl(taskId, hotelId, checkin, checkout, requestClass, requestId),
                new L2RetryStrategy<>());
    }

    private TravellineClient clientForRequestClass(ERequestClass requestClass) {
        switch (requestClass) {
            case RC_INTERACTIVE:
                return interactiveThrottledTravellineClient;
            case RC_CALENDAR:
            case RC_BACKGROUND:
            default:
                return backgroundThrottledTravellineClient;
        }
    }

    private CompletableFuture<HotelOfferAvailability> lookupOffersImpl(String taskId, String hotelId, LocalDate checkin,
                                                                       LocalDate checkout, ERequestClass requestClass,
                                                                       String httpRequestId) {
        return l2Cache.transactionally(transaction -> {
            log.debug("Task {}: looking up offers via l2 caches", taskId);
            CompletableFuture<CachedHotelInventory> loadCachedInventoryFuture = retryHelper.withRetry(
                            "LoadInventoryCache",
                            () -> loadInventoryCacheForHotel(taskId, hotelId, transaction), new L2RetryStrategy<>())
                    .exceptionally(t -> {
                        cacheErrorCounter.increment();
                        log.error("Task {}: unable to update inventory cache for hotel {}, assuming no cache", taskId,
                                hotelId, t);
                        return null;
                    });
            CompletableFuture<CachedOfferAvailability> loadCachedAvailabilityFuture =
                    l2Cache.getAvailability(taskId, hotelId, checkin, checkout, transaction)
                            .exceptionally(t -> {
                                log.error("Task {}: unable to lookup availability cache for hotel {} on dates " +
                                                "{} — {}, assuming no cache",
                                        taskId, hotelId, checkin, checkout, t);
                                cacheErrorCounter.increment();
                                return null;
                            });
            return CompletableFuture.allOf(loadCachedInventoryFuture, loadCachedAvailabilityFuture)
                    .thenCompose(ignored -> {
                        var cachedInventory = loadCachedInventoryFuture.join();
                        var availability = loadCachedAvailabilityFuture.join();
                        if (cachedInventory != null) {
                            CompletableFuture<HotelOfferAvailability> response =
                                    processCached(taskId, hotelId, checkin, checkout, cachedInventory, availability);
                            if (response != null) {
                                return response;
                            }
                        }
                        return makeRequest(taskId, hotelId, checkin, checkout, requestClass, httpRequestId,
                                transaction,
                                cachedInventory);
                    });
        });
    }

    /**
     * <pre>
     * If we are here it means either one of the following:
     * 1) Exception while loading hotel inventory (inventoryUpdateFuture returned null)
     * 2) Cached availability is not found (cache miss) or outdated
     * 3) Exception while loading cached availability (availabilityFuture returned null)
     * All of these requires to make actual call to partner for availability.
     * </pre>
     */
    private CompletableFuture<HotelOfferAvailability> makeRequest(String taskId,
                                                                  String hotelId,
                                                                  LocalDate checkin,
                                                                  LocalDate checkout,
                                                                  ERequestClass requestClass,
                                                                  String requestId,
                                                                  CacheTransaction transaction,
                                                                  CachedHotelInventory cachedInventory) {
        TravellineClient client = clientForRequestClass(requestClass);
        log.info("Task {}: L2 cache missed, outdated or caused error, have to execute actual partner " +
                "call", taskId);
        return client
                .findOfferAvailability(hotelId, checkin, checkout, requestId)
                .whenComplete((r, t) -> {
                    if (t != null) {
                        cacheFatalCounter.increment();
                        log.error("Error while querying partner", t);
                    }
                })
                .thenCompose(r ->
                        processPartnerResponse(taskId, hotelId, checkin, checkout, transaction, cachedInventory, r));
    }

    private CompletableFuture<HotelOfferAvailability> processPartnerResponse(String taskId, String hotelId,
                                                                             LocalDate checkin,
                                                                             LocalDate checkout,
                                                                             CacheTransaction transaction,
                                                                             CachedHotelInventory cachedInventory,
                                                                             HotelOfferAvailability r) {
        log.info("Task {}: got partner availability response for hotel {} on dates {} — {}",
                taskId, hotelId, checkin, checkout);
        if (cachedInventory != null) {
            var versions =
                    Helpers.generateCachedVersions(cachedInventory.getInventory(),
                            checkin,
                            checkout);
            if (versions != null) {
                if (transaction != null) {
                    log.info("Task {}: will cache availability response for hotel {} on " +
                            "dates {} — {}", taskId, hotelId, checkin, checkout);
                    return l2Cache.putAvailability(
                                    CachedOfferAvailability.builder()
                                            .hotelId(hotelId)
                                            .checkinDate(checkin)
                                            .checkoutDate(checkout)
                                            .actualizationTimestamp(Instant.now())
                                            .availabilityResponse(ProtoUtils.toTJson(r))
                                            .versions(versions)
                                            .build(), transaction)
                            .whenComplete(availabilityCompletionLogger(taskId, hotelId,
                                    checkin,
                                    checkout, "updating"))
                            .thenApply(put -> r);
                } else {
                    log.warn("Task {}: will NOT cache availability response for hotel {} " +
                                    "on dates {} — {} as there is no transaction",
                            taskId, hotelId, checkin, checkout);
                    return CompletableFuture.completedFuture(r);
                }

            } else {
                log.info("Task {}: will NOT cache availability response for hotel {} on " +
                                "dates {} — {} as it is already outdated",
                        taskId, hotelId, checkin, checkout);
                return CompletableFuture.completedFuture(r);
            }
        } else {
            log.warn("Task {}: will NOT cache availability response for hotel {} " +
                            "on dates {} — {} as it is forbidden",
                    taskId, hotelId, checkin, checkout);
            return CompletableFuture.completedFuture(r);
        }
    }

    private CompletableFuture<HotelOfferAvailability> processCached(String taskId, String hotelId,
                                                                    LocalDate checkin,
                                                                    LocalDate checkout,
                                                                    CachedHotelInventory cachedInventory,
                                                                    CachedOfferAvailability availability) {
        if (cachedInventory.getVersion() == NOT_FOUND_IN_HOTELS_INVENTORY) {
            cacheForbiddenCounter.increment();
            String message = String.format("Task %s: hotel %s is not found in partner's hotel " +
                            "inventory",
                    taskId, hotelId);
            return CompletableFuture.failedFuture(createUnknownHotelException(message));
        }
        if (cachedInventory.getVersion() == UNAVAILABLE_INVENTORY_VERSION) {
            cacheForbiddenCounter.increment();
            String message = String.format("Task %s: hotel %s is not allowed by partner for " +
                    "searching", taskId, hotelId);
            return CompletableFuture.failedFuture(createDisabledHotelException(message));
        }
        if (availability != null) {
            if (!Helpers.checkVersionsAreValid(availability.getVersions())) {
                log.warn("Task {}: invalid (future) availability versions " +
                        "for hotel {} on dates {} — {} ", taskId, hotelId, checkin, checkout);
                cacheInvalidCounter.increment();
            } else if (Helpers.checkCacheIsValid(hotelId,
                    cachedInventory.getInventory(), checkin, checkout,
                    availability.getVersions())) {
                log.info("Task {}: cache hit: availability response found for hotel {} on dates " +
                                "{} — {}",
                        taskId, hotelId, checkin, checkout);
                cacheHitCounter.increment();
                return CompletableFuture.completedFuture(ProtoUtils.fromTJson(availability.getAvailabilityResponse(),
                        HotelOfferAvailability.class));
            } else {
                log.info("Task {}: cache miss: availability response for hotel {} on dates " +
                                "{} — {} is outdated",
                        taskId, hotelId, checkin, checkout);
                cacheOutdatedCounter.increment();
            }
        } else {
            log.info("Task {}: cache miss: no cache for hotel {} on dates {} — {}",
                    taskId, hotelId, checkin, checkout);
            cacheMissCounter.increment();
        }
        return null;
    }

    private ErrorException createDisabledHotelException(String message) {
        return new ErrorException(TError.newBuilder()
                .setCode(EErrorCode.EC_DISABLED_HOTEL)
                .setMessage(message)
                .build());

    }

    private ErrorException createUnknownHotelException(String message) {
        return new ErrorException(TError.newBuilder()
                .setCode(EErrorCode.EC_NOT_FOUND)
                .setMessage(message)
                .build());

    }

    private CompletableFuture<CachedHotelInventory> loadInventoryCacheForHotel(String taskId, String hotelId,
                                                                               CacheTransaction transaction) {
        if (transaction == null) {
            return CompletableFuture.failedFuture(new RuntimeException("No transaction present"));
        }
        log.debug("Task {}: loading inventory from L2 cache", taskId);
        return l2Cache.getInventory(hotelId, transaction)
                .thenApply(inventory -> {
                    if (inventory != null) {
                        log.debug("Task {}: non-empty inventory loaded", taskId);
                        return inventory;
                    } else {
                        log.debug("Task {}: empty inventory loaded", taskId);
                        return CachedHotelInventory.builder()
                                .version(NOT_FOUND_IN_HOTELS_INVENTORY)
                                .hotelId(hotelId)
                                .build();
                    }
                });
    }

    private CompletableFuture<List<CachedHotelInventory>> updateInventoryForHotels(CacheTransaction transaction,
                                                                                   CompletableFuture<ListHotelsResponse> actualInventoryFuture) {
        if (transaction == null) {
            return CompletableFuture.failedFuture(new RuntimeException("No transaction present"));
        }
        Instant updateBeganAt = Instant.now();
        Set<HotelListItem> itemsUpToDate = ConcurrentHashMap.newKeySet();
        Set<HotelListItem> itemsToUpdate = ConcurrentHashMap.newKeySet();
        Set<HotelListItem> itemsToAdd = ConcurrentHashMap.newKeySet();
        Set<String> itemsToRemove = ConcurrentHashMap.newKeySet();
        var cachedInventoryFuture = l2Cache.getInventoryVersions(transaction);
        log.info("Inventory update: loading inventory versions");
        return CompletableFuture.allOf(actualInventoryFuture, cachedInventoryFuture)
                .thenAccept(
                        ignored -> {
                            var cachedInventory = cachedInventoryFuture.join();
                            var actualInventory =
                                    actualInventoryFuture.join().getHotels().stream()
                                            .filter(h -> h.getInventoryVersion() > 0)
                                            .collect(Collectors.toList());
                            log.info("Inventory update: L2 cache contains {} hotels", cachedInventory.size());
                            log.info("Inventory update: partner reports {} hotels", actualInventory.size());
                            for (HotelListItem item : actualInventory) {
                                if (cachedInventory.containsKey(item.getCode())) {
                                    long oldVersion = cachedInventory.remove(item.getCode());
                                    long newVersion = item.getInventoryVersion();
                                    if (newVersion > oldVersion) {
                                        itemsToUpdate.add(item);
                                    } else {
                                        itemsUpToDate.add(item);
                                    }
                                } else {
                                    itemsToAdd.add(item);
                                }
                            }
                            itemsToRemove.addAll(cachedInventory.keySet());
                        }
                ).thenCompose(ignored -> {
                    log.info("Inventory update: will add {}, update {}, remove {} hotel entries. {} entries are up " +
                                    "to date",
                            itemsToAdd.size(), itemsToUpdate.size(), itemsToRemove.size(), itemsUpToDate.size());
                    List<CompletableFuture<CachedHotelInventory>> futuresToWait = new ArrayList<>();
                    for (HotelListItem newItem : itemsToAdd) {
                        futuresToWait.add(updateHotelInventory(newItem.getCode(),
                                newItem.getInventoryVersion(), updateBeganAt, transaction).whenComplete((r, t) -> {
                            if (r != null && t == null) {
                                newHotelsSuccessCounter.increment();
                            } else {
                                newHotelsErrorCounter.increment();
                            }
                        }));
                    }
                    for (HotelListItem updatedItem : itemsToUpdate) {
                        futuresToWait.add(updateHotelInventory(updatedItem.getCode(), updatedItem.getInventoryVersion(),
                                updateBeganAt, transaction).whenComplete((r, t) -> {
                            if (r != null && t == null) {
                                updatedHotelsSuccessCounter.increment();
                            } else {
                                updatedHotelsErrorCounter.increment();
                            }
                        }));
                    }
                    for (String id : itemsToRemove) {
                        futuresToWait.add(removeHotelInventory(id, updateBeganAt, transaction).whenComplete((r, t) -> {
                            if (t == null) {
                                removedHotelsSuccessCounter.increment();
                            } else {
                                removedHotelsErrorCounter.increment();
                            }
                        }));
                    }
                    return CompletableFuture.allOf(futuresToWait.toArray(new CompletableFuture[0]))
                            .thenApply(ign -> futuresToWait.stream()
                                    .map(CompletableFuture::join)
                                    .collect(Collectors.toList()))
                            .whenComplete((r, t) -> {
                                if (t == null) {
                                    long nullCount = r.stream().filter(Objects::isNull).count();
                                    if (nullCount == 0) {
                                        log.info("Inventory update: completed");
                                    } else {
                                        log.warn("Inventory update: completed with {} skipped hotels", nullCount);
                                    }
                                } else {
                                    log.error("Inventory update: failed", t);
                                }
                            });
                });
    }

    private CompletableFuture<CachedHotelInventory> removeHotelInventory(String hotelId,
                                                                         Instant updateTimestamp,
                                                                         CacheTransaction transaction) {
        log.info("Removing inventory cache for hotel {}", hotelId);
        var event = TOfferInvalidationEvent.newBuilder()
                .setHotelId(THotelId.newBuilder()
                        .setOriginalId(hotelId)
                        .setPartnerId(EPartnerId.PI_TRAVELLINE)
                        .build())
                .setCurrency(ECurrency.C_RUB) // all TL offers are generated with RUB currency
                .setTimestamp(ProtoUtils.fromInstant(updateTimestamp))
                .addFilters(TFilter.newBuilder()
                        .setTargetIntervalFilter(TTargetIntervalFilter.newBuilder().build()) // Empty
                        // TargetIntervalFilter matches all dates
                        .build())
                .setOfferInvalidationSource(EOfferInvalidationSource.OIS_TRAVELLINE_SEARCHER)
                .build();
        messageBus.send(TOfferInvalidationMessage.newBuilder().setEvent(event).build(), null)
                .whenComplete(inventoryCompletionLogger(hotelId, "sending invalidation notification", false));

        CompletableFuture<CachedHotelInventory> res =
                l2Cache.removeInventory(hotelId, transaction).thenApply(ignored -> CachedHotelInventory.builder()
                        .hotelId(hotelId)
                        .version(NOT_FOUND_IN_HOTELS_INVENTORY)
                        .build());
        return res.whenComplete(inventoryCompletionLogger(hotelId, "removing hotel", false));
    }

    private CompletableFuture<CachedHotelInventory> updateHotelInventory(String hotelId, long version,
                                                                         Instant updateTimestamp,
                                                                         CacheTransaction transaction) {
        var actualInventoryFuture = inventoryThrottledTravellineClient
                .getHotelInventory(hotelId)
                .handle((r, t) -> {
                    if (t != null) {
                        log.warn("Error while fetching actual inventory for hotel {}: {}", hotelId, t);
                        return null;
                    } else {
                        return r;
                    }
                });
        var cachedInventoryFuture = l2Cache.getInventory(hotelId, transaction);
        return CompletableFuture.allOf(actualInventoryFuture, cachedInventoryFuture).thenCompose(ignored -> {
            final CachedHotelInventory cachedInventory = cachedInventoryFuture.join();
            final HotelInventoryResponse actualInventory = actualInventoryFuture.join();
            if (actualInventory == null) {
                // Error or throttling while fetching inventory from TL: do not update cache, return null
                return CompletableFuture.completedFuture(null);
            }
            List<TInventoryItem> updatedItems = new ArrayList<>();
            List<TInventoryItem> addedItems = new ArrayList<>();
            List<TInventoryItem> removedItems = new ArrayList<>();
            List<TInventoryItem> upToDateItems = new ArrayList<>();
            TInventoryList newInventory = Helpers.matchByDate(hotelId,
                    (cachedInventory == null) ? null : cachedInventory.getInventory(),
                    actualInventory, updatedItems, addedItems, removedItems, upToDateItems);
            String verb = (cachedInventory == null) ? "adding hotel" : "updating hotel";
            log.info("{} cache for hotel {}: {} new, {} updated, {} removed, {} up-to-date items",
                    verb, hotelId, addedItems.size(), updatedItems.size(), removedItems.size(), upToDateItems.size());
            newInventoryCounter.increment(addedItems.size());
            updatedInventoryCounter.increment(updatedItems.size());
            removedInventoryCounter.increment(removedItems.size());
            CachedHotelInventory cachedHotelInfo = CachedHotelInventory.builder()
                    .hotelId(hotelId)
                    .version(version)
                    .inventory(newInventory)
                    .actualizationTimestamp(updateTimestamp)
                    .build();
            if (addedItems.size() + updatedItems.size() + removedItems.size() > 0) {
                var items = Stream.concat(addedItems.stream(), Stream.concat(updatedItems.stream(),
                                removedItems.stream()))
                        .map(TInventoryItem::getDate)
                        .sorted()
                        .collect(Collectors.toUnmodifiableList());
                ArrayList<TFilter> filters = new ArrayList<>();
                var start = 0;
                for (int i = 0; i < items.size(); i++) {
                    if (i + 1 == items.size() || !items.get(i + 1).equals(items.get(i) + 1)) {
                        filters.add(TFilter.newBuilder()
                                .setTargetIntervalFilter(TTargetIntervalFilter.newBuilder()
                                        .setDateFromInclusive(LocalDate.ofEpochDay(items.get(start)).toString())
                                        .setDateToInclusive(LocalDate.ofEpochDay(items.get(i)).toString())
                                        .build())
                                .build());
                        start = i + 1;
                    }
                }
                var event = TOfferInvalidationEvent.newBuilder()
                        .setHotelId(THotelId.newBuilder()
                                .setOriginalId(hotelId)
                                .setPartnerId(EPartnerId.PI_TRAVELLINE)
                                .build())
                        .setCurrency(ECurrency.C_RUB) // all TL offers are generated with RUB currency
                        .setTimestamp(ProtoUtils.fromInstant(updateTimestamp))
                        .addAllFilters(filters)
                        .setOfferInvalidationSource(EOfferInvalidationSource.OIS_TRAVELLINE_SEARCHER)
                        .build();
                messageBus.send(TOfferInvalidationMessage.newBuilder().setEvent(event).build(), null)
                        .whenComplete(inventoryCompletionLogger(hotelId, "sending invalidation notification", false));
            }
            return l2Cache.putInventory(cachedHotelInfo, transaction)
                    .whenComplete(inventoryCompletionLogger(hotelId, verb, false));
        });

    }

    private BiConsumer<Object, Throwable> inventoryCompletionLogger(String hotelId, String process, boolean debug) {
        return (r, t) -> {
            if (t == null) {
                String message = "Inventory cache, hotel {}: done {}";
                if (debug) {
                    log.debug(message, hotelId, process);
                } else {
                    log.info(message, hotelId, process);
                }
            } else {
                log.error("Inventory cache, hotel {}: error while {}", hotelId, process, t);
            }
        };
    }

    private BiConsumer<CachedOfferAvailability, Throwable> availabilityCompletionLogger(
            String taskId, String hotelId, LocalDate checkIn, LocalDate checkOut, String process) {
        return (r, t) -> {
            if (t == null) {
                log.info("Task {}: offer availability cache, hotel {}, dates {}-{}: done {}", taskId, hotelId, checkIn,
                        checkOut, process);
            } else {
                log.error("Task {}: offer availability cache, hotel {}, dates {}-{}: error while {}", taskId, hotelId,
                        checkIn, checkOut, process, t);
            }
        };
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        log.info("Starting background cache updates every {} s",
                properties.getFullCacheUpdateInterval().toSeconds());
        executor.scheduleAtFixedRate(this::updateCacheSync, 0,
                properties.getFullCacheUpdateInterval().toMillis(),
                TimeUnit.MILLISECONDS);
    }

    @Override
    public void destroy() {
        MoreExecutors.shutdownAndAwaitTermination(executor, 1, TimeUnit.SECONDS);
    }

    public static class L2RetryStrategy<T> extends DefaultRetryStrategy<T> {
        @Override
        public boolean shouldRetryOnException(Exception ex) {
            return ex instanceof ConcurrentUpdateException || ex instanceof MountInfoNotReadyException;
        }

        @Override
        public int getNumRetries() {
            return 20;
        }
    }
}
