package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability;

import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;

import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventory;
import ru.yandex.travel.hotels.common.partners.travelline.model.HotelInventoryResponse;
import ru.yandex.travel.hotels.proto.TInventoryItem;
import ru.yandex.travel.hotels.proto.TInventoryList;
import ru.yandex.travel.hotels.proto.TVersionList;

public class Helpers {
    private static final Logger log = org.slf4j.LoggerFactory.getLogger(CachedTravellineAvailabilitySearcher.class);

    static TInventoryList matchByDate(String hotelId, TInventoryList cachedInventory,
                                      HotelInventoryResponse actualInventory,
                                      List<TInventoryItem> updatedItems, List<TInventoryItem> addedItems,
                                      List<TInventoryItem> removedItems, List<TInventoryItem> upToDateItems) {
        int i = 0;
        int j = 0;
        var newInventoryBuilder = TInventoryList.newBuilder();
        while (true) {
            TInventoryItem cached = (cachedInventory != null && i < cachedInventory.getItemsCount())
                    ? cachedInventory.getItems(i) : null;
            HotelInventory actual = j < actualInventory.getHotelInventories().size() ?
                    actualInventory.getHotelInventories().get(j) : null;
            int comparator;
            if (cached == null && actual == null) {
                break;
            } else {
                if (cached != null && actual != null) {
                    comparator = Long.compareUnsigned(cached.getDate(), actual.getDate().toEpochDay());
                } else if (cached == null) { // no cached left, some actual remain => new items
                    comparator = 1;
                } else { // cached items present, no actuals => need to remove
                    comparator = -1;
                }
            }

            if (comparator == 0) {
                if (actual.getVersion() > cached.getVersion()) {
                    log.debug("Hotel {}: date {} changed: was {}, now {}", hotelId, actual.getDate(),
                            cached.getVersion(),
                            actual.getVersion());
                    TInventoryItem updateItem = TInventoryItem.newBuilder()
                            .setDate(actual.getDate().toEpochDay())
                            .setVersion(actual.getVersion())
                            .build();
                    newInventoryBuilder.addItems(updateItem);
                    updatedItems.add(updateItem);
                } else {
                    if (actual.getVersion() < cached.getVersion()) {
                        log.warn("Hotel {}: updated version is lower than cached one", hotelId);
                    }
                    upToDateItems.add(cached);
                    newInventoryBuilder.addItems(cached);
                }
                i++;
                j++;
            } else if (comparator < 0) { // cached is less than actual => cached date is no longer present
                removedItems.add(cached);
                i++;
            } else { // cached is greater than actual => new item
                TInventoryItem newItem = TInventoryItem.newBuilder()
                        .setDate(actual.getDate().toEpochDay())
                        .setVersion(actual.getVersion())
                        .build();
                newInventoryBuilder.addItems(newItem);
                addedItems.add(newItem);
                j++;
            }
        }
        return newInventoryBuilder.build();
    }

    static TVersionList generateCachedVersions(TInventoryList inventory, LocalDate checkin, LocalDate checkout) {
        Preconditions.checkArgument(checkin.isBefore(checkout), "Unexpected date range");

        int i = findIndexOfFirstMatchingItem(inventory, TInventoryItem::getDate, checkin.toEpochDay());
        if (i < 0) {
            return null;
        }
        var builder = TVersionList.newBuilder();
        LocalDate dateToCheck = checkin;
        while (dateToCheck.isBefore(checkout)) {
            if (i >= inventory.getItemsCount()) {
                return null;
            }
            if (!dateToCheck.equals(LocalDate.ofEpochDay(inventory.getItems(i).getDate()))) {
                return null;
            }
            builder.addVersions(inventory.getItems(i).getVersion());
            dateToCheck = dateToCheck.plusDays(1);
            i++;
        }
        return builder.build();
    }

    static boolean checkCacheIsValid(String hotelId, TInventoryList inventory, LocalDate checkin, LocalDate checkout,
                                     TVersionList cachedVersions) {
        Preconditions.checkArgument(checkin.isBefore(checkout), "Unexpected date range");
        Preconditions.checkArgument((checkout.toEpochDay() - checkin.toEpochDay()) == cachedVersions.getVersionsCount(),
                "Unexpected amount of date versions");
        int i = findIndexOfFirstMatchingItem(inventory, TInventoryItem::getDate, checkin.toEpochDay());
        if (i < 0) {
            log.debug("Hotel {} outdated: cached inventory does not contain checkin date {}", hotelId, checkin);
            return false;
        }
        int v = 0;
        LocalDate dateToCheck = checkin;
        while (dateToCheck.isBefore(checkout)) {
            if (i >= inventory.getItemsCount()) {
                log.debug("Hotel {} outdated: cached inventory does not date {}", hotelId, dateToCheck);
                return false;
            }
            LocalDate actDate = LocalDate.ofEpochDay(inventory.getItems(i).getDate());
            if (!dateToCheck.equals(actDate)) {
                log.debug("Hotel {} outdated: expected {} to be cached at position {}, actual is {}", hotelId,
                        dateToCheck, i,
                        actDate);
                return false;
            }
            var actualVersion = inventory.getItems(i).getVersion();
            var cachedVersion = cachedVersions.getVersions(v);
            if (actualVersion > cachedVersion) {
                log.debug("Hotel {} outdated: date {}, actual version {} is greater than cached version {}",
                        hotelId, dateToCheck, actualVersion, cachedVersion);
                return false;
            }
            dateToCheck = dateToCheck.plusDays(1);
            i++;
            v++;
        }
        return true;
    }

    static boolean checkVersionsAreValid(TVersionList cachedVersions) {
        long timestampInOneDayFromNow = Instant.now().plus(1, ChronoUnit.DAYS).toEpochMilli() / 1000;
        for (var version : cachedVersions.getVersionsList()) {
            if (version > timestampInOneDayFromNow) {
                return false;
            }
        }
        return true;
    }

    static <K> int findIndexOfFirstMatchingItem(TInventoryList list,
                                                Function<TInventoryItem, Comparable<? super K>> keyProducer, K target) {
        int low = 0;
        int high = list.getItemsCount() - 1;

        while (low <= high) {
            int mid = (low + high) >>> 1;
            Comparable<? super K> midVal = keyProducer.apply(list.getItems(mid));
            int cmp = midVal.compareTo(target);

            if (cmp < 0) {
                low = mid + 1;
            } else if (cmp > 0) {
                high = mid - 1;
            } else {
                return mid;
            }
        }
        return -1;
    }
}
