package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability;

import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.function.Function;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Tuple3;

@Slf4j
@RequiredArgsConstructor
public class L2CacheImplementation implements L2Cache {

    private final InventoryRepository inventoryRepository;
    private final AvailabilityRepository availabilityRepository;
    private final TransactionSupplier transactionSupplier;


    @Override
    public CompletableFuture<Map<String, Long>> getInventoryVersions(CacheTransaction transaction) {
        return inventoryRepository.getInventoryVersion(transaction);
    }

    @Override
    public CompletableFuture<CachedHotelInventory> getInventory(String hotelId, CacheTransaction transaction) {
        return inventoryRepository.get(hotelId, transaction);
    }

    @Override
    public CompletableFuture<CachedHotelInventory> putInventory(CachedHotelInventory inventory,
                                                                CacheTransaction transaction) {
        return inventoryRepository.put(inventory, transaction);
    }

    @Override
    public CompletableFuture<Void> removeInventories(List<String> hotelIds, CacheTransaction transaction) {
        return inventoryRepository.remove(hotelIds, transaction);
    }

    @Override
    public CompletableFuture<CachedOfferAvailability> getAvailability(String taskId, String hotelId, LocalDate checkin,
                                                                      LocalDate checkout,
                                                                      CacheTransaction transaction) {
        if (transaction == null) {
            return CompletableFuture.failedFuture(new RuntimeException("No transaction present"));
        }
        return availabilityRepository.get(Tuple3.tuple(hotelId, checkin, checkout), transaction);
    }

    @Override
    public CompletableFuture<CachedOfferAvailability> putAvailability(CachedOfferAvailability availability,
                                                                      CacheTransaction transaction) {
        return availabilityRepository.put(availability, transaction);
    }

    @Override
    public CompletableFuture<Void> removeAvailabilities(List<Tuple3<String, LocalDate, LocalDate>> keys,
                                                        CacheTransaction transaction) {
        return availabilityRepository.remove(keys, transaction);
    }

    @Override
    public CompletableFuture<List<String>> listHotelsActualizedBefore(Instant instant, CacheTransaction transaction) {
        return inventoryRepository.listHotelsActualizedBefore(instant, transaction);
    }

    @Override
    public CompletableFuture<List<Tuple3<String, LocalDate, LocalDate>>> listAvailabilityKeysForOffersBeforeDate(LocalDate date, CacheTransaction transaction) {
        return availabilityRepository.listAvailabilityKeysForOffersBeforeDate(date, transaction);
    }

    @Override
    public <R> CompletableFuture<R> transactionally(Function<CacheTransaction, CompletableFuture<R>> functionToCall) {
        return transactionSupplier.startTransaction()
                .exceptionally(t -> {
                    log.error("Unable to start transaction", t);
                    return null;
                })
                .thenCompose(transaction -> {
                    var call = functionToCall.apply(transaction);
                    Preconditions.checkNotNull(call, "Transactional function should return a future, not null");
                    return call.exceptionally(e -> {
                        if (transaction != null) {
                            log.error("Error during transactional execution, will close the transaction", e);
                            transaction.close();
                        } else {
                            log.error("Error during transactionless execution", e);
                        }
                        if (e instanceof RuntimeException) {
                            throw (RuntimeException) e;
                        } else {
                            throw new CompletionException(e);
                        }
                    }).thenCompose(r -> {
                        if (transaction != null) {
                            return transaction.commit().thenApply(ignored -> r);
                        } else {
                            return CompletableFuture.completedFuture(r);
                        }
                    });
                });
    }
}
