package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheRepository;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;

@Slf4j
public abstract class InmemoryCacheRepository<K, V> implements CacheRepository<K, V> {
    protected final Map<K, V> cacheMap;
    private final Map<K, InmemoryTransaction> transactionMap;
    private final Lock lock;

    public InmemoryCacheRepository() {
        cacheMap = new HashMap<>();
        transactionMap = new HashMap<>();
        lock = new ReentrantLock();
    }

    @Override
    public CompletableFuture<V> get(K id, CacheTransaction transaction) {
        return CompletableFuture.completedFuture(cacheMap.get(id));
    }

    @Override
    public CompletableFuture<V> put(V item, CacheTransaction transaction) {
        InmemoryTransaction thisTransaction = (InmemoryTransaction) transaction;
        K key = mapKey(item);
        thisTransaction.getRemoves(this).remove(key);
        thisTransaction.getUpdates(this).put(key, item);
        return CompletableFuture.completedFuture(item);
    }

    @Override
    public CompletableFuture<Void> remove(List<K> ids, CacheTransaction transaction) {
        InmemoryTransaction thisTransaction = (InmemoryTransaction) transaction;
        ids.forEach(id -> {
            thisTransaction.getUpdates(this).remove(id);
            thisTransaction.getRemoves(this).add(id);
        });
        return CompletableFuture.completedFuture(null);
    }

    void acquire() {
        lock.lock();
    }

    void release() {
        lock.unlock();
    }

    boolean mayCommitTransaction(InmemoryTransaction transaction) {
        Set<K> keysToUpdate = new HashSet<>(transaction.getUpdates(this).keySet());
        keysToUpdate.addAll(transaction.getRemoves(this));
        for (K key : keysToUpdate) {
            if (transactionMap.containsKey(key) && transactionMap.get(key).getEnded().isAfter(transaction.getStarted())) {
                return false;
            }
        }
        return true;
    }

    void commitTransaction(InmemoryTransaction transaction) {
        for (var entry : transaction.getUpdates(this).entrySet()) {
            cacheMap.put(entry.getKey(), entry.getValue());
            transactionMap.put(entry.getKey(), transaction);
        }
        for (var key : transaction.getRemoves(this)) {
            cacheMap.remove(key);
            transactionMap.put(key, transaction);
        }
    }

    protected abstract K mapKey(V entity);
}
