package ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.inmemory;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.CacheTransaction;
import ru.yandex.travel.hotels.searcher.services.cache.travelline.availability.ConcurrentUpdateException;

@Slf4j
public class InmemoryTransaction implements CacheTransaction {
    private final Map<InmemoryCacheRepository<?, ?>, Map<?, ?>> updates;
    private final Map<InmemoryCacheRepository<?, ?>, Set<?>> removes;
    @Getter
    private final Instant started;
    private final UUID id;
    @Getter
    private Instant ended;


    public InmemoryTransaction() {
        id = UUID.randomUUID();
        log.info("Started transaction " + id);
        updates = new HashMap<>();
        removes = new HashMap<>();
        started = Instant.now();
    }

    public <K, V> Map<K, V> getUpdates(InmemoryCacheRepository<K, V> repository) {
        if (updates.containsKey(repository)) {
            return (Map<K, V>) updates.get(repository);
        } else {
            Map<K, V> res = new HashMap<>();
            updates.put(repository, res);
            return res;
        }
    }

    public <K> Set<K> getRemoves(InmemoryCacheRepository<K, ?> repository) {
        if (removes.containsKey(repository)) {
            return (Set<K>) removes.get(repository);
        } else {
            Set<K> res = new HashSet<>();
            removes.put(repository, res);
            return res;
        }
    }

    @Override
    public CompletableFuture<Void> commit() {
        log.info("Committing transaction " + id);
        ended = Instant.now();
        var repositoriesToCommitTo = new HashSet<>(updates.keySet());
        repositoriesToCommitTo.addAll(removes.keySet());
        var repoList = new ArrayList<>(repositoriesToCommitTo);
        repoList.sort(Comparator.comparing(i -> i.getClass().getName()));
        try {
            repoList.forEach(InmemoryCacheRepository::acquire);
            if (repoList.stream().allMatch(r -> r.mayCommitTransaction(this))) {
                repoList.forEach(r -> r.commitTransaction(this));
            } else {
                log.info("Row conflict in transaction " + id);
                return CompletableFuture.failedFuture(new ConcurrentUpdateException(null));
            }
        } finally {
            repoList.forEach(InmemoryCacheRepository::release);
        }
        log.info("Transaction " + id + " committed");
        return CompletableFuture.completedFuture(null);
    }

    @Override
    public void close() {
        log.info("Closing transaction " + id);
        ended = Instant.now();
    }
}
